// Copyright (C) 2025 EDF
// All Rights Reserved
// This code is published under the GNU Lesser General Public License (GNU LGPL)
#ifndef GRIDADAPT2D_H
#define GRIDADAPT2D_H
#include <memory>
#include <vector>
#include <list>
#include <array>
#include <Eigen/Dense>
#include "StOpt/core/grids/GridAdaptBase.h"
#include "StOpt/core/grids/Mesh2D.h"

/** \file GridAdapt2D.h
 * \brief Defines a 2D grid with refinement possibility
 * \author  Xavier WARIN
 */

namespace StOpt
{
  class GridAdapt2D : public GridAdaptBase
  {
  private :
    
    std::list< std::pair<std::shared_ptr< Mesh2D> , std::shared_ptr< std::vector< Eigen::ArrayXi > > > >  m_meshes ; /// a grid is a collection of meshes and position  in grid
    //  std::vector< std::array<int,2> >  is a collection of (i,j) defining the position in the grid.  m_mesh is not necessary
    double m_xMin ; /// x min of grid
    double m_xMax ; /// x max of grid
    double m_yMin ; /// y_min of grid
    double m_yMax ; /// y_max of grid
   
    
  public:

    // default constructor
    GridAdapt2D(){}

    /// \brief Constructor of an unform grid mesh
    /// \param p_xMin  minimal   grid x axis
    /// \param p_xMax  maximal   grid x axis
    /// \param p_nbMeshX number of mesh in x
    /// \param p_yMin  minimal   grid ordinate
    /// \param p_yMax  maximal   grid ordinate
    /// \param p_nbMeshY number of mesh in y
    GridAdapt2D( const double & p_xMin , const double & p_xMax, const int & p_nbMeshX, const double & p_yMin, const double & p_yMax, const int &  p_nbMeshY);

    /// \brief second constructor
    /// \param p_xMin  minimal   grid x axis
    /// \param p_xMax  maximal   grid x axis
    /// \param p_yMin  minimal   grid ordinate
    /// \param p_yMax  maximal   grid ordinate
    /// \param p_meshe  all the meshes
    /// \param p_points all the vertices 
     GridAdapt2D(const double & p_xMin , const double & p_xMax, const double & p_yMin, const double & p_yMax,  \
		 const std::list<  std::pair<std::shared_ptr< Mesh2D> , std::shared_ptr< std::vector< Eigen::ArrayXi > > > > & p_meshes,  const std::vector< Eigen::ArrayXd > & p_points );
    
    /// \brief thrid constructor without meshes
    /// \param p_xMin  minimal   grid x axis
    /// \param p_xMax  maximal   grid x axis
    /// \param p_yMin  minimal   grid ordinate
    /// \param p_yMax  maximal   grid ordinate
    /// \param p_points all the vertices 
    GridAdapt2D(const double & p_xMin , const double & p_xMax, const double & p_yMin, const double & p_yMax,  \
		 const std::vector< Eigen::ArrayXd > & p_points );
    
    /// \brief split a mesh in the list  : create new meshes and remove old mesh  form list
    void splitMesh( std::pair<std::shared_ptr< Mesh2D>, std::shared_ptr< std::vector< Eigen::ArrayXi > > >  & p_meshToSplit);

    /// \brief from  an hypercube : defines all meshes intersecting this hypercube
    /// \param p_xMin  minimal  hypercube x axis 
    /// \param p_xMax  maximal  hypercube x axis
    /// \param p_yMin  minimal  hypercube ordinate
    /// \param p_yMax  maximal  hypercube ordinate
    std::list< std::pair<std::shared_ptr< Mesh2D>, std::shared_ptr< std::vector< Eigen::ArrayXi > > > >  intersect(const double & p_xMin, const double & p_xMax, const double & p_yMin, const double & p_yMax) const;

    /// get dimension of the grid
    virtual int getDim() const {return 2;}

    // get back Meshes
    inline std::list< std::pair<std::shared_ptr< Mesh2D>, std::shared_ptr< std::vector< Eigen::ArrayXi > > > >  getMeshes() const { return m_meshes ;}

    /// for a point get the mesh associated
    /// \param p_x x axis
    /// \param p_y ordinate
    std::shared_ptr< Mesh2D>  getMeshWithPoint(const double & p_x, const double & p_y) const;

    /// get back x axis value for point given its number
    inline double getXCoord(const int & p_ipt) const { return m_points[p_ipt][0];}
    /// get back y axis value for point given its number
    inline double getYCoord(const int & p_ipt) const { return m_points[p_ipt][1];}
    /// get back an array with coordinates
    inline Eigen::ArrayXd getCoord(const int & p_ipt) const { return  Eigen::ArrayXd::Map(m_points[p_ipt].data(), 2);}

    inline double getXMin() const { return m_xMin;}
    inline double getXMax() const { return m_xMax;}
    inline double getYMin() const { return m_yMin;}
    inline double getYMax() const { return m_yMax;}

    
  };
}
#endif
