#!/usr/bin/env python3
#
# Copyright (c) 2021-2023 Paul Mattes.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above copyright
#       notice, this list of conditions and the following disclaimer in the
#       documentation and/or other materials provided with the distribution.
#     * Neither the names of Paul Mattes nor the names of his contributors
#       may be used to endorse or promote products derived from this software
#       without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY PAUL MATTES "AS IS" AND ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
# EVENT SHALL PAUL MATTES BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
# OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
# WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
# OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# s3270 paste test

import requests
from subprocess import Popen, PIPE, DEVNULL
import sys
import unittest
import Common.Test.playback as playback
import Common.Test.cti as cti

class TestS3270Paste(cti.cti):

    # s3270 paste test
    def test_s3270_paste(self):

        port, ts = cti.unused_port()
        null = 'NUL:' if sys.platform.startswith('win') else '/dev/null'
        with playback.playback(self, null, port=port,) as p:
            ts.close()

            # Start s3270.
            hport, socket = cti.unused_port()
            s3270 = Popen(cti.vgwrap(['s3270', '-httpd', str(hport), '-nvt', f'127.0.0.1:{port}']), stdin=DEVNULL, stdout=DEVNULL)
            self.children.append(s3270)
            socket.close()
            self.check_listen(hport)

            # Wait for the connection to complete.
            self.try_until(lambda: 'connected' in requests.get(f'http://127.0.0.1:{hport}/3270/rest/json/Query(ConnectionState)').json()['result'][0], 2, "didn't connect")

            # Pump in a string that almost wraps.
            a79 = ''.join(['A' for i in range(79)])
            requests.get(f'http://127.0.0.1:{hport}/3270/rest/json/String({a79})')

            # Paste in a string that wraps.
            requests.get(f'http://127.0.0.1:{hport}/3270/rest/json/PasteString(4242)')

            # Get the cursor location.
            r = requests.get(f'http://127.0.0.1:{hport}/3270/rest/json/Query(cursor1)')
            col = r.json()['result'][0].split()[3]

            # Make sure it is 2 (not 80).
            rx = r.json()
            self.assertEqual(2, int(col), f'result is {rx}')

        # Wait for s3270 to exit.
        r = requests.get(f'http://127.0.0.1:{hport}/3270/rest/json/Quit()')
        self.vgwait(s3270)

if __name__ == '__main__':
    unittest.main()
