#!/bin/bash
# Copyright (c) 1998 Lawrence Livermore National Security, LLC and other
# HYPRE Project Developers. See the top-level COPYRIGHT file for details.
#
# SPDX-License-Identifier: (Apache-2.0 OR MIT)

# Generate code for multiprecision _pre functions
#
# The script takes a prototype info file (generated by 'gen_proto_info.sh') and
# an output file prefix, then generates files prefix.c and prefix.h containing
# function implementations and prototypes for each line of prototype info.
#
# Usage:   <this-script> <proto-info> <prefix>
# Example: <this-script> mup.pre.ext.proto mup_pre_ext

scriptdir=`dirname $0`

PFILE=$1
OUTC="$2.c"
OUTH="$2.h"

echo "" > $OUTC
echo "" > $OUTH

# loop over lines and generate code for each function
awk -v filename="$PFILE" -v outc="$OUTC" -v outh="$OUTH" 'BEGIN {
   FS=" , "
   # Read the prototype info file
   while (getline < filename)
   {
      fret  = $1
      fdef  = $2
      tab   = "   "
      p_str = ""
      s_str = ""
      for(i=3; i<=NF; i++)
      {
         argall = sprintf("%s", $i)
         # Find the last legal C token in the argument (should not start with numbers)
         # This helps to address arguments like const double foo[3]
         laststart  = 1;
         lastlength = 0;
         while ( match(substr($i, laststart + lastlength), /[a-zA-Z_][a-zA-Z0-9_]*/) )
         {
            laststart  = laststart + lastlength + RSTART - 1
            lastlength = RLENGTH
         }
         argvar = substr($i, laststart, lastlength)
         sub(/^[[:blank:]]*/, "", argall); sub(/[[:blank:]]*$/, "", argall)
         sub(/^[[:blank:]]*/, "", argvar); sub(/[[:blank:]]*$/, "", argvar)
         p_str = sprintf("%s %s", p_str, argall)
         s_str = sprintf("%s %s", s_str, argvar)
         if(i<NF)
         {
            p_str = sprintf("%s,", p_str)
            s_str = sprintf("%s,", s_str)
         }
      }

      if (NF < 3)
      {
         # This is a special case Foo(void) function
         p_str = " void "
         s_str = " "
         arg_pre = " HYPRE_Precision precision "
      }
      else
      {
         p_str = sprintf("%s ", p_str)
         s_str = sprintf("%s ", s_str)
         arg_pre = sprintf(" HYPRE_Precision precision,%s", p_str)
      }

      # First replace HYPRE_Real* and HYPRE_Complex* with void*
      gsub(/(HYPRE_Real|HYPRE_Complex)[[:blank:]]*[*]+/, "void *", arg_pre)
      gsub(/(HYPRE_Real|HYPRE_Complex)/, "hypre_long_double", arg_pre)

      print fret"\n"fdef"_pre("arg_pre");\n"         >> outh

      print "/*--------------------------------------------------------------------------*/\n" >> outc
      print fret"\n"fdef"_pre("arg_pre")"                                                      >> outc
      print "{"                                                                                >> outc
      print tab "switch (precision)"                                                           >> outc
      print tab "{"                                                                            >> outc
      print tab tab "case HYPRE_REAL_SINGLE:"                                                  >> outc
      print tab tab tab "return "fdef"_flt("s_str");"                                          >> outc
      print tab tab "case HYPRE_REAL_DOUBLE:"                                                  >> outc
      print tab tab tab "return "fdef"_dbl("s_str");"                                          >> outc
      print tab tab "case HYPRE_REAL_LONGDOUBLE:"                                              >> outc
      print tab tab tab "return "fdef"_long_dbl("s_str");"                                     >> outc
      print tab tab "default:" >> outc
      if(fret == "void")
      {
         print tab tab tab "hypre_error_w_msg(HYPRE_ERROR_GENERIC, \"Unknown solver precision\");" >> outc
      }
      else
      {
         print tab tab tab "{ "fret" value = 0; hypre_error_w_msg(HYPRE_ERROR_GENERIC, \"Unknown solver precision\"); return value; }" >> outc
      }
      print tab "}"                                                                            >> outc
      print "}\n"                                                                              >> outc
   }
   close(filename)
}'
