// Copyright (C) 2001 Nils Bokermann <Nils.Bokermann@mediaWays.net>
//
// PURPOSE OF THIS FILE: Implement the ldap_add functions
//
// - Automatic Version Information via RCS:
//   $Id: search.cxx,v 1.2 2001/12/19 05:26:56 nilsb Exp $
//   $Source: /cvsroot/openh323gk/openh323gk/ldap/src/search.cxx,v $
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "ldapapi.h"
#include <ldap-int.h>
#include "LDAP_SFilter.h"

#include <iostream>

#ifndef lint
// mark object with version info
static const char vcid[] = "@(#) $Id: search.cxx,v 1.2 2001/12/19 05:26:56 nilsb Exp $";
static const char vcHid[] = __LDAP_OPENH323_H;
#endif

int ldap_search_ext (LDAP *ld, LDAP_CONST char *base, int scope, LDAP_CONST char *filter,
		     char **attrs, int attrsonly, LDAPControl **serverctrls,
		     LDAPControl **clientctrls, struct timeval *timeout, 
		     int sizelimit, int *msgidp){

  struct timeval *time = timeout;
  if(NULL==time){
    time=new struct timeval;
    time->tv_sec=(0==ld->ld_options->ldo_timelimit) ? 0 : ld->ld_options->ldo_timelimit;
    time->tv_usec=0;
  }
  
  if(timeout &&(time->tv_sec==0)&&(time->tv_usec==0))
    return LDAP_PARAM_ERROR;
  if(timeout && (time->tv_sec==0))
    time->tv_sec=1;

  LDAP_LDAPMessage msg;

  *msgidp=++(ld->msg_id);
  msg.m_messageID=*msgidp;
  msg.m_protocolOp=LDAP_LDAPMessage_protocolOp(LDAP_LDAPMessage_protocolOp::e_searchRequest);
  
  LDAP_SearchRequest & search=msg.m_protocolOp;

  search.m_baseObject=base;
  LDAP_SearchRequest_scope & lscope=search.m_scope;
  switch (scope) {
    case LDAP_SCOPE_BASE:
      lscope=LDAP_SearchRequest_scope::e_baseObject;
      break;
    case LDAP_SCOPE_ONELEVEL:
      lscope=LDAP_SearchRequest_scope::e_singleLevel;
      break;
    case LDAP_SCOPE_SUBTREE:
    default:
      lscope=LDAP_SearchRequest_scope::e_wholeSubtree;
      break;
  }
  search.m_derefAliases=LDAP_SearchRequest_derefAliases(ld->ld_options->ldo_deref);
  search.m_sizeLimit=PASN_Integer(((sizelimit<0) ? 0 : sizelimit));
  search.m_attrsOnly=PASN_Boolean(0!=attrsonly);
  search.m_timeLimit=PASN_Integer(time->tv_sec);

  if(filter==NULL) {
    search.m_filter=LDAP_SFilter("(objectclass=*)");
  }else{
    search.m_filter=LDAP_SFilter((char *)filter);
  }
  LDAP_ArrayOf_AttributeType &attr=search.m_attributes;
  if(attrs!=NULL) {
    int i=0;
    while(NULL!=attrs[i]) {
      attr.SetSize(i+1);
      attr[i]=attrs[i];
      i++;
    }    
  }

  PBER_Stream encoding_stream;
  msg.Encode(encoding_stream);
#ifdef DEBUG
  cerr << search.m_filter << endl;
  cerr << msg << endl;
#endif
#ifdef BER_DEBUG
  cerr << encoding_stream << endl;
#endif
  
  if(encoding_stream.Write(*(ld->socket))) {
    return LDAP_SUCCESS;
  }
  ld->ld_errno=LDAP_UNAVAILABLE;
  return ld->ld_errno;
}
  
int ldap_search_ext_s (LDAP *ld, LDAP_CONST char *base, int scope, LDAP_CONST char *filter,
		       char **attrs, int attrsonly, LDAPControl **serverctrls, 
		       LDAPControl **clientctrls, struct timeval *timeout, 
		       int sizelimit, LDAPMessage **res) {
  int msgid;
  int rv;
  LDAPMessage *msg;
  rv=ldap_search_ext(ld, base, scope, filter, attrs, attrsonly, serverctrls, clientctrls, 
		  timeout, sizelimit, &msgid);
  
  if(LDAP_SUCCESS!=rv)
    return rv;
  
  if(ldap_result(ld, msgid, 1, NULL, &msg ) == -1 ) 
    return (ld->ld_errno);
  
  res=&msg;
  return LDAP_SUCCESS;
}

int ldap_search (LDAP *ld, LDAP_CONST char *base, int scope, LDAP_CONST char *filter, 
		 char **attrs, int attrsonly) {
  int msgid;
  int rv;

  rv=ldap_search_ext(ld, base, scope, filter, attrs, attrsonly, NULL, NULL, NULL, 
		     LDAP_NO_LIMIT, &msgid);
  if(LDAP_SUCCESS!=rv){
    ld->ld_errno=rv;
    return (-1);
  }
  return msgid;
}

int ldap_search_s (LDAP *ld, LDAP_CONST char *base, int scope, LDAP_CONST char *filter, 
		   char **attrs, int attrsonly, LDAPMessage **res){
  LDAPMessage *msg;
  int msgid;
  msgid=ldap_search(ld, base, scope, filter, attrs, attrsonly);
  if(-1==msgid)
    return ld->ld_errno;
  
  if(ldap_result(ld, msgid, 1, NULL, &msg ) == -1)
    return ld->ld_errno;
  *res=msg;
  return LDAP_SUCCESS;
}

int ldap_search_st (LDAP *ld, LDAP_CONST char *base, int scope, LDAP_CONST char *filter, 
		    char **attrs, int attrsonly, struct timeval *timeout, 
		    LDAPMessage **res){
  LDAPMessage *msg;
  int msgid;
  msgid=ldap_search(ld, base, scope, filter, attrs, attrsonly);
  if(-1==msgid)
    return ld->ld_errno;
  
  if(ldap_result(ld, msgid, 1, timeout, &msg ) == -1)
    return ld->ld_errno;
  *res=msg;
  return LDAP_SUCCESS;
}

//
// End of add.cxx
//


  
