// Copyright (C) 2001 Nils Bokermann <Nils.Bokermann@mediaWays.net>
//
// PURPOSE OF THIS FILE: Implent function to get the DN
//
// - Automatic Version Information via RCS:
//   $Id: getdn.cxx,v 1.1 2001/12/17 16:36:17 nilsb Exp $
//   $Source: /cvsroot/openh323gk/openh323gk/ldap/src/getdn.cxx,v $
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "ldapapi.h"
#include <ldap-int.h>

#ifndef lint
// mark object with version info
static const char vcid[] = "@(#) $Id: getdn.cxx,v 1.1 2001/12/17 16:36:17 nilsb Exp $";
static const char vcHid[] = __LDAP_OPENH323_H;
#endif

char *ldap_get_dn (LDAP *ld, LDAPMessage *entry) {
  if((NULL==ld)||(NULL==entry)) {
    ld->ld_errno=LDAP_PARAM_ERROR;
    return NULL;
  }
  if(!(&(entry->message->m_protocolOp.GetObject()))->IsClass("LDAP_SearchResponse")) {
    ld->ld_errno=LDAP_PARAM_ERROR;
    return NULL;
  }
  LDAP_SearchResponse &res=entry->message->m_protocolOp;
  if(res.GetTag()==LDAP_SearchResponse::e_entry) {
    LDAP_SearchResponse_entry & ent=res;
    return strndup((const char *) ent.m_objectName.GetPointer(),
		   ent.m_objectName.GetSize());
  }
  LDAP_LDAPResult & ent=res;
  return strndup((const char *) ent.m_matchedDN.GetPointer(),
		 ent.m_matchedDN.GetSize());
}

static char * ldap_charray2str(char ** in, char *sep) {
  if (NULL==sep)
    sep=" ";
  
  PString out;

  while(NULL!=*in) {
    out&=*in;
    out&=sep;
    in++;
  }
  return strndup(out,strlen(out));
}

char *ldap_dn2ufn (char *dn) {
  char	*ufn;
  char	**vals;
  int i;
  
  /* produces completely untyped UFNs */
  
  if( dn == NULL ) {
    return NULL;
  }

  vals = ldap_explode_dn( dn , 0 );
  if( vals == NULL ) {
    return NULL;
  }

  for ( i = 0; vals[i]; i++ ) {
    char **rvals;

    rvals = ldap_explode_rdn( vals[i] , 1 );
    if ( rvals == NULL ) {
      ldap_value_free(vals);
      return NULL;
    }

    ldap_memfree(vals[i]);
    vals[i] = ldap_charray2str( rvals, " + " );
    ldap_value_free(rvals);
  }

  ufn = ldap_charray2str( vals, ", " );
  
  ldap_value_free(vals);
  return ufn;
}
#define NAME_TYPE_LDAP_DN 5
#define NAME_TYPE_LDAP_RDN 6
#define INQUOTE		1
#define OUTQUOTE	2

#define ISASCII(uc)     ((uc) < 0x80)
#define LDAP_UTF8_ISASCII(p) ( * (const unsigned char *) (p) < 0x80 )
#define LDAP_UTF8_OFFSET(p) ( LDAP_UTF8_ISASCII(p) \
	? 1 : ldap_utf8_offset((p)) )

#define LDAP_UTF8_NEXT(p) (	LDAP_UTF8_ISASCII(p) \
	? (char *)(p)+1 : ldap_utf8_next((p)) )

#define LDAP_UTF8_INCR(p) ((p) = LDAP_UTF8_NEXT(p))

char* ldap_utf8_next (const char * p){
  int i;
  const unsigned char *u = (const unsigned char *) p;
  
  if(LDAP_UTF8_ISASCII(u)) {
    return (char *) &p[1];
  }
  
  for(i=1; i<6; i++) {
    if((u[i] & 0xc0) != 0x80) {
      return (char *) &p[i];
    }
  }

  return (char *) &p[i];
}

int ldap_utf8_offset(const char * p){
  return LDAP_UTF8_NEXT(p) - p;
}


int ldap_utf8_isspace (const char *p) {
  unsigned c = * (const unsigned char *) p;

  if(!ISASCII(c)) return 0;
  
  switch(c) {
    case ' ':
    case '\t':
    case '\n':
    case '\r':
    case '\v':
    case '\f':
      return 1;
  }
  
  return 0;
}


static char ** explode_name(const char *name, int notypes, int is_type) {
  const char *p, *q, *rdn;
  vector<char *> parts;
  int	offset, state, have_equals, count = 0, endquote, len;
  
  /* safe guard */
  if(name == NULL) name = "";
  
  /* skip leading whitespace */
  while( ldap_utf8_isspace( name )) {
    LDAP_UTF8_INCR( name );
  }

  p = rdn = name;
  offset = 0;
  state = OUTQUOTE;
  have_equals=0;
  
  do {
    /* step forward */
    p += offset;
    offset = 1;

    switch ( *p ) {
      case '\\':
	if ( p[1] != '\0' ) {
	  offset = LDAP_UTF8_OFFSET(++p);
	}
	break;
      case '"':
	if ( state == INQUOTE )
	  state = OUTQUOTE;
	else
	  state = INQUOTE;
	break;
      case '=':
	if( state == OUTQUOTE ) have_equals++;
	break;
      case '+':
	if (is_type == NAME_TYPE_LDAP_RDN)
	  goto end_part;
	break;
      case '/':
	break;
      case ';':
      case ',':
	if (is_type == NAME_TYPE_LDAP_DN)
	  goto end_part;
	break;
      case '\0':
    end_part:
	if ( state == OUTQUOTE ) {
	  ++count;
	  have_equals=0;
	  
	  endquote = 0;
	  
	  if ( notypes ) {
	    for ( q = rdn; q < p && *q != '='; ++q ) {
	      /* EMPTY */;
	    }
	    
	    if ( q < p ) {
	      rdn = ++q;
	    }
	    
	    if ( *rdn == '"' ) {
	      ++rdn;
	    }
	    
	    if ( p[-1] == '"' ) {
	      endquote = 1;
	      --p;
	    }
	  }
	  
	  len = p - rdn;
	  char *tmp;
	  if ((tmp = new char [len + 1]) != NULL) {
	    memcpy(tmp, rdn, len);
	    parts.push_back(tmp);
	    
	    if( !endquote ) {
	      /* skip trailing spaces */
	      while(len > 0 && ldap_utf8_isspace(&parts[count-1][len-1])) {
		--len;
	      }
	    }
	    
	    parts[ count-1 ][ len ] = '\0';
	  }
	  
	  /*
	   *  Don't forget to increment 'p' back to where
	   *  it should be.  If we don't, then we will
	   *  never get past an "end quote."
	   */
	  if ( endquote == 1 )
	    p++;
	  
	  rdn = *p ? &p[1] : p;
	  while ( ldap_utf8_isspace( rdn ) )
	    ++rdn;
	} break;
    }
  } while ( *p );
 
  char **ret = NULL;

  ret=new (char *)[parts.size()+1];
  for(unsigned int i=0; i<parts.size(); i++) 
    ret[i]=parts[i];
  ret[parts.size()]=NULL;

  
  return(ret);
}


char ** ldap_explode_dn (char const *dn, int notypes) {
  return explode_name(dn, notypes, NAME_TYPE_LDAP_DN);
}

char ** ldap_explode_rdn (char const *rdn, int notypes) {
  return explode_name(rdn, notypes, NAME_TYPE_LDAP_RDN);  
}

//
// End of getdn.cxx
//
