# arch-tag: david@allouche.net - 2003-11-17 15:29:01 469107000
# Copyright (C) 2003  John Goerzen, David Allouche
# <jgoerzen@complete.org> <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Construction of tla commands
"""

import os.path
from pybaz.pathname import DirName, FileName
from pybaz import errors
from pybaz._escaping import *
from pybaz._nameparser import NameParser

tlasyn = None
tlaobj = None


def gettlasyntax():
    global tlasyn, tlaobj
    if tlasyn != None:
        return tlasyn
    if text_cmd(['-V']).splitlines()[0].find('tla-1.0.') != -1:
        tlasyn = '1.0'
        tlaobj = Tla10()
    else:
        tlasyn = '1.1'
        tlaobj = Tla11()
    tlaobj.name_escaping = (0 == status_cmd(['escape'], (0,1)))
    return tlasyn


class Tla10:
    tagging_method = 'tagging-method'
    add = 'add-tag'
    move = 'move-tag'
    delete = 'delete-tag'
    update = 'update --in-place .'
    replay = 'replay --in-place .'
    log_versions = 'logs'
    log_ls = 'log-ls'
    merges = None
    get_changeset = 'get-patch'


class Tla11:
    tagging_method = 'id-tagging-method'
    add = 'add'
    move = 'move'
    delete = 'delete'
    update = 'update'
    replay = 'replay'
    log_versions = 'log-versions'
    log_ls = 'logs'
    merges = 'merges'
    get_changeset = 'get-changeset'


def cmd():
    global tlaobj
    gettlasyntax()
    return tlaobj


### Utility functions ###

def _check_expected_param(expected):
    msg_format = ("'expected' param must be a sequence of positive"
                  " integers but was: %r")
    try:
        iterator = iter(expected)
    except TypeError:
        raise TypeError, msg_format % expected
    for status in iterator:
        if not isinstance(status, int):
            raise TypeError, msg_format % expected
        if status < 0:
            raise ValueError, msg_format % expected


### Subrocess spawning ###

def _backend():
    import pybaz
    return pybaz.backend

def null_cmd(args, chdir=None):
    """Deprecated, for internal use only."""
    _backend().null_cmd(args, chdir)

def one_cmd(args, chdir=None):
    """Deprecated, for internal use only."""
    return _backend().one_cmd(args, chdir)

def sequence_cmd(args, chdir=None, expected=(0,)):
    """Deprecated, for internal use only."""
    return _backend().sequence_cmd(args, expected, chdir)

def text_cmd(args, chdir=None):
    """Deprecated, for internal use only."""
    return _backend().text_cmd(args, chdir)

def status_cmd(args, expected):
    """Deprecated, for internal use only."""
    return _backend().status_cmd(args, expected)

def status_one_cmd(args, chdir, expected):
    """Deprecated, for internal use only."""
    return _backend().status_one_cmd(args, expected, chdir)

def status_text_cmd(args, expected):
    """Deprecated, for internal use only."""
    return _backend().status_text_cmd(args, expected)


### tla commands ###

# User Commands

def default_archive():
    status, output = status_one_cmd(('my-default-archive',), None, (0,1))
    if status != 0: return None
    return output


def set_default_archive(archive):
    null_cmd(('my-default-archive', archive))


# Project Tree commands

def init_tree(dir, version=None, nested=False):
    args = ['init-tree', '--dir', dir]
    if nested: args.append('--nested')
    if version is not None: args.append(version)
    null_cmd(args)

def tree_root(dir):
    status, output = status_text_cmd(('tree-root', dir), (0,1))
    if status == 1: raise errors.TreeRootError(dir)
    return output[:-1]

def sync_tree(dir, revision):
    null_cmd(('sync-tree', '--dir', dir, revision))

def resolved(dir, all=False):
    args = ['resolved', '--dir', dir]
    if all:
        args.append('--all')
    null_cmd(args)

def undo(dir, revision=None, output=None, quiet=False, throw_away=False):
    args = ['undo', '--dir', dir]
    if quiet: args.append('--quiet')
    if throw_away:
        args.append('--no-output')
    elif output:
        args.extend(('--output', output))
    if revision: args.append(revision)
    null_cmd(args)

def log_for_merge(dir):
    args = ['log-for-merge', '--dir', dir]
    return text_cmd(args)


def redo(dir, patch=None, keep=False, quiet=False):
    args = ['redo', '--dir', dir]
    if quiet: args.append('--quiet')
    if keep: args.append('--keep')
    if patch: args.extend(('--patch', patch))
    null_cmd(args)

def update(dir):
    """FIXME: add the other parameters."""
    ### Correct implementation
    # args = ['update', '--dir', dir]
    # return null_cmd(args)
    ### Work around bug in tla 1.2.1
    null_cmd(['update'], chdir=dir)

def replay(dir):
    """FIXME: add the other parameters."""
    args = ['replay', '--dir', dir]
    null_cmd(args)


# Project Tree Inventory commands

def tagging_method(dir):
    return one_cmd((cmd().tagging_method, '--dir', dir))

def set_tagging_method(dir, method):
    null_cmd((cmd().tagging_method, '--dir', dir, method))

def add(file):
    null_cmd((cmd().add, file))

def delete(file):
    null_cmd((cmd().delete, file))

def move(src, dest):
    return null_cmd((cmd().move, src, dest))


# Patch Set Commands

def changeset(orig, mod, dest, files=()):
    return null_cmd(('changeset', orig, mod, dest) + files)


# Archive Transaction Commands

def get_patch(revision, dir):
    P = NameParser(revision)
    # XXX Work around bug in 1.2.2rc2 and corresponding integration revisions
    # XXX where get-changeset would fail with "no default archive set".
    args = [cmd().get_changeset, '-A', P.get_archive(), P.get_nonarch(), dir]
    null_cmd(args)

# Archive Commands

def cacherev(revision, cache=None):
    if cache is None:
        return null_cmd(('cacherev', revision))
    else:
        return null_cmd(('cacherev', '--cache', cache, revision))


def cachedrevs(version):
    return sequence_cmd(('cachedrevs', version))


def uncacherev(revision):
    null_cmd(('uncacherev', revision))


def iter_merges(version1, version2=None, reverse=False, metoo=True):
    subcmd = cmd().merges
    if subcmd is None:
        raise NotImplementedError, \
              "merges is not implemented in tla 1.0"""
    args = [ subcmd ]
    if reverse: args.append('--reverse')
    args.append('--full')
    args.append(version1)
    if version2 is not None: args.append(version2)
    for line in sequence_cmd(args):
        target, source = line.split('\t')
        if metoo or '%s--%s' % (version1, target) != source:
            yield target, source


# Patch Log Commands

def make_log(dir):
    return text_cmd(('make-log', '--dir', dir))[:-1]

def add_log_version(dir, version):
    null_cmd(('add-log-version', '--dir', dir, version))

def remove_log_version(dir, version):
    null_cmd(('remove-log-version', '--dir', dir, version))


def iter_log_versions(dir, reverse=False,
                      archive=None, category=None, branch=None, version=None):
    opts = (cmd().log_versions, '--dir', dir)
    if reverse: opts += ('--reverse',)
    if archive: opts += ('--archive', archive)
    if category: opts += ('--category', category)
    if branch: opts += ('--branch', branch)
    if version: opts += ('--vsn', version)
    return sequence_cmd(opts)


def iter_log_ls(dir, version, reverse=False):
    opts = [ cmd().log_ls, '--full', '--dir', dir ]
    if reverse: opts.append('--reverse')
    opts.append(version)
    return sequence_cmd(opts)


# Multi-project Configuration Commands

# Commands for Branching and Merging

def tag(source_revision, tag_version):
    null_cmd(('branch', source_revision, tag_version))


# Local Cache Commands

def file_find(tree, file_name, revision):
    args = ('file-find', '--new-file', file_name, revision)
    # XXX Work around missing --silent option, ignore chatter
    output = text_cmd(args, chdir=tree)
    path = output.splitlines()[-1]
    if path == '/dev/null': return None
    if cmd().name_escaping:
        return name_unescape(path)
    else:
        return path

def iter_pristines(dir_name):
    args = ('pristines', '--dir', dir_name)
    return sequence_cmd(args)

def add_pristine(dir_name, revision):
    args = ('add-pristine', '--dir', dir_name, revision)
    null_cmd(args)

def find_pristine(dir_name, revision):
    args = ('find-pristine', '--dir', dir_name, revision)
    return one_cmd(args)


# Revision Library Commands

def iter_revision_libraries():
    args = ('my-revision-library', '--silent')
    return sequence_cmd(args, expected=(0,1))

def register_revision_library(dirname):
    null_cmd(('my-revision-library', '--silent', dirname))

def unregister_revision_library(dirname):
    null_cmd(('my-revision-library', '--silent', '--delete', dirname))

def library_archives():
    return sequence_cmd(('library-archives',))

def library_categories(archive):
    return sequence_cmd(('library-categories', '--archive', archive))

def library_branches(category):
    return  sequence_cmd(('library-branches', category))

def library_versions(branch, reverse=False):
    if not reverse:
        return sequence_cmd(('library-versions', branch))
    else:
        return sequence_cmd(('library-versions', '--reverse', branch))

def library_revisions(version, reverse=False):
    if not reverse:
        return sequence_cmd(('library-revisions', version))
    else:
        return sequence_cmd(('library-revisions', '--reverse', version))


def library_add(revision):
    null_cmd(('library-add', revision))

def library_remove(revision):
    null_cmd(('library-remove', revision))

def library_find(revision):
    return text_cmd(('library-find', revision))[:-1]


### Pika escaping ###

def tla_name_escape(text):
   return text_cmd(('escape', text))

def tla_name_unescape(text):
    return text_cmd(('escape', '--unescaped', text))


### Misc features ###

def in_tree(dir):
    """Return True if dir is, or is inside, a Arch source tree."""
    # Must use tree-root and not tree-version because the tree-version
    # may be unset (after init-tree)
    status = status_cmd(('tree-root', dir), expected=(0,1))
    return not status


def has_explicit_id(path):
    assert os.path.exists(path)
    arch_ids = DirName('.arch-ids')
    if os.path.isfile(path):
        dir_, base = FileName(path).splitname()
        return os.path.exists(dir_/arch_ids/(base+'.id'))
    if os.path.isdir(path):
        return os.path.exists(DirName(path)/arch_ids/'=id')
    raise AssertionError, 'not regular file or directory: ' + path
