/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2006 Ariya Hidayat (ariya@kde.org)
 * Copyright (C) 2006-2007 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#include <stdio.h>
#include <string.h>

#include <libetonyek/libetonyek.h>
#include <libodfgen/libodfgen.hxx>

#include "writerperfect_utils.hxx"
#include "OutputFileHelper.hxx"
#include "StringDocumentHandler.hxx"
#include "UsageHelper.hxx"

#define TOOLNAME "key2odp"

using libetonyek::EtonyekDocument;

class OdpOutputFileHelper : public OutputFileHelper
{
public:
	OdpOutputFileHelper(const char *outFileName, const char *password) :
		OutputFileHelper(outFileName, password) {}
	~OdpOutputFileHelper() {}

	bool convertDocument(librevenge::RVNGInputStream &input, bool isFlat)
	{
		OdpGenerator collector;
		StringDocumentHandler stylesHandler, contentHandler, manifestHandler, settingsHandler;
		if (isFlat)
			collector.addDocumentHandler(&contentHandler, ODF_FLAT_XML);
		else
		{
			collector.addDocumentHandler(&contentHandler, ODF_CONTENT_XML);
			collector.addDocumentHandler(&manifestHandler, ODF_MANIFEST_XML);
			collector.addDocumentHandler(&settingsHandler, ODF_SETTINGS_XML);
			collector.addDocumentHandler(&stylesHandler, ODF_STYLES_XML);
		}
		if (!EtonyekDocument::parse(&input, &collector))
			return false;
		if (isFlat)
		{
			printf("%s\n", contentHandler.cstr());
			return true;
		}

		const char s_mimetypeStr[] = "application/vnd.oasis.opendocument.presentation";
		if (!writeChildFile("mimetype", s_mimetypeStr, (char)0) ||
		        !writeChildFile("META-INF/manifest.xml", manifestHandler.cstr()) ||
		        !writeChildFile("content.xml", contentHandler.cstr()) ||
		        !writeChildFile("settings.xml", settingsHandler.cstr()) ||
		        !writeChildFile("styles.xml", stylesHandler.cstr()))
			return false;

		librevenge::RVNGStringVector objects=collector.getObjectNames();
		for (unsigned i=0; i<objects.size(); ++i)
		{
			StringDocumentHandler objectHandler;
			if (collector.getObjectContent(objects[i], &objectHandler))
				writeChildFile(objects[i].cstr(), objectHandler.cstr());
		}
		return true;
	}

	static EtonyekDocument::Confidence isSupportedFormat(librevenge::RVNGInputStream &input, EtonyekDocument::Type *type = 0)
	{
		return EtonyekDocument::isSupported(&input, type);
	}
};

int printUsage(const char *name)
{
	UsageHelper usage(name, "converts Apple Keynote documents to ODF.");
	usage.addToDescription("If OUTPUT is omitted, the result is printed as Flat ODF to standard output.\n");
	usage.addStdoutOption();
	return usage.printUsage();
}

int main(int argc, char *argv[])
{
	if (argc < 2)
		return printUsage(TOOLNAME);

	char *szInputFile = 0;
	char *szOutFile = 0;
	bool stdOutput = false;

	for (int i = 1; i < argc; i++)
	{
		if (!strcmp(argv[i], "--stdout"))
			stdOutput = true;
		else if (!strcmp(argv[i], "--help"))
			return printUsage(TOOLNAME);
		else if (!strcmp(argv[i], "--version"))
			return UsageHelper::printVersion(TOOLNAME);
		else if (!szInputFile && strncmp(argv[i], "--", 2))
			szInputFile = argv[i];
		else if (szInputFile && !szOutFile && strncmp(argv[i], "--", 2))
			szOutFile = argv[i];
		else
			return printUsage(TOOLNAME);
	}

	if (!szInputFile)
		return printUsage(TOOLNAME);

	if (szOutFile && stdOutput)
		szOutFile = 0;

	OdpOutputFileHelper helper(szOutFile, 0);

	shared_ptr<librevenge::RVNGInputStream> input;
	if (librevenge::RVNGDirectoryStream::isDirectory(szInputFile))
		input.reset(new librevenge::RVNGDirectoryStream(szInputFile));
	else
		input.reset(new librevenge::RVNGFileStream(szInputFile));

	EtonyekDocument::Type type = EtonyekDocument::TYPE_UNKNOWN;
	const EtonyekDocument::Confidence confidence = helper.isSupportedFormat(*input, &type);
	if ((EtonyekDocument::CONFIDENCE_NONE == confidence) || (EtonyekDocument::TYPE_KEYNOTE != type))
	{
		fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Keynote Document.\n");
		return 1;
	}

	if (EtonyekDocument::CONFIDENCE_SUPPORTED_PART == confidence)
	{
		input.reset(librevenge::RVNGDirectoryStream::createForParent(szInputFile));

		if (EtonyekDocument::CONFIDENCE_EXCELLENT != helper.isSupportedFormat(*input))
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Keynote Document.\n");
			return 1;
		}
	}

	if (!helper.convertDocument(*input, szOutFile==0))
	{
		fprintf(stderr, "ERROR : Couldn't convert the document\n");
		return 1;
	}

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
