###############################################################################
# Tools.tcl
#
# This file implements some general utility for the rest of WineSetupTk source
# code.
#
# Copyright (c) 2000 by Martin Pilka for CodeWeavers
###############################################################################

proc lfirst {aList} {
  return [lindex $aList 0]
}

proc lupperb {aList} {
  return [expr [llength $aList]-1]
}

proc GCN {} {
  # generate string like "COMMENT001" with automatic numbering
  global CommentNum
  if {[info exists CommentNum]} { 
    incr CommentNum 1 
  } else { 
    set CommentNum 0 
  }
  return [format "COMMENT%03d" $CommentNum]
}

proc CenterWin {apWin args} {
  # this is really ugly. the width and height of centered window are parameters of this procedure. i don't know 
  # how to find out them without ugly effects on the screen. see the comments in next procedure.

  if {$args == ""} {
    wm withdraw $apWin
    update idletasks
    set Wid [winfo reqwidth $apWin]
    set Hei [winfo reqheight $apWin]
    set X [expr ([winfo screenwidth $apWin]-$Wid)/2]
    set Y [expr ([winfo screenheight $apWin]-$Hei)/2]
    wm geometry $apWin +$X+$Y
    wm deiconify $apWin
  } else {
    set Wid [lindex $args 0]
    set Hei [lindex $args 1]
    set X [expr ([winfo screenwidth $apWin]-$Wid)/2]
    set Y [expr ([winfo screenheight $apWin]-$Hei)/2]
    wm geometry $apWin +$X+$Y
  }
}

proc CenterChild {apParent apChild args} {
  if {$args == ""} {
    wm withdraw $apChild
    update idletasks
    set X [expr {[winfo rootx $apParent] + ([winfo width $apParent]-[winfo reqwidth $apChild])/2}]
    set Y [expr {[winfo rooty $apParent] + ([winfo height $apParent]-[winfo reqheight $apChild])/2}]
    wm geometry $apChild +$X+$Y
    wm deiconify $apChild
  } else {
    set ChildWid [lindex $args 0]
    set ChildHei [lindex $args 1]
    set ParentCenterX [expr [winfo x $apParent] + ([winfo width $apParent]/2)]
    set ParentCenterY [expr [winfo y $apParent] + ([winfo height $apParent]/2)]
    set X [expr $ParentCenterX - ($ChildWid/2)]
    set Y [expr $ParentCenterY - ($ChildHei/2)]
    wm geometry $apChild +$X+$Y
  }
}

proc ShortString {aStr aMax} {
  if {[string length $aStr] <= $aMax} { return $aStr }
  if {$aMax < 5} {
    puts "ShortString: aMax parameter cannot be less than 5, if format is required ($aMax)."
    return $aStr
  }
  set Half [expr $aMax/2]
  if {$aMax % 2 == 0} {
    set PathBegin [string range $aStr 0 [expr $Half-2-1]]
    set PathEnd [string range $aStr end-[expr $Half-2] end]    
  } else {
    set PathBegin [string range $aStr 0 [expr $Half-2]]
    set PathEnd [string range $aStr end-[expr $Half-2] end]    
  }
  
  set Str [join [list $PathBegin ... $PathEnd] {}]
  if {[string length $Str] != $aMax} { puts "ShortString: Implementation error." }
  return $Str
}

proc MsgBox {aMsg} {
  if {[info commands Obj] != ""} {
    # variable ::Obj could exists. This happens for example if you select "Help" button and press space key twice very quickly.
    return Cancel
  }

  CMsgDialog Obj . .msg WineSetupTk Message $aMsg OK OK
  Obj DoModal
  delete object Obj
}

proc MsgDialog {aWinTitle aTitle aMsg aButtons aDefault} {
  if {[info commands Obj] != ""} {
    # variable ::Obj could exists. This happens for example if you select "Help" button and press space key twice very quickly.
    return Cancel
  }

  CMsgDialog Obj . .dialog $aWinTitle $aTitle $aMsg $aButtons $aDefault
  set Result [Obj DoModal]
  delete object Obj
  return $Result
}

proc MsgScrollBox {aMsg} {
  if {[info commands Obj] != ""} {
    # variable ::Obj could exists. This happens for example if you select "Help" button and press space key twice very quickly.
    return Cancel
  }
  
  CMsgScrollDialog Obj . .msg WineSetupTk Message $aMsg OK OK
  Obj DoModal
  delete object Obj
}

proc MsgScrollDialog {aWinTitle aTitle aMsg aButtons aDefault {apWinGeom ""}} {
  if {[info commands Obj] != ""} {
    # variable ::Obj could exists. This happens for example if you select "Help" button and press space key twice very quickly.
    return Cancel
  }

  if {$apWinGeom == ""} {
    set WinGeom ""
  } else {
    eval set WinGeom $$apWinGeom
  }
  CMsgScrollDialog Obj . .dialog $aWinTitle $aTitle $aMsg $aButtons $aDefault $WinGeom
  set Result [Obj DoModal]
  if {$apWinGeom != ""} { set $apWinGeom [Obj GetGeom] }
  delete object Obj
  return $Result
}

proc TakeGlobalVarsSnapshot {} {
  global GVSIndex
  global GVSnapshot
  if {![info exists GVSIndex]} { set GVSIndex -1 }
  if {![info exists GVSnapshot]} { set GVSnapshot(0) "" }
  incr GVSIndex
  set GVSnapshot($GVSIndex) [info globals]
  return $GVSIndex
}

proc DeleteNewGlobalVars {} {
  global GVSIndex
  global GVSnapshot
  
  if {![info exists GVSIndex] || $GVSIndex<0} {
    puts "DeleteNewGlobalVars: No snapshot on stack."
    exit
  }  

  set OldSnapshot $GVSnapshot($GVSIndex)
  set NewSnapshot [info globals]

  foreach Var $NewSnapshot {
    if {[lsearch $OldSnapshot $Var] < 0} { puts "$Var unset" ; unset ::$Var }
  }

  unset GVSnapshot($GVSIndex)
  incr GVSIndex -1
  return [expr $GVSIndex+1]
}

proc Translate {aItem aList1 aList2} {
  foreach Val1 $aList1 Val2 $aList2 { 
    if {$aItem == $Val1} {
      return $Val2
    }
  }
  return ""
}

proc BackupFile {aFileName apErrMsg} {
  upvar $apErrMsg ErrMsg
  if {[catch {exec date +%Y.%m.%d.%H.%M} Date]} {
    puts "BackupFile: Missing date command: $Date"
    set BakFN "${aFileName}.bak"
  } else {
    set BakFN "${aFileName}.${Date}.bak"
  }

  if {![file readable $aFileName]} {
    set ErrMsg "Error opening $aFileName for reading."
    return ""
  }
  if {![IsFileWritable $BakFN]} { 
    set ErrMsg "Error opening $BakFN for writing."
    return ""
  }

  set CatchRes [catch {file copy -force $aFileName $BakFN} CopyRes]
  if {$CatchRes != 0} {
    set ErrMsg "Error while copying file $aFileName to $BakFN:\n$CopyRes"
    return ""
  }

  set ErrMsg ""  
  return $BakFN
}

proc IsFileWritable {aFN} {
  set FN [file nativename $aFN]
  if {[file exists $FN]} {
    return [file writable $FN]
  } else {
    set AlreadyExists $FN
    set Count 0
    while {![file exists $AlreadyExists]} { set AlreadyExists [file dirname $AlreadyExists] ; incr Count }
    set FirstNonExists $FN
    for {set i 1} {$i < $Count} {incr i} { set FirstNonExists [file dirname $FirstNonExists] }
 
    catch {file mkdir $FN}
    set Result [file writable $FN]
    catch {file delete -force $FirstNonExists}
    return $Result
  }
}

proc Unix2Dos {aPath args} {
  set Path $aPath
  if {$args != ""} {
    set Root [lindex $args 0]
    if {[string range $Root end end] != "/"} {
      set Root $Root/
    }
    set Letter [lindex $args 1]
    regsub $Root $Path "" Path
    set Path $Letter:\\$Path
  }
  regsub -all / $Path \\ Path
  return $Path
}

proc Dos2Unix {aPath args} {
  set Path $aPath
  if {$args != ""} {
    set Root [lindex $args 0]
    if {[string range $Root end end] != "/"} {
      set Root $Root/
    }
    regsub {^.:\\} $Path $Root Path
  } 
  regsub -all {\\} $Path / Path
  return $Path
}

proc CreateEmptyFile {aFile apErrMsg} {
  upvar $apErrMsg ErrMsg

  if {[catch {open $aFile w} FileID]} {
    set ErrMsg "Error creating empty file $aFile"
    return ERROR
  }
  close $FileID

  return OK
}

proc CreateTreeFromTxt {aPrefix aTxtFile apErrMsg} {
  upvar $apErrMsg ErrMsg

  if {[catch {open $aTxtFile r} FileID]} {
    set ErrMsg "CreateTreeFromTxt: Error opening file $aTxtFile."
    return OPEN_ERROR
  }

  set LineNum 0
  while {[eof $FileID] == 0} {
    incr LineNum
    set Line [string trim [gets $FileID]]
    if {$Line == ""} { continue }

    set FirstChar [string range $Line 0 0]
    if {$FirstChar == "#"} { continue }

    if {[llength $Line] < 4} {
      set ErrMsg "CreateTreeFromTxt: Syntax error on line $LineNum in file $aTxtFile."
      close $FileID
      return SYNTAX_ERROR
    }

    set Permissions [lindex $Line 0]
    set Owner       [lindex $Line 1]
    set Group       [lindex $Line 2]
    set File        [join [list $aPrefix [lrange $Line 3 end]] "/"]

    set FileType [string range $Permissions 0 0]
    set Permissions [string range $Permissions 1 end]
    
    # Tcl8.0 understand only numbered permission
    set PermNum 0
    set AddNum 1
    for {set i 8} {$i >= 0} {incr i -1} {
      if {[string index $Permissions $i] != "-"} { set PermNum [expr $PermNum+$AddNum] }
      set AddNum [expr $AddNum*2]
    }
    set Permissions $PermNum

    switch -- $FileType {
      "d" {
         # directory
         if {[catch {file mkdir $File} ErrMsg]} {
           set ErrMsg "CreateTreeFromTxt: File $aTxtFile line $LineNum: $ErrMsg"
           close $FileID
           return CREATE_ERROR
         }
      }
      "-" {
         # file
         if {[CreateEmptyFile $File ErrMsg] != "OK"} {
           set ErrMsg "CreateTreeFromTxt: File $aTxtFile line $LineNum: $ErrMsg"
           close $FileID
           return CREATE_ERROR
         }
      }
      default { 
        set ErrMsg "CreateTreeFromTxt: Unknown file type ($FileType) on line $LineNum in file $aTxtFile."
        close $FileID
        return SYNTAX_ERROR
      }
    } ;# switch

    if {[catch {file attributes $File -permissions $Permissions} ErrMsg]} {
      set ErrMsg "CreateTreeFromTxt: File $aTxtFile line $LineNum: $ErrMsg"
      close $FileID
      return CREATE_ERROR
    }

    # goto trick
    while 1 {
      if {$Owner == "*" && [catch {exec whoami} Owner]} {
        puts "CreateTreeFromTxt: error executing 'whoami' command: $Owner"
        break
      }
      if {$Group == "*" && [catch {exec id -gn} Group]} {
        puts "CreateTreeFromTxt: error executing 'id -gn' command: $Group"
        break
      }
      if {[catch {file attributes $File -owner $Owner -group $Group} ErrMsg]} {
        set ErrMsg "CreateTreeFromTxt: File $aTxtFile line $LineNum: $ErrMsg"
        break
      }
      break
    }

  } ;# while
  
  close $FileID

  set ErrMsg ""
  return OK
}

proc GotoWebSite {aURL} {
  catch {exec netscape $aURL &} PID
  if {[regexp {[^0-9]} $PID]} {
    MsgDialog "Error" ":-(" "Unable to launch your WEB browser. Please do it manually." OK OK
    return ERROR
  }
  return OK
}

# convert an XPM image to P6 PPM format. $aTransparentColor is hex triplet (#AB56CD) and is used as the substitution for transparent color, becouse this 
# feature isn't supported in PPM.
proc xpm2ppm {aFName apErrMsg {aTransparentCol #DEDADE}} {
  upvar $apErrMsg ErrMsg

  if {[catch {open $aFName r} FileInID]} {
    set ErrMsg "xpm2ppm: $FileInID"
    return OPEN_ERROR
  }

  set OutFName [file rootname $aFName].ppm
  if {[catch {open $OutFName w} FileOutID]} {
    set ErrMsg "xpm2ppm: $FileOutID"
    close $FileInID
    return OPEN_ERROR
  }

  set LineNum 0
  set Line [gets $FileInID]
  incr LineNum
  # check header
  if {$Line != "/* XPM */"} {
    set ErrMsg "xpm2ppm: Error in file $aFName on line $LineNum. Expected \"/* XPM */\"."
    close $FileInID
    close $FileOutID
    return XPM_FORMAT_ERROR
  }

  # skip one line, it should be "static char ..."
  gets $FileInID
  incr LineNum
  
  # read width, height, number of colors and number of bytes used per color
  set Line [gets $FileInID]
  incr LineNum
  set Line [string range $Line 1 end-2]
  if {[llength $Line] != 4} {
    set ErrMsg "xpm2ppm: Error in file $aFName on line $LineNum. String with 4 decimal values expected. Note: hotspots and extensions aren't supported."
    close $FileInID
    close $FileOutID    
    return XPM_FORMAT_ERROR
  }
  set Width [lindex $Line 0]
  set Height [lindex $Line 1]
  set NColors [lindex $Line 2]
  set CharsPerPixel [lindex $Line 3]

  # below this line we expect that the file is valid XPM format

  # write it to out file
  # magic number
  puts $FileOutID "P6"
  # size
  puts $FileOutID "$Width $Height"
  # maximal value of RGB elements - always 255
  puts $FileOutID "255"

  # reads palette
  for {set i 0} {$i < $NColors} {incr i} {
    set Line [gets $FileInID]
    incr LineNum
    set ColorIdx [string_ range $Line 1 $CharsPerPixel]
    set RGB [lindex [split $Line] end]
    if {[string first None $RGB] >=0} { set RGB $aTransparentCol }
    set R [string_ range $RGB 1 2]
    set G [string_ range $RGB 3 4]
    set B [string_ range $RGB 5 6]
    
    set Palette($ColorIdx) [binary format H2H2H2 $R $G $B]
  }

  # reads image itself
  set Max [expr $Width*$CharsPerPixel]
  for {set i 0} {$i < $Height} {incr i} {
    set Line [gets $FileInID]
    set Line [string_ range $Line 1 $Max]
    for {set j 0} {$j < $Max} {incr j $CharsPerPixel} {
      set ColorIdx [string_ range $Line $j [expr $j+$CharsPerPixel-1]]
      puts -nonewline $FileOutID $Palette($ColorIdx)
    }
  }

  close $FileInID
  close $FileOutID
  return OK
}

proc IncrAsciiCode {aLetter aInc} {
  binary scan $aLetter c AsciiCode
  incr AsciiCode $aInc
  return [binary format c $AsciiCode]
}

# converts from unicode to ansi
# FIXME: poor unicode support
proc StrWtoA {aStr} {
  # change '\"' for '"'
  regsub -all {\\"} $aStr {"} Result
  # change '\\' for '\'
  regsub -all {\\\\} $Result {\\} Result
  return $Result
}

# converts from ansi to unicode
# FIXME: poor unicode support
proc StrAtoW {aStr} {
  # change '\' for '\\'
  regsub -all {\\} $aStr {\\\\} Result
  # change '"' for '\"'
  regsub -all {\"} $Result {\"} Result
  return $Result
}


