/* ==================================================== ======== ======= *
 *
 *  udoc.cpp
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uudoc.cpp	ubit:03.05.06"
#include "config.h"
#include <string.h>
#include <stdio.h>
#include <iostream>
#include <unistd.h>
#include "ubit/ubit.hpp"
#include "ubit/udir.hpp"
#include "ubit/ext/udoc.hpp"
using namespace std;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UDocManager* UDoc::manager;

struct UDocManager {
  struct Mapping {
    const char* suffix;
    class UDoc::Reader* reader;
  };
  std::vector<Mapping> readers;
  UDoc::Reader* default_reader;
  
  UDocManager();
  
  void setDefaultReader(UDoc::Reader* reader);
  void addReader(const char* suffix, UDoc::Reader* reader);
};

/* ==================================================== ======== ======= */

UDocManager::UDocManager() {
  default_reader = null;

  UImaDoc::Reader* image_reader = new UImaDoc::Reader();
  addReader("xpm", image_reader);
  addReader("jpg", image_reader);
  addReader("jpeg",image_reader);
  addReader("gif", image_reader);
  
  UTextDoc::Reader* text_reader = new UTextDoc::Reader();
  setDefaultReader(text_reader);

  addReader("txt", text_reader);
  addReader("java",text_reader);
  addReader("c",   text_reader);
  addReader("h",   text_reader);
  addReader("cpp", text_reader);
  addReader("hpp", text_reader);
  addReader("cc",  text_reader);
  addReader("hh",  text_reader);

  UExternalDoc::Reader* pdf_reader =
    new UExternalDoc::Reader("PDF Document", "acroread");
  addReader("pdf", pdf_reader);

  UExternalDoc::Reader* office_reader =
    new UExternalDoc::Reader("Office Document", "soffice");
  addReader("doc", office_reader);
  addReader("ppt", office_reader);
  addReader("xls", office_reader);

  UExternalDoc::Reader* bin_reader =
    new UExternalDoc::Reader("Binary File", "");
  addReader("o", bin_reader);
  addReader("a", bin_reader);
  addReader("la", bin_reader);
  addReader("so", bin_reader);
  addReader("dynlib", bin_reader);
}

/* ==================================================== ======== ======= */

void UDocManager::setDefaultReader(UDoc::Reader* reader) {
  default_reader = reader;
}

void UDocManager::addReader(const char* suffix, UDoc::Reader* reader) {
  if (!suffix || !*suffix || !reader) {
    UError::error("warning@UDoc::addReader","null suffix or null reader");
    return;
  }

  Mapping mp;
  mp.suffix = strdup(suffix);
  mp.reader = reader;
  readers.push_back(mp);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UDoc::UDoc(const UStr& _pathname) :
  pathname(_pathname) {
}

UDoc* UDoc::readDoc(const UStr& _pathname, Callbacks* call, Errors* err) {

  if (!manager) manager = new UDocManager();
  
  UStr fext = _pathname.getFileSuffix();
  const char* fext_chars = fext.chars();

  if (!fext_chars || !*fext_chars) {
    if (manager->default_reader)
      return manager->default_reader->read(_pathname, call, err);
    else return null;
  }

  for (int k = manager->readers.size()-1; k >= 0; k--) {
    if (strcasecmp(manager->readers[k].suffix, fext_chars) == 0) {
      if (manager->readers[k].reader)
        return manager->readers[k].reader->read(_pathname, call, err);
      else return null;
    }
  }

  // not found
  if (manager->default_reader)
    return manager->default_reader->read(_pathname, call, err);
  return null;
}

/* ==================================================== ======== ======= */

void UDoc::setDefaultReader(UDoc::Reader* reader) {
  if (!manager) manager = new UDocManager();
  manager->setDefaultReader(reader);
}

void UDoc::addReader(const char* suffix, UDoc::Reader* reader) {
  if (!manager) manager = new UDocManager();
  manager->addReader(suffix, reader);
}

/* ==================================================== ======== ======= */

void UDoc::Errors::set(int _stat) {
  stat =_stat;
  messages.clear();
}

void UDoc::Errors::set(int _stat, const UStr& _messages) {
  stat = _stat;
  messages = _messages;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UTextDoc::UTextDoc(const UStr& _pathname, UStr& _content) : UDoc(_pathname) {
  addAttr(uedit());   // !!editable
  add(_content);
}

UDoc* UTextDoc::Reader::read(const UStr& _pathname,
                             Callbacks* call, Errors* err) {
  UStr* str = new UStr();
  int stat;
  
  if ((stat = str->readFile(_pathname)) <= 0) {
    delete str;
    if (err) err->set(stat);
    return null;
  }
  else {
    if (err) err->set(UFilestat::Opened);
    return new UTextDoc(_pathname, *str);
  }
}

/* ==================================================== ======== ======= */

UImaDoc::UImaDoc(const UStr& _pathname, UIma& _content): UDoc(_pathname) {
  add(_content);
}

UDoc* UImaDoc::Reader::read(const UStr& _pathname,
                            Callbacks* call, Errors* err) {
  UIma* ima = new UIma();
  int stat;
  
  if ((stat = ima->readFile(_pathname)) <= 0) {
    delete ima;
    if (err) err->set(stat);
    return null;
  }
  else {
    if (err) err->set(UFilestat::Opened);
    return new UImaDoc(_pathname, *ima);
  }
}

/* ==================================================== ======== ======= */

UExternalDoc::UExternalDoc(const UStr& _pathname,
                           const UStr& _doctype, 
                           const UStr& _command) :
  UDoc(_pathname),
  doctype(_doctype),
  command(_command)
{
  addlist
  (
   uhmargin(10) + uvmargin(10) + uvspacing(10)
   + UFont::large
   + ugroup(UFont::bold+ "File: ")
   + ulinkbutton(pathname + ucall(this, &UExternalDoc::openDoc))
   + "\n"
   + ugroup(UFont::bold + "Type: " + doctype)
   + "\n"
   + "Open with: " + utextfield(uwidth(150) + UColor::orange + command)
   + "\n"
   + status
   );
}

void UExternalDoc::openDoc() {
  UStr s = command & " " & pathname & "&";
  int stat = system(s.chars());
  status = "Status: ";
  status &= UIntg(stat);
}

UExternalDoc::Reader::Reader(const char* _doctype, const char* _command) :
  doctype(_doctype),
  command(_command) {
}

UDoc* UExternalDoc::Reader::read(const UStr& _pathname,
                                 Callbacks* call, Errors* err) {
  return new UExternalDoc(_pathname, doctype, command);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UDocbox::UDocbox(const UArgs& a) {
  pname    = ustr();
  pscale   = uscale();
  pglass   = ubox();

  pcontent = uvbox(a);

  // scale attache au doc pas au glass !
  pcontent->addAttr(*pscale);  // tjrs en 1er
  pcontent->addAttr(uvmargin(6));
  pcontent->addAttr(uhmargin(6));
  pcontent->addAttr(utop());
  pcontent->addAttr(uhflex());
  pcontent->addAttr(UBgcolor::white);

  pglass->addAttr(UBgcolor::none);
  pglass->addAttr(upos(0,0));
  pglass->addAttr(uwidth(50000));
  pglass->addAttr(uheight(100000)); // a revoir

  //addAttr(*pscale);  // scale attache au doc pas au glass !
  addlist(*pcontent + *pglass);
}

/* ==================================================== ======== ======= */

void UDocbox::setContent(const UStr& _name, UDoc& _doc) {
  *pname = _name;
  pdoc = _doc;
  // pglass->removeAll(true, false);// NON!!
  pcontent->removeAll(true, false); 
  pcontent->add(_doc, false);
  pcontent->update();
  //update();
}

void UDocbox::clearContent() {
  *pname = "";
  pdoc = null;
  //pglass->removeAll(true, false);// NON!!
  pcontent->removeAll(true, false);
  pcontent->update();
  //update();
}

/* ==================================================== ======== ======= */

const UStr& UDocbox::getName() const {
  return *pname;
}

void UDocbox::setName(const UStr& _name) {
  *pname = _name;
}

void UDocbox::setScale(int val) {
  pscale->set(val);
}

void UDocbox::incrScale(int val) {
  pscale->incr(val);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

