//
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/geo_domain.h"
#include "rheolef/test.h"

namespace rheolef {

template <class T, class M>
test_rep<T,M>::test_rep (const space_type& V)
: _V(V),
  _opt(),
  _dom(),
  _basis_on_quad(),
  _is_on_band(false),
  _ignore_sys_coord(false),
  _gh(),
  _tilde_L(),
  _dis_inod_K(),
  _dis_inod_L(),
  _piola_on_quad(),
  _elements_on_bgd_dom(false),
  _have_dg_on_sides(false)
{
}
template <class T, class M>
test_rep<T,M>::test_rep (const test_rep<T,M>& x)
: _V(x._V),
  _opt(x._opt),
  _dom(x._dom),
  _basis_on_quad(x._basis_on_quad),
  _is_on_band(x._is_on_band),
  _ignore_sys_coord(x._ignore_sys_coord),
  _gh(x._gh),
  _tilde_L(x._tilde_L),
  _dis_inod_K(x._dis_inod_K),
  _dis_inod_L(x._dis_inod_L),
  _piola_on_quad(x._piola_on_quad),
  _elements_on_bgd_dom(x._elements_on_bgd_dom),
  _have_dg_on_sides(x._have_dg_on_sides)
{
  trace_macro ("*** PHYSICAL COPY OF TEST_REP ***");
}
template <class T, class M>
void
test_rep<T,M>::initialize (const geo_basic<float_type,M>& dom, const quadrature<T>& quad, bool ignore_sys_coord) const
{
  _is_on_band = false;
  _elements_on_bgd_dom = false;
  _have_dg_on_sides = false;
  if (_V.get_geo().variant() != geo_abstract_base_rep<float_type>::geo_domain || _V.get_geo().name() == dom.name()) {
    if (dom.variant() != geo_abstract_base_rep<T>::geo_domain_indirect
        || 1 + dom.map_dimension() != dom.get_background_geo().map_dimension()
	|| _V.get_numbering().is_continuous()) {
      // usual case
      _dom = dom;
    } else {
      // DG on dom="sides" or "boundary" or other d-1 sides domain
      // but for a side K, will access to its neighbours L0 & L1 for e.g. dot(grad(u),normal())
      _dom = dom.get_background_geo();
      _have_dg_on_sides = true;
    }
  } else {
    // space X (omega);
    // space Y (omega["boundary"]);
    // trial u(X);
    // test  v(Y);
    // form m = integrate ("boundary", u*v);
    //  => v is on compacted geo_domain Y.geo but elements K 
    //    during integration process are on domain omega["boundary"]
    //  => element K need to be replaced by an element in Y.geo
    //     for computing correctly Y.dis_idof(K)
    check_macro (_V.get_geo().get_background_domain().name() == dom.name(),
	"unexpected integration domain \"" << dom.name() << "\" for space based on domain \"" 
		<< _V.get_geo().get_background_domain().name()
		<<"\" with geo \"" << _V.get_geo().name() << "\"");
    _dom = _V.get_geo();
    _elements_on_bgd_dom = true;
  }
  _basis_on_quad.set (quad, get_vf_space().get_numbering().get_basis());
  _piola_on_quad.set (quad, get_vf_space().get_geo().get_piola_basis());
  _ignore_sys_coord = ignore_sys_coord;
}
template <class T, class M>
void
test_rep<T,M>::initialize (const band_basic<float_type,M>& gh, const quadrature<T>& quad, bool ignore_sys_coord) const
{
  _is_on_band = true;
  _elements_on_bgd_dom = false;
  _have_dg_on_sides = false;
  _dom = gh.level_set();
  _gh = gh;
  _basis_on_quad.set (quad, get_vf_space().get_numbering().get_basis());
  _piola_on_quad.set (quad, get_vf_space().get_geo().get_piola_basis());
  _ignore_sys_coord = ignore_sys_coord;
}
template <class T, class M>
void
test_rep<T,M>::initialize (const  space_basic<float_type,memory_type>& Xh, bool ignore_sys_coord) const
{
  _is_on_band = false;
  _dom = Xh.get_geo();
  _basis_on_quad.set (Xh.get_numbering().get_basis(), get_vf_space().get_numbering().get_basis());
  _piola_on_quad.set (Xh.get_numbering().get_basis(), get_vf_space().get_geo().get_piola_basis());
  _ignore_sys_coord = ignore_sys_coord;
}
template <class T, class M>
typename test_rep<T,M>::size_type
test_rep<T,M>::element_initialize (const geo_element& K) const
{
  if (!_is_on_band) {
    if (!_elements_on_bgd_dom) {
      if (!_have_dg_on_sides || K.dimension() == _dom.map_dimension()) {
        _dom.dis_inod (K, _dis_inod_K);
      } else {
	// omits "inner(u)" on a sides-domain => assume a boundary side
        size_type L_map_d = K.dimension() + 1;
        check_macro (L_map_d == _dom.map_dimension(),
	  "unexpected dimension for side K.dis_ie={"<<K.dis_ie()<<"} in domain "<<_dom.name());
        check_macro (K.master(1) == std::numeric_limits<size_type>::max(),
          "unexpected non-boundary side K.dis_ie="<<K.dis_ie());
        size_type L_dis_ie = K.master(0);
        check_macro (L_dis_ie != std::numeric_limits<size_type>::max(),
          "unexpected isolated side K.dis_ie="<<K.dis_ie());
        const geo_element& L = _dom.dis_get_geo_element (L_map_d, L_dis_ie);
        _tilde_L = L.variant();
        _dom.dis_inod (L, _dis_inod_K);
        side_information_type sid;
        L.get_side_informations (K, sid);
        _basis_on_quad.restrict_on_side (L, sid);
        _piola_on_quad.restrict_on_side (L, sid);
      }
    } else {
      const geo_element& dom_K = _dom.bgd2dom_geo_element (K);
      _dom.dis_inod (dom_K, _dis_inod_K);
    }
    return _basis_on_quad.pointset_size (K);
  } else {
    // find L in the band such that K is the intersected side of L
    // as the zero level set isosurface
    size_type first_dis_ie = _gh.level_set().sizes().ownership_by_dimension [K.dimension()].first_index();
    size_type K_ie = K.dis_ie() - first_dis_ie;
    check_macro (K.dis_ie() >= first_dis_ie, "unexpected intersected side element K.dis_ie="<<K.dis_ie());
    size_type L_ie = _gh.sid_ie2bnd_ie (K_ie);
    const geo_element& L = _gh.band() [L_ie];
    _tilde_L = L.variant();
    _gh.level_set().dis_inod (K, _dis_inod_K);
    _gh.band().dis_inod      (L, _dis_inod_L);
    return _basis_on_quad.pointset_size (_tilde_L);
  }
}
template <class T, class M>
typename test_rep<T,M>::size_type
test_rep<T,M>::element_initialize_on_side (const geo_element& K, const side_information_type& sid) //BUG with const for DG
{
  _basis_on_quad.restrict_on_side (K, sid);
  _piola_on_quad.restrict_on_side (K, sid);
  size_type nloc = element_initialize (K);
  return nloc;
}
// ----------------------------------------------    
// basis
// ----------------------------------------------    
template <class T, class M>
void
test_rep<T,M>::_basis_evaluate_init (const reference_element& hat_K, size_type q, size_type loc_ndof) const
{
  if (!_is_on_band) {
    reference_element ref_K = hat_K;
    if (_have_dg_on_sides && hat_K.dimension() != _dom.map_dimension()) {
      // DG on boundary: L has a face that contains K
      ref_K = _tilde_L;
    }
    check_macro (loc_ndof == get_vf_space().get_constitution().loc_ndof (ref_K), 
	"invalid value local ndof="<<loc_ndof<<": expect loc_ndof="
  	<<get_vf_space().get_constitution().loc_ndof (ref_K)
  	<<" with approx " << get_vf_space().get_approx()
        <<" on element variant=" << ref_K.name());
    _basis_on_quad.evaluate (ref_K, q);
  } else {
    check_macro (loc_ndof == get_vf_space().get_constitution().loc_ndof (_tilde_L), 
	"invalid value local ndof="<<loc_ndof<<": expect loc_ndof="
  	<<get_vf_space().get_constitution().loc_ndof (_tilde_L));
    // use the basis on L, the element from the band that contains the 
    // intersected side K on the zero level set surface
    point_basic<T> xq = piola_transformation (_gh.level_set(), _piola_on_quad, hat_K, _dis_inod_K, q);
    point_basic<T> tilde_xq = inverse_piola_transformation (_gh.band(), _tilde_L, _dis_inod_L, xq);
    _basis_on_quad.evaluate (_tilde_L, tilde_xq);
  }
}
template <class T, class M>
void
test_rep<T,M>::_basis_evaluate (const reference_element& hat_K, size_type q, std::vector<T>& value) const
{
  size_type boq_sz = _basis_on_quad.pointset_size(hat_K);
  _basis_evaluate_init (hat_K, q, value.size());
  for (size_type loc_idof = 0; loc_idof < value.size(); ++loc_idof) {
    value[loc_idof] = _basis_on_quad.value (loc_idof); 
  }
}
template <class T, class M>
void
test_rep<T,M>::_basis_evaluate (const reference_element& hat_K, size_type q, std::vector<point_basic<T> >& value) const
{
  // LIMITATION: assume vector as a product of scalar basis
  _basis_evaluate_init (hat_K, q, value.size());
  size_type n_comp = get_vf_space().size();
  size_type loc_comp_ndof = value.size() / n_comp;
  fill (value.begin(), value.end(), point_basic<T>()); // do not remove !
  for (size_type loc_comp_idof = 0; loc_comp_idof < loc_comp_ndof; ++loc_comp_idof) {
    T val = _basis_on_quad.value (loc_comp_idof); 
    for (size_type i_comp = 0; i_comp < n_comp; ++i_comp) {
      size_type loc_idof = i_comp*loc_comp_ndof + loc_comp_idof;
      value[loc_idof][i_comp] = val;
    }
  }
}
template <class T, class M>
void
test_rep<T,M>::_basis_evaluate (const reference_element& hat_K, size_type q, std::vector<tensor_basic<T> >& value) const
{
  // LIMITATION: assume tensor as a product of scalar basis
  _basis_evaluate_init (hat_K, q, value.size());
  size_type n_comp = get_vf_space().size();
  size_type loc_comp_ndof = value.size() / n_comp;
  fill (value.begin(), value.end(), tensor_basic<T>()); // do not remove !
  space_constant::valued_type valued_tag = get_vf_space().valued_tag();
  space_constant::coordinate_type sys_coord = _dom.coordinate_system();
  for (size_type loc_comp_idof = 0; loc_comp_idof < loc_comp_ndof; ++loc_comp_idof) {
    T val = _basis_on_quad.value (loc_comp_idof); 
    for (size_type ij_comp = 0; ij_comp < n_comp; ++ij_comp) {
      size_type loc_idof = ij_comp*loc_comp_ndof + loc_comp_idof;
      std::pair<size_type,size_type> ij = space_constant::tensor_subscript (space_constant::tensor, sys_coord, ij_comp);
      value[loc_idof](ij.first, ij.second) = val;
      if (valued_tag == space_constant::tensor && ij.first != ij.second) { 
	// symmetric tensor: otherwise valued_tag == space_constant::unsymmetric_tensor
        value[loc_idof](ij.second, ij.first) = val;
      }
    }
  }
}
template <class T, class M>
void
test_rep<T,M>::_basis_evaluate (const reference_element& hat_K, size_type q, std::vector<tensor3_basic<T> >& value) const
{
  fatal_macro ("basis_evaluate: tensor3, not yet");
}
template <class T, class M>
void
test_rep<T,M>::_basis_evaluate (const reference_element& hat_K, size_type q, std::vector<tensor4_basic<T> >& value) const
{
  fatal_macro ("basis_evaluate: tensor4, not yet");
}
// ----------------------------------------------    
// grad basis
// ----------------------------------------------    
/*
 Let F be the Piola transformation from the reference element:
   F : hat_K --> K
       hat_x --> x = F(hat_x)
 Then the gradient of a basis function u defined on K writes:
   grad(u)(xq) = DF^{-T}*hat_grad(hat_u)(hat_xq)

 When we are working with K as a side of a banded level set surface {phi(x)=0},
 then things are more complex.
 Let
   L in Lambda : a element of the bounding box Lambda
   K in Omega  : a side of the surface Omega, included in L, as K = L inter {phi(x)=0}.
 Let the two Piola transformations from the reference elements:
   F : tilde_L --> L
       tilde_x --> x = F(tilde_x)
   G : hat_K --> K
       hat_x --> x = G(hat_x)

 Let u is a basis function on L: it is defined over tilde_L as u_tilde:
   u(x) = tilde_u (F^{-1}(x))    for all x in K

 The quadrature formula is defined in hat_K with integration point hat_xq and weights hat_wq.
 Thus, integration point hat_xq may be transformed via F^{-1} o G.
 Then, for x=xq=G(hat_xq) :

   u(G(hat_xq)) = tilde_u(F^{-1}(G(hat_xq)))

 Its derivative expresses with a product of the jacobian DF^-T(tilde_x) :

   grad(u)(x) = DF^-T (F^{-1}(x)) grad(u)(F^{-1}(x))  for all x in K

 and then, for x=xq=G(hat_xq)

   grad(u)(G(hat_xq)) = DF^-T (F^{-1}(G(hat_xq))) grad(u)(F^{-1}(G(hat_xq)))

*/
template <class T, class M>
void
test_rep<T,M>::_grad_basis_evaluate_init (const reference_element& hat_K_in, size_type q, size_type loc_ndof) const
{
  reference_element hat_K = hat_K_in;
  if (_have_dg_on_sides && hat_K.dimension() != _dom.map_dimension()) {
      // DG on boundary: L has a face that contains K
      hat_K = _tilde_L;
  }
  if (!_is_on_band) {
    check_macro (loc_ndof == get_vf_space().get_constitution().loc_ndof (hat_K), 
	"invalid value local ndof="<<loc_ndof<<": expect loc_ndof="
  	<< get_vf_space().get_constitution().loc_ndof (hat_K));
  } else {
    check_macro (loc_ndof == get_vf_space().get_constitution().loc_ndof (_tilde_L), 
	"invalid value local ndof="<<loc_ndof<<": expect loc_ndof="
  	<<get_vf_space().get_constitution().loc_ndof (_tilde_L));
  }
}
template <class T, class M>
void
test_rep<T,M>::_grad_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<T>& value) const
{
  fatal_macro ("grad: unexpected scalar-valued result");
}
// grad(scalar) -> vector
template <class T, class M>
void
test_rep<T,M>::_grad_basis_evaluate (const reference_element& hat_K_in, size_type q, const details::grad_option_type& opt, std::vector<point_basic<T> >& value) const
{
  reference_element hat_K = hat_K_in;
  if (_have_dg_on_sides && hat_K.dimension() != _dom.map_dimension()) {
    // DG on boundary: L has a face that contains K
    hat_K = _tilde_L;
  }
  // grad(u) = DF^{-T}*hat_grad(hat_u)
  _grad_basis_evaluate_init (hat_K, q, value.size());
  size_type       d =  _dom.dimension();
  size_type K_map_d = hat_K.dimension();
  tensor_basic<float_type> DF, invDF;
  if (!_is_on_band) {
    _basis_on_quad.evaluate_grad (hat_K, q);
    jacobian_piola_transformation (_dom, _piola_on_quad, hat_K, _dis_inod_K, q, DF);
    invDF = pseudo_inverse_jacobian_piola_transformation (DF, d, K_map_d);
  } else {
    point_basic<T> xq = piola_transformation (_gh.level_set(), _piola_on_quad, hat_K, _dis_inod_K, q);
    point_basic<T> tilde_xq = inverse_piola_transformation (_gh.band(), _tilde_L, _dis_inod_L, xq);
    _basis_on_quad.evaluate_grad (_tilde_L, tilde_xq);
    jacobian_piola_transformation (_gh.band(), _piola_on_quad, _tilde_L, _dis_inod_L, tilde_xq, DF);
    size_type L_map_d = _tilde_L.dimension();
    invDF = pseudo_inverse_jacobian_piola_transformation (DF, d, L_map_d);
    if (opt.surfacic) {
      tensor_basic<float_type> DG, P, invDF_P;
      // apply also the tangential projection P=(I-n*n)
      // grad(u) = P*DF^{-T}*hat_grad(hat_u) = (DF^{-1}*P)^T*hat_grad(hat_u)
      jacobian_piola_transformation (_dom, _piola_on_quad, hat_K, _dis_inod_K, q, DG);
      map_projector (DG, d, K_map_d, P);
      prod (invDF, P, invDF_P, L_map_d, d, d);
      invDF = invDF_P;
    }
  }
  for (size_type loc_idof = 0, loc_ndof = value.size(); loc_idof < loc_ndof; ++loc_idof) {
    // TODO: only multiplies with physical dimension d
    value[loc_idof] = invDF.trans_mult (_basis_on_quad.grad_value (loc_idof));
  }
}
// grad(vector) -> tensor
template <class T, class M>
void
test_rep<T,M>::_grad_basis_evaluate (const reference_element& hat_K_in, size_type q, const details::grad_option_type& opt, std::vector<tensor_basic<T> >& value) const
{
  reference_element hat_K = hat_K_in;
  if (_have_dg_on_sides && hat_K.dimension() != _dom.map_dimension()) {
    // DG on boundary: L has a face that contains K
    hat_K = _tilde_L;
  }
  _grad_basis_evaluate_init (hat_K, q, value.size());
  size_type       d = _dom.dimension();
  size_type K_map_d = hat_K.dimension();
  point_basic<T> xq;
  tensor_basic<float_type> invDF, P;
  if (!_is_on_band) {
    // --------------------------------------------------
    // usual case (volume or surface)
    // --------------------------------------------------
    _basis_on_quad.evaluate_grad (hat_K, q);
    tensor_basic<float_type> DF;
    jacobian_piola_transformation (_dom, _piola_on_quad, hat_K, _dis_inod_K, q, DF);
    invDF = pseudo_inverse_jacobian_piola_transformation (DF, d, K_map_d);
    if (opt.surfacic) {
      // will apply also the tangential projection P=(I-n*n)
      map_projector (DF, d, K_map_d, P);
    }
  } else {
    // --------------------------------------------------
    // banded level set case
    // --------------------------------------------------
    point_basic<T> xq = piola_transformation (_dom, _piola_on_quad, hat_K, _dis_inod_K, q);
    point_basic<T> tilde_xq = inverse_piola_transformation (_gh.band(), _tilde_L, _dis_inod_L, xq);
    _basis_on_quad.evaluate_grad (_tilde_L, tilde_xq);
    size_type L_map_d = _tilde_L.dimension();
    tensor_basic<float_type> DF;
    jacobian_piola_transformation (_gh.band(), _piola_on_quad, _tilde_L, _dis_inod_L, tilde_xq, DF);
    invDF = pseudo_inverse_jacobian_piola_transformation (DF, d, L_map_d);
    if (opt.surfacic) {
      // will apply also the tangential projection P=(I-n*n)
      tensor_basic<float_type> DG;
      jacobian_piola_transformation (_dom, _piola_on_quad, hat_K, _dis_inod_K, q, DG);
      map_projector (DG, d, K_map_d, P);
    }
  }
  // --------------------------------------------------
  // common code for usual and banded-level-set cases
  // --------------------------------------------------
  size_type n_comp = get_vf_space().size();
  size_type loc_comp_ndof = value.size() / n_comp;
  fill (value.begin(), value.end(), tensor_basic<T>()); // do not remove !
  for (size_type loc_comp_idof = 0; loc_comp_idof < loc_comp_ndof; ++loc_comp_idof) {
    // grad(phi) = DF^{-T}*grad_hat_phi
    point_basic<T> grad_phi = invDF.trans_mult (_basis_on_quad.grad_value (loc_comp_idof));
    if (! opt.symmetrized) {
      for (size_type i_comp = 0; i_comp < n_comp; ++i_comp) {
        size_type loc_idof = i_comp*loc_comp_ndof + loc_comp_idof;
        for (size_type j_comp = 0; j_comp < n_comp; ++j_comp) {
          value[loc_idof](i_comp, j_comp) = grad_phi [j_comp];
        }
      }
    } else { // compute D(u) instead of grad(u)
      for (size_type i_comp = 0; i_comp < n_comp; ++i_comp) {
        size_type loc_idof = i_comp*loc_comp_ndof + loc_comp_idof;
        for (size_type j_comp = 0; j_comp < n_comp; ++j_comp) {
          value[loc_idof](i_comp, j_comp) += 0.5*grad_phi [j_comp];
          value[loc_idof](j_comp, i_comp) += 0.5*grad_phi [j_comp];
        }
      }
    }
  }
  if (opt.surfacic) {
    // surfacic grad, where P=I-nxn is the map projector
    //    grad_s(u) =   grad(u)*P
    // or    Ds(u)  = P*D(u)*P
    for (size_type loc_idof = 0, loc_ndof = value.size(); loc_idof < loc_ndof; ++loc_idof) {
      // TODO: reduce product to d <= 3. faster when d=2
      // TODO: pre-multiplies invDF by P, then right mult by P and last symmetrize: faster ?
      if (! opt.symmetrized) {
        value[loc_idof] =   value[loc_idof]*P;
      } else {
        value[loc_idof] = P*value[loc_idof]*P;
      }
    }
  }
  // set the specific value(theta,theta) = vr/r term in the axi case
  space_constant::coordinate_type sys_coord = _dom.coordinate_system();
  if (_ignore_sys_coord || sys_coord == space_constant::cartesian) return;
  _basis_on_quad.evaluate (hat_K, q);
  xq = piola_transformation (_dom, _piola_on_quad, hat_K, _dis_inod_K, q);
  size_type i_comp_r     = (sys_coord == space_constant::axisymmetric_rz) ? 0 : 1;
  size_type i_comp_theta = 2;
  T r = xq[i_comp_r];
  check_macro (1+abs(r) != 1, "grad(): singular axisymmetric (1/r) weight (HINT: avoid interpolate() or change quadrature formulae)");
  // pb: r==0 when using gauss_lobatto, e.g. for the characteristic method
  //     e.g. in VF: 2D(u):D(v) dx -> (ur/r)*(vr/r)*r dr*dz = (ur*vr)/r dr*dz
  // or for direct interpolation of grad(uh), D(uh), div(uh), etc in the axisym. case
  for (size_type loc_comp_idof = 0; loc_comp_idof < loc_comp_ndof; ++loc_comp_idof) {
    size_type loc_idof = i_comp_r*loc_comp_ndof + loc_comp_idof;
    const T& phi = _basis_on_quad.value (loc_comp_idof);
    value[loc_idof](i_comp_theta, i_comp_theta) = phi/r;
  }
}
// grad(tensor) -> tensor3
template <class T, class M>
void
test_rep<T,M>::_grad_basis_evaluate (const reference_element& hat_K_in, size_type q, const details::grad_option_type& opt, std::vector<tensor3_basic<T> >& value) const
{
  reference_element hat_K = hat_K_in;
  if (_have_dg_on_sides && hat_K.dimension() != _dom.map_dimension()) {
    // DG on boundary: L has a face that contains K
    hat_K = _tilde_L;
  }
  // g(i,j,k) = d sigma(i,j) / d xk
  _grad_basis_evaluate_init (hat_K, q, value.size());
  size_type       d = _dom.dimension();
  size_type K_map_d = hat_K.dimension();
  check_macro (!_is_on_band,  "grad(tensor): band not yet supported");
  check_macro (!opt.surfacic, "grad(tensor): surfacic not yet supported");
  _basis_on_quad.evaluate_grad (hat_K, q);
  tensor_basic<float_type> DF;
  jacobian_piola_transformation (_dom, _piola_on_quad, hat_K, _dis_inod_K, q, DF);
  tensor_basic<float_type> invDF = pseudo_inverse_jacobian_piola_transformation (DF, d, K_map_d);
  //
  // then fill the value[loc_idoc](i,j,k) array
  fill (value.begin(), value.end(), tensor3_basic<T>()); // do not remove !
  size_type n_comp = get_vf_space().size();
  size_type loc_comp_ndof = value.size() / n_comp;
  space_constant::valued_type valued_tag = get_vf_space().valued_tag();
  space_constant::coordinate_type sys_coord = _dom.coordinate_system();
  bool do_symmetry = (valued_tag != space_constant::unsymmetric_tensor);
  for (size_type loc_comp_idof = 0; loc_comp_idof < loc_comp_ndof; ++loc_comp_idof) {
    // grad(phi) = DF^{-T}*grad_hat_phi
    point_basic<T> grad_phi = invDF.trans_mult (_basis_on_quad.grad_value (loc_comp_idof));
    for (size_type ij_comp = 0; ij_comp < n_comp; ++ij_comp) {
      size_type loc_idof = ij_comp*loc_comp_ndof + loc_comp_idof;
      std::pair<size_type,size_type> ij = space_constant::tensor_subscript (space_constant::tensor, sys_coord, ij_comp);
      for (size_type k_comp = 0; k_comp < n_comp; ++k_comp) {
        value[loc_idof](ij.first, ij.second, k_comp) = grad_phi [k_comp];
        if (do_symmetry && ij.first != ij.second) {
          value[loc_idof](ij.second, ij.first, k_comp) = grad_phi [k_comp];
        }
      }
    }
  }
}
template <class T, class M>
void
test_rep<T,M>::_grad_basis_evaluate (const reference_element& hat_K_in, size_type q, const details::grad_option_type& opt, std::vector<tensor4_basic<T> >& value) const
{
  fatal_macro ("grad_basis_evaluate: tensor4, not yet");
}
// ----------------------------------------------    
// div basis
// ----------------------------------------------    
template <class T, class M>
void
test_rep<T,M>::_div_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<T>& value) const
{
  std::vector<tensor_basic<T> > value1 (value.size());
  grad_basis_evaluate (hat_K, q, _opt, value1);
  for (size_t i = 0, n = value.size(); i < n; ++i) {
      value[i] = tr (value1[i]);
  }
}
template <class T, class M>
void
test_rep<T,M>::_div_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<point_basic<T> >& value) const
{
  // TODO: requires tensor3=grad(tensor) or a direct div(tensor) computation
  fatal_macro ("div(tensor): not yet");
}
template <class T, class M>
void
test_rep<T,M>::_div_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<tensor_basic<T> >& value) const
{
  fatal_macro ("div(tensor3): not yet");
}
// ----------------------------------------------    
// curl basis
// ----------------------------------------------    
template <class T, class M>
void
test_rep<T,M>::_curl_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<T>& value) const
{
  // scalar result => d=2 & arg is vector
  // curl(v) = dv1/dx0 - dv0/dx1 = g10 - g01 when g=grad(v)=dvi/dxj is a tensor
  std::vector<tensor_basic<T> > value1 (value.size());
  grad_basis_evaluate (hat_K, q, _opt, value1);
  for (size_t i = 0, n = value.size(); i < n; ++i) {
    const tensor_basic<T>& g = value1[i];
    value[i] = g(1,0) - g(0,1);
  }
  space_constant::coordinate_type sys_coord = _dom.coordinate_system();
  if (sys_coord == space_constant::cartesian) return;
  if (! opt.batchelor_curl) return;
  // Batchelor curl, for computing the stream function in axisymmetric geometries
  // scurl(v) = dvr/dz - dvz/dr - vz/r ==> substract vz/r
  std::vector<point_basic<T> > value2 (value.size());
  basis_evaluate (hat_K, q, value2);
  point_basic<T> xq = piola_transformation (_dom, _piola_on_quad, hat_K, _dis_inod_K, q);
  size_type i_comp_z     = (sys_coord == space_constant::axisymmetric_rz) ? 1 : 0;
  size_type i_comp_r     = (sys_coord == space_constant::axisymmetric_rz) ? 0 : 1;
  float_type r = xq[i_comp_r];
  check_macro (1+abs(r) != 1, "scurl(): singular axisymmetric (1/r) weight (HINT: avoid interpolate() or change quadrature formulae)");
  for (size_t loc_idof = 0, loc_ndof = value2.size(); loc_idof < loc_ndof; ++loc_idof) {
    value[loc_idof] -= value2[loc_idof][i_comp_z]/r; // -g(2,2) : already computed : no its vr/r, not vz/r !
  }
}
template <class T, class M>
void
test_rep<T,M>::_curl_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<point_basic<T> >& value) const
{
  // vector result => d=2 & arg is scalar or d=3 and arg is vector
  size_type d = _dom.dimension();
  if (d == 2) {
    // arg is scalar:
    // curl(w) = [dw/dx1 ; - dw/dx0] = [g1; -g0] when g=grad(v)=dw/dxi is a vector
    std::vector<point_basic<T> > value1 (value.size());
    grad_basis_evaluate (hat_K, q, _opt, value1);
    for (size_t i = 0, n = value.size(); i < n; ++i) {
      const point_basic<T>& g = value1[i];
      value[i][0] =  g[1];
      value[i][1] = -g[0];
    }
    space_constant::coordinate_type sys_coord = _dom.coordinate_system();
    if (sys_coord == space_constant::cartesian) return;
    // v = [vz,vr] = curl(w) = [dw/dr + w/r ; - dw/dz] : add w/r to the z-component
    if (opt.batchelor_curl) return; // Batchelor curl: do not perform this add
    std::vector<T> value2 (value.size());
    basis_evaluate (hat_K, q, value2);
    point_basic<T> xq = piola_transformation (_dom, _piola_on_quad, hat_K, _dis_inod_K, q);
    size_type i_comp_z     = (sys_coord == space_constant::axisymmetric_rz) ? 1 : 0;
    size_type i_comp_r     = (sys_coord == space_constant::axisymmetric_rz) ? 0 : 1;
    float_type r = xq[i_comp_r];
    check_macro (1+abs(r) != 1, "grad(): singular axisymmetric (1/r) weight (HINT: avoid interpolate() or change quadrature formulae)");
    for (size_t loc_idof = 0, loc_ndof = value2.size(); loc_idof < loc_ndof; ++loc_idof) {
      value[loc_idof][i_comp_z] += value2[loc_idof]/r;
    }
  } else {
    // d=3: arg is vector:
    //           [dw2/dx1 -dw1/dx2]   [ g21 - g12 ]
    // curl(w) = [dw0/dx2 -dw2/dx0] = [ g02 - g20 ] when g=grad(w)=dwi/dxj is a tensor 
    //           [dw1/dx0 -dw0/dx1]   [ g10 - g01 ]
    std::vector<tensor_basic<T> > value1 (value.size());
    grad_basis_evaluate (hat_K, q, _opt, value1);
    for (size_t i = 0, n = value.size(); i < n; ++i) {
      const tensor_basic<T>& g = value1[i];
      value[i][0] = g(2,1) - g(1,2);
      value[i][1] = g(0,2) - g(2,0);
      value[i][2] = g(1,0) - g(0,1);
    }
  }
}
template <class T, class M>
void
test_rep<T,M>::_curl_basis_evaluate (const reference_element& hat_K, size_type q, const details::grad_option_type& opt, std::vector<tensor_basic<T> >& value) const
{
  fatal_macro ("curl: unexpected tensor-valued result");
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------

#define _RHEOLEF_instanciation(T,M)	 			\
template class test_rep<T,M>;

_RHEOLEF_instanciation(Float,sequential)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_instanciation(Float,distributed)
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
