/*
 * QGLViewerXML.cpp
 * $Id: QGLViewerXML.cpp,v 1.6 2002/11/13 14:09:28 gwetekam Exp $
 *
 * Copyright (C) 2000, 2001 Michael Meissner
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */


/** documentation stuff

  @author Michael Meissner
    
  @version 0.0 //see cvs docu
          
*/

// QGLViewer
//////////////
#include "QGLViewerXML.h"


/////////////////////
// GENERIC HELPERS //
/////////////////////


// Function : addNode
// Purpose  : Creates a new xml node with parent parent and member attribute as
//            specified in member (optional). The node is added to the parent node. 
// Comments :
QDomElement QGLViewerXML::addNode(QDomElement& parent, const QString& member)
/****************************************************************************/
{
   QDomElement domElem = parent.ownerDocument().createElement(QString());
   if (!member.isNull())
      domElem.setAttribute("member", member);
   parent.appendChild(domElem);

   return domElem;
}

// Function : queryNode
// Purpose  : Queries the node with member attribute as specified out of 
//            the children of the parent node. 
// Comments :
QDomElement QGLViewerXML::queryNode(const QDomElement& parent,
				    const QString& member)
/****************************************************************************/
{
   QDomNodeList domList = parent.childNodes();
   for (int n=0; n<domList.length(); n++) {
      QDomElement child = domList.item(n).toElement();
      if (child.isNull())
	 continue;
      if (child.attribute("member").lower().compare(member.lower()) == 0)
	 return child;
   }

   return QDomElement();
}



// Function : readXML
// Purpose  : Reads CP2D
// Comments :
bool QGLViewerXML::readXML(const QDomElement& domElem, CP2D& point)
/****************************************************************************/
{
   if (domElem.tagName().compare("CP2D") != 0) 
      return false;

   QString Attrib1 = domElem.attribute("x");
   QString Attrib2 = domElem.attribute("y");
   point = CP2D(Attrib1.toFloat(), Attrib2.toFloat());

   return true;
}



// Function : writeXML
// Purpose  : Reads CP2D
// Comments :
bool QGLViewerXML::writeXML(QDomElement domElem, const CP2D& point)
/****************************************************************************/
{
   domElem.setTagName("CP2D");
   domElem.setAttribute("x", point[0]);
   domElem.setAttribute("y", point[1]);

   return true;
}



// Function : readXML
// Purpose  : Reads CP3D
// Comments :
bool QGLViewerXML::readXML(const QDomElement& domElem, CP3D& point)
/****************************************************************************/
{
   if (domElem.tagName().compare("CP3D") != 0) 
      return false;

   QString Attrib1 = domElem.attribute("x");
   QString Attrib2 = domElem.attribute("y");
   QString Attrib3 = domElem.attribute("z");
   point = CP3D(Attrib1.toFloat(), Attrib2.toFloat(), Attrib3.toFloat());

   return true;
}



// Function : writeXML
// Purpose  : Reads CP3D
// Comments :
bool QGLViewerXML::writeXML(QDomElement domElem, const CP3D& point)
/****************************************************************************/
{
   domElem.setTagName("CP3D");
   domElem.setAttribute("x", point[0]);
   domElem.setAttribute("y", point[1]);
   domElem.setAttribute("z", point[2]);

   return true;
}



// Function : readXML
// Purpose  : Reads CP4D
// Comments :
bool QGLViewerXML::readXML(const QDomElement& domElem, CP4D& point)
/****************************************************************************/
{
   if (domElem.tagName().compare("CP4D") != 0) 
      return false;

   QString Attrib1 = domElem.attribute("x");
   QString Attrib2 = domElem.attribute("y");
   QString Attrib3 = domElem.attribute("z");
   QString Attrib4 = domElem.attribute("w");
   point = CP4D(Attrib1.toFloat(), Attrib2.toFloat(), Attrib3.toFloat(),
		Attrib4.toFloat());

   return true;
}

// Function : writeXML
// Purpose  : Reads CP4D
// Comments :
bool QGLViewerXML::writeXML(QDomElement domElem, const CP4D& point)
/****************************************************************************/
{
   domElem.setTagName("CP4D");
   domElem.setAttribute("x", point[0]);
   domElem.setAttribute("y", point[1]);
   domElem.setAttribute("z", point[2]);
   domElem.setAttribute("w", point[3]);

   return true;
}




// Function : readXML
// Purpose  : Reads CV2D
// Comments :
bool QGLViewerXML::readXML(const QDomElement& domElem, CV2D& point)
/****************************************************************************/
{
   if (domElem.tagName().compare("CV2D") != 0) 
      return false;

   QString Attrib1 = domElem.attribute("x");
   QString Attrib2 = domElem.attribute("y");
   point = CV2D(Attrib1.toFloat(), Attrib2.toFloat());

   return true;
}

// Function : writeXML
// Purpose  : Reads CV2D
// Comments :
bool QGLViewerXML::writeXML(QDomElement domElem, const CV2D& point)
/****************************************************************************/
{
   domElem.setTagName("CV2D");
   domElem.setAttribute("x", point[0]);
   domElem.setAttribute("y", point[1]);

   return true;
}


// Function : readXML
// Purpose  : Reads CV3D
// Comments :
bool QGLViewerXML::readXML(const QDomElement& domElem, CV3D& point)
/****************************************************************************/
{
   if (domElem.tagName().compare("CV3D") != 0) 
      return false;

   QString Attrib1 = domElem.attribute("x");
   QString Attrib2 = domElem.attribute("y");
   QString Attrib3 = domElem.attribute("z");
   point = CV3D(Attrib1.toFloat(), Attrib2.toFloat(), Attrib3.toFloat());

   return true;
}

// Function : writeXML
// Purpose  : Reads CV3D
// Comments :
bool QGLViewerXML::writeXML(QDomElement domElem, const CV3D& point)
/****************************************************************************/
{
   domElem.setTagName("CV3D");
   domElem.setAttribute("x", point[0]);
   domElem.setAttribute("y", point[1]);
   domElem.setAttribute("z", point[2]);

   return true;
}


// Function : readXML
// Purpose  : Reads CV4D
// Comments :
bool QGLViewerXML::readXML(const QDomElement& domElem, CV4D& point)
/****************************************************************************/
{
   if (domElem.tagName().compare("CV4D") != 0) 
      return false;

   QString Attrib1 = domElem.attribute("x");
   QString Attrib2 = domElem.attribute("y");
   QString Attrib3 = domElem.attribute("z");
   QString Attrib4 = domElem.attribute("w");
   point = CV4D(Attrib1.toFloat(), Attrib2.toFloat(), Attrib3.toFloat(),
		Attrib4.toFloat());

   return true;
}

// Function : writeXML
// Purpose  : Reads CV4D
// Comments :
bool QGLViewerXML::writeXML(QDomElement domElem, const CV4D& point)
/****************************************************************************/
{
   domElem.setTagName("CV4D");
   domElem.setAttribute("x", point[0]);
   domElem.setAttribute("y", point[1]);
   domElem.setAttribute("z", point[2]);
   domElem.setAttribute("w", point[3]);

   return true;
}




// Function : readXML
// Purpose  : Reads CQuat
// Comments :
bool QGLViewerXML::readXML(const QDomElement& domElem, CQuat& point)
/****************************************************************************/
{
   if (domElem.tagName().compare("CQuat") != 0) 
      return false;

   QString Attrib1 = domElem.attribute("qw");
   QString Attrib2 = domElem.attribute("qx");
   QString Attrib3 = domElem.attribute("qy");
   QString Attrib4 = domElem.attribute("qz");
   point = CQuat(Attrib1.toFloat(), Attrib2.toFloat(), Attrib3.toFloat(),
		 Attrib4.toFloat());

   return true;
}

// Function : writeXML
// Purpose  : Reads CQuat
// Comments :
bool QGLViewerXML::writeXML(QDomElement domElem, const CQuat& point)
/****************************************************************************/
{
   domElem.setTagName("CQuat");
   domElem.setAttribute("qw", point.wv());
   domElem.setAttribute("qx", point.xv());
   domElem.setAttribute("qy", point.yv());
   domElem.setAttribute("qz", point.zv());

   return true;
}




// Function : readXML
// Purpose  : Reads CBoundingBox3D
// Comments :
bool QGLViewerXML::readXML(const QDomElement& domElem, CBoundingBox3D& bbox)
/****************************************************************************/
{
   if (domElem.tagName().compare("CBoundingBox3D") != 0) 
      return false;

   CP3D ll, ur;
   if (!readXML(queryNode(domElem, "LowerLeft"), ll)
       || !readXML(queryNode(domElem, "UpperRight"), ur))
      return false;
   bbox = CBoundingBox3D(ll, ur);

   return true;
}

// Function : writeXML
// Purpose  : Reads CBoundingBox3D
// Comments :
bool QGLViewerXML::writeXML(QDomElement domElem, const CBoundingBox3D& bbox)
/****************************************************************************/
{
   domElem.setTagName("CBoundingBox3D");
   if (!writeXML(addNode(domElem, "LowerLeft"), bbox.getLowerLeft())
       || !writeXML(addNode(domElem, "UpperRight"), bbox.getUpperRight()))
      return false;

   return true;
}



// Function : readXML
// Purpose  : Reads CMat4D
// Comments :
bool QGLViewerXML::readXML(const QDomElement& domElem, CMat4D& mat)
/****************************************************************************/
{
   if (domElem.tagName().compare("CMat4D") != 0) 
      return false;

   CV4D c1, c2, c3, c4;
   if (!readXML(queryNode(domElem, "Column1"), c1)
       || !readXML(queryNode(domElem, "Column2"), c2)
       || !readXML(queryNode(domElem, "Column3"), c3)
       || !readXML(queryNode(domElem, "Column4"), c4))
      return false;
   mat = CMat4D(c1, c2, c3, c4);

   return true;
}

// Function : writeXML
// Purpose  : Reads CMat4D
// Comments :
bool QGLViewerXML::writeXML(QDomElement domElem, const CMat4D& mat)
/****************************************************************************/
{
   domElem.setTagName("CMat4D");
   if (!writeXML(addNode(domElem, "Column1"), mat(0))
       || !writeXML(addNode(domElem, "Column2"), mat(1))
       || !writeXML(addNode(domElem, "Column3"), mat(2))
       || !writeXML(addNode(domElem, "Column4"), mat(3)))
      return false;

   return true;
}





// Function : readXML
// Purpose  : Reads CCamera
// Comments :
bool QGLViewerXML::readXML(const QDomElement& domElem, CCamera& cam)
/****************************************************************************/
{
   if (domElem.tagName().compare("CCamera") != 0) 
      return false;

   CP3D EyePos, RefPoint;
   CV3D ViewUp;
   CBoundingBox3D BBox;
   if (!readXML(queryNode(domElem, "EyePos"), EyePos)
       || !readXML(queryNode(domElem, "RefPoint"), RefPoint)
       || !readXML(queryNode(domElem, "ViewUp"), ViewUp)
       || !readXML(queryNode(domElem, "BBox"), BBox))
      return false;

   QString CamType = domElem.attribute("Type");
   QString VerAngle = domElem.attribute("VerAngle");
   QString VPHeight = domElem.attribute("VPHeight");
   QString Ratio = domElem.attribute("Ratio");
   QString NearPlane = domElem.attribute("NearPlane");
   QString FarPlane = domElem.attribute("FarPlane");
   QString TimeStep = domElem.attribute("TimeStep", "0");
   cam = CCamera(EyePos, RefPoint, ViewUp, BBox,
		 VerAngle.toDouble(), VPHeight.toInt(), Ratio.toDouble(),
		 NearPlane.toDouble(), FarPlane.toDouble(),
		 (CCamera::CameraType)(CamType.toInt()), TimeStep.toInt());

   return true;
}

// Function : writeXML
// Purpose  : Reads CCamera
// Comments :
bool QGLViewerXML::writeXML(QDomElement domElem, const CCamera& cam)
/****************************************************************************/
{
   domElem.setTagName("CCamera");

   domElem.setAttribute("Type", (int)(cam.getCameraType()));

   double rdHorAngle, rdVerAngle;
   cam.getHVAngle(rdHorAngle, rdVerAngle);
   domElem.setAttribute("VerAngle", rdVerAngle);

   int nVPHeight, nVPWidth;
   cam.getVPRes(nVPWidth, nVPHeight);
   domElem.setAttribute("VPHeight", nVPHeight);

   domElem.setAttribute("Ratio", tan(rdHorAngle)/tan(rdVerAngle));

   double rdNearPlane, rdFarPlane;
   cam.getClipPlanes(rdNearPlane, rdFarPlane);
   domElem.setAttribute("NearPlane", rdNearPlane);
   domElem.setAttribute("FarPlane", rdFarPlane);
   /* if Camera is for a 4DVolume, save TimeStep */
   if(cam.getTimeStep())
      domElem.setAttribute("TimeStep", cam.getTimeStep());

   if (!writeXML(addNode(domElem, "EyePos"), cam.getEyePos())
       || !writeXML(addNode(domElem, "RefPoint"), cam.getRefPoint())
       || !writeXML(addNode(domElem, "ViewUp"), cam.getViewUp())
       || !writeXML(addNode(domElem, "BBox"), cam.getBoundingBox()))
      return false;

   return true;
}



// Function  : readXML
// Parameters: const QDomElement& domElem, CCameraKeyPathPoint& camPoint
// Purpose   : Reads CCameraKeyPathPoint.
// Comments  : See docu for details.
bool QGLViewerXML::readXML(const QDomElement& domElem, 
			   CCameraKeyPathPoint& camPoint)
  /***************************************************************************/
{
   if (domElem.tagName().compare("CCameraKeyPathPoint") != 0) 
      return false;

   CCameraKeyPathAttributes Atts;
   CCamera Cam;

   if (!QGLViewerXML::readXML(QGLViewerXML::queryNode(domElem, "CCamera"), Cam))
      return false;
   if (!readXML(QGLViewerXML::queryNode(domElem, "Attributes"),Atts))
      Atts = CCameraKeyPathAttributes();
 
   QString qName = domElem.attribute("Name");
    
   QCString qString(qName);
   camPoint = CCameraKeyPathPoint(Cam,Atts,qString);

   return true;
}



// Function  : writeXML
// Parameters: QDomElement domElem, const CCameraKeyPathPoint& camPoint, bool fParams
// Purpose   : Writes CCameraKeyPathPoint. The attributes are only saved if 
//             fParams is true.
// Comments  : See docu for details.
bool QGLViewerXML::writeXML(QDomElement domElem, 
			    const CCameraKeyPathPoint& camPoint, 
			    bool fParams)
  /****************************************************************************/
{ 
   domElem.setTagName("CCameraKeyPathPoint");

   if (!QGLViewerXML::writeXML(QGLViewerXML::addNode(domElem, "CCamera"), camPoint.getCamera()))
      return false;
   domElem.setAttribute("Name", camPoint.getName());
      
   if (fParams) {
      if (!writeXML(QGLViewerXML::addNode(domElem, "Attributes"), camPoint.getAttributes()))
         return false;
    }
  
   return true;
}


// Function  : readXML
// Parameters: const QDomElement& domElem, CCameraKeyPathAttributes& atts
// Purpose   : Reads CCameraKeyPathAttributes.
// Comments  : See docu for details
bool QGLViewerXML::readXML(const QDomElement& domElem, 
			   CCameraKeyPathAttributes& atts)
  /****************************************************************************/
{
   if (domElem.tagName().compare("CCameraKeyPathAttributes") != 0) 
      return false;

   QString Attrib1 = domElem.attribute("Frames");
   QString Attrib2 = domElem.attribute("Tension");
   QString Attrib3 = domElem.attribute("Continuity");
   QString Attrib4 = domElem.attribute("Bias");
   atts = CCameraKeyPathAttributes(Attrib1.toInt(), Attrib2.toFloat(), 
                                   Attrib3.toFloat(),Attrib4.toFloat());

   return true;
}


// Function  : writeXML
// Parameters: DomElement domElem, const CCameraKeyPathAttributes& atts
// Purpose   : Writes CCameraKeyPathAttributes.
// Comments  : See docu for details.
bool QGLViewerXML::writeXML(QDomElement domElem, 
			    const CCameraKeyPathAttributes& atts)
  /****************************************************************************/
{
   domElem.setTagName("CCameraKeyPathAttributes");
   domElem.setAttribute("Frames", atts.getFrames());
   domElem.setAttribute("Tension", atts.getTension());
   domElem.setAttribute("Continuity", atts.getContinuity());
   domElem.setAttribute("Bias", atts.getBias());
  
   return true;
}
