/*
 * QGLExaminerViewer.h
 * $Id: QGLExaminerViewer.h,v 1.9 2001/11/20 16:23:48 guenth Exp $
 *
 * Copyright (C) 1999, 2000 Markus Janich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Class QGLExaminerViewer
//  Purpose     : Provides funcionality of a OpenGL viewer


#ifndef __QGLEXAMINERVIEWER_H_
#define __QGLEXAMINERVIEWER_H_


// Qt
///////
#include <qlineedit.h>
#include <qtimer.h>

// System
///////////


// Own
///////////
#include "QGLViewer.h"


// Defines
////////////
#define NUM_WIDGETS 13       // number of widgets to delete when
                             // switching decoration


// Forward declarations
//////////////////////////
class QLabel;
class QSlider;
class QPushButton;
class QBoxLayout;
class QTimer;
//class QLineEdit;

//class CV4D;
class QToggleButton;


#define ZOOMSCALING       100.0    // used for translating slidervalue to real (double) value.



/// Examiner viewer class
/**
  * This is a viewer which inherits functionality from the baseclass QGLViewer.
  * It implements the necessary functions which handle the mouseevents
  * and creates a layoutmanager to handle the frame with the QGLWidget in it.
  * It also creates some buttons and sliders which have new functionalities
  * and it handles input from keyboard (try pressing 'i').
  *
  * @author Markus Janich
  *
  * Here's an example for an inherited viewer (CSimpleViewer) which implements catching a key event
  * when the key 'v' was pressed. It also shoes an application which instantiates an object of
  * that viewer and implements the slots PaintGL(), ResizeGL() and InitializeGL() inside the
  * render program (CSimpleRenderer).
  * <hr>
  * Files for a derived viewer:
  * \include QSimpleViewer.h
  * <hr>
  * \include QSimpleViewer.cpp
  * <hr>
  * Files for a program doing the rendering:
  * \include CSimpleRenderer.h
  * <hr>
  * \include CSimpleRenderer.cpp
  * <hr>
  * Main program:
  * \include main.cpp
  */


class QGLExaminerViewer : public QGLViewer {
  Q_OBJECT

public:

  /**
    * Default Constructor.
    */
  QGLExaminerViewer(QWidget * parent=0, 
		    const char * name=0, const QGLWidget * shareWidget = 0, 
		    WFlags f=0, bool viewertype=true,
		    const QGLFormat &format=QGLFormat::defaultFormat())
    : QGLViewer(parent, name, shareWidget, f, viewertype, format)
    {
      init(viewertype);
      /* Nothing else to do. */
    };

  /**
    * Constructor including a camera for the home position of the scene.
    *
    * \par NOTE: The current camera will also be set to the given homecam.
    */
  QGLExaminerViewer(const CCamera &homecam,
		    QWidget * parent=0, 
		    const char * name=0, const QGLWidget * shareWidget = 0, 
		    WFlags f=0, bool viewertype=true,
		    const QGLFormat &format=QGLFormat::defaultFormat())
    : QGLViewer(homecam, parent, name, shareWidget, f, viewertype, format)
    {
      init(viewertype);
      /* Nothing else to do. */
    };

  /**
    * Constructor including the bounding box of the scene.
    */
  QGLExaminerViewer(const CBoundingBox3D &cBBox,
		    QWidget * parent=0, 
		    const char * name=0, const QGLWidget * shareWidget = 0, 
		    WFlags f=0, bool viewertype=true,
		    const QGLFormat &format=QGLFormat::defaultFormat())
    : QGLViewer(cBBox, parent, name, shareWidget, f, viewertype, format)
    {
      init(viewertype);
      /* Nothing else to do. */
    };

  /**
    * Default Destructor.
    */
  virtual ~QGLExaminerViewer() {};

  /**
    * Reimplements inherited function for interaction with zoom-slider.
    */
  virtual void setCamera(const CCamera &Camera, CameraType which=CurrentCam);

  /**
    * Reimplements inherited function.
    */
  virtual void setFullViewer(bool state);

  /**
    * Reimplements inherited function to stop the timer if running.
    */
  virtual void enableMouseEvents(bool state) {
    if (m_pqTimer->isActive())
      m_pqTimer->stop();

    QGLViewer::enableMouseEvents(state);
  };


signals:
  /////////////
  // SIGNALS //
  /////////////


  /**
    * This signal is emitted when the window of the viewer is closed.
    */
  void sigViewerClosed(void);



public slots:
  //////////////////
  // PUBLIC SLOTS //
  //////////////////

  /**
    * Sets the value of the slider which handles the rotationangle
    * around the x-axis to zero.
    */
 void sltResetXSlider();

  /**
    * Sets the value of the slider which handles the rotationangle
    * around the y-axis to zero.
    */
  void sltResetYSlider();

  /**
    * Sets the value of the slider, which handles the fovy angle,
    * to the new value. The allowed range goes from 0 to 18000.
    * @see ScalingChanged(int)
    */
  void sltSetZoomSlider(int scaling);

  /**
    * Reimplementation of the inherited slot.
    */
  virtual void sltToggleStereo();



protected slots:
  /////////////////////
  // PROTECTED SLOTS //
  /////////////////////

  /**
    * Reimplementation of the inherited slot.
    */
  virtual void sltToggleRenderMode();


  /**
    * Reimplementes inherited function for interaction
    * with zoom-slider and other GUI things.
    */
  virtual void sltGoHome();


  /**
    * This slot sets the current camera to the one
    * passed by the argument.
    */
  virtual void sltSetCamera(const CCamera &cCamera) {
    setCamera(cCamera);
  }


private slots:
  ///////////////////
  // PRIVATE SLOTS //
  ///////////////////

  /**
    * Rotate camera around the x-axis with an angle of 'iAngle' degrees.
    * This slot emits the signal sigCameraChanged() for the application which
    * owns the viewer.
    */
 void sltCamXRotate(int iAngle);

  /**
    * Rotate camera around the y-axis with an angle of 'iAngle' degrees.
    * This slot emits the signal sigCameraChanged() for the application which
    * owns the viewer.
    */
  void sltCamYRotate(int iAngle);

  /**
    * Adjusts the camera while zooming.
    */
  void sltSetZoom(int zoom_factor);

  /**
    * Sets the value of the slider, which handles the fovy angle,
    * to the new value. The allowed range goes from 0 to 180 (degrees).
    */
  void sltSetZoomSlider(double rdFovy);
  void sltSetZoomSlider(const QString &qText);
  void sltSetZoomText(int n);

  /**
    * Saves the ViewUp direction in 'vUp' and the ViewRight direction
    * in 'vRight'.
    */
  void sltSaveVectors();

  /**
    * Rotates the camera around the x- and y-axis depending on
    * the values saved in 'm_nXDiff', 'm_nYDiff', 'm_rfXAlpha' and
    * 'm_rfYAlpha'. Usually it will be connected with a Timer for
    * animated rotation.
    */
  void sltCamRotStep();

  /**
    * Toggles between full and simple viewer.
    */
  void sltToggleDeco();

  /**
    * Toggles between the tooltip on and off.
    * Initially the tooltips are off
    */
  void sltToggleToolTips();

  /**
    * Toggles spinning on and off.
    */
  void sltToggleSpinning() {
    setItemChecked(m_nSpinOnOffID, !(isItemChecked(m_nSpinOnOffID)));
  };

  /**
    * Switches between old and new mousehandling.
    */
  void sltSwitchBehave() {
    setItemChecked(m_nSwitchBehaveID, !(isItemChecked(m_nSwitchBehaveID)));
    m_fOldBehave = m_fOldBehave ? false : true;
  };

  /**
    * Toggles the render mode by using the menu.
    */
  void sltMenuToggleRenderMode();

  /**
    * Manages the timeout for animation.
    */
  void sltTimeOut() { m_fAnimationOn = false; };

  /** Shows an dialog to enter a new refresh rate. */
  void sltSetRefreshRate();


protected:
  ///////////////////////
  // PROTECTED METHODS //
  ///////////////////////
  
  /**
    * Implements a handler for keyPressEvents.
    * You can overwrite this method in your derived class. Original implementation
    * manage the following keys:
    * <br>i            :  opens a box with this key definitions
    * <br>CURSOR_UP    :  rotate "back" around x-axis
    * <br>CURSOR_DOWN  :  rotate "to the front" around x-axis
    * <br>CURSOR_LEFT  :  rotate left around y-axis
    * <br>CURSOR_RIGHT :  rotate right around y-axis
    * <br>a            :  view all
    * <br>s            :  set homecamera to current camera
    * <br>h            :  set current camera to homecamera
    * <br>p            :  toggle projectionmode
    * <br>+            :  zoom in
    * <br>-            :  zoom out
    * <br>ESC or q     :  exit application
    * <br>CTRL+CURSOR_LEFT  :  translate to the left
    * <br>CTRL+CURSOR_RIGHT :  translate to the right
    * <br>CTRL+CURSOR_UP    :  translate up
    * <br>CTRL+CURSOR_DOWN  :  translate down
    * <br>SHIFT+CURSOR_UP   :  move forward
    * <br>SHIFT+CURSOR_DOWN :  move back
    */
  virtual void keyPressEvent(QKeyEvent *event);

  /**
    * Implements a handler for CloseEvents.
    * The default emits a sigViewerClosed()-signal and then
    * calls the closeEvent()-function of the QWidget-class.
    */
  virtual void closeEvent(QCloseEvent *event);

  /**
    * Implements the method from the derived class QGLViever.
    */
  virtual void ManageMouseMove(QMouseEvent *event);

  /**
    * Implements the method from the derived class QGLViever.
    */
  virtual void ManageMousePress(QMouseEvent *event);

  /**
    * Implements the method from the derived class QGLViever.
    */
  virtual void ManageMouseRelease(QMouseEvent *event);

  /** Implements inherited method. */
  bool eventFilter(QObject *pqObject, QEvent *pqEvent);

  /**
    * Manage the layout of a fullviewer after it is constructed or 'setFullViewer(true)' is called.
    */
  virtual void makeFullViewer();

  /**
    * Manage the layout of a non-fullviewer after it is constructed or 'setFullViewer(false)' is called.
    */
  virtual void makeSimpleViewer();


private:
  /////////////////////
  // PRIVATE METHODS //
  /////////////////////

  /**
    * Initializes all stuff that is unique to all constructors.
    */
  void init(bool viewertype);


  /////////////////////
  // PRIVATE MEMBERS //
  /////////////////////

  int m_nMousePosX, m_nMousePosY;      // last mouse position
  int m_nXDiff, m_nYDiff;              // saves the difference of the mouse 
                                       // position between two mouseevents

  float m_rfXAlpha, m_rfYAlpha;        // used for rotating the scene around the axis
                                       // of the view direction relative to the position
                                       // of the mouse pointer inside the drawarea.

  int m_nQuadrant;                     // saves the quadrant where the last mouseaction happened.

  CV3D m_cUp, m_cRight, m_cDir;        // saves the ViewUp, ViewRight and
                                       // ViewDirection vector by 'sltSaveVectors()'

  int m_nLastRotAngle;                 // last angle (in degrees) of rotation using the sliders

  bool m_fLeftButtonPressed;           // saves left button state
  bool m_fMiddleButtonPressed;         // saves middle button state

  QBoxLayout *m_pqTopLayout;           // pointer to the toplayout

  QSlider *m_pqXRot,                   // pointer to slider of x-rotation
          *m_pqYRot,                   // pointer to slider of y-rotation
          *m_pqZoom;                   // pointer to slider of zooming

  QLineEdit *m_pqZoomText;

  QTimer *m_pqTimer,                   // timer for rotation animation
         *m_pqTimeOut;                 // timer to manage the timeout
  QWidget *m_apqWidgets[NUM_WIDGETS];  // saves the pointers of all widgets (needed for switching
                                       // between full and simple viewer.
  int m_nSelectionID;                  // saves the ID of the menuentry 'Selection on'
  int m_nDecoID;                       // saves the ID of the menuentry 'Decoration on'
  int m_nTTipsID;                      // saves the ID of the menuentry 'Tooltips on'
  int m_nSpinOnOffID;		       // saves the ID of the menuentry 'Spinning On/Off'
  int m_nSwitchBehaveID;               // saves the ID of the menuentry 'Behave like old ExaminerViewer'
  int m_nStereoID;                     // saves the ID of the menuentry 'Stereo'
  bool m_fToolTips;                    // saves the state of the tooltips
  bool m_fAnimationOn;                 // triggers the timeout for starting the animation
  bool m_fOldBehave;                   // saves the state how the mousemoves behave like, old or new style
  bool m_fShiftPressed;                // saves the state of the shift key while rotating the camera 
};

#endif // __QGLEXAMINERVIEWER_H_
