/*
 * CV4D.h
 * $Id: CV4D.h,v 1.4 2003/06/24 14:50:02 anxo Exp $
 *
 * Copyright (C) 1999, 2000 Markus Janich, Michael Meissner, Rainer Jaeger
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */



#ifndef __CV4D_H_
#define __CV4D_H_


// System
///////////
#include <math.h>
#ifdef _MSC_VER
#if _MSC_VER >= 1300
#include <iostream>
#endif
#else
#include <iostream.h>
#endif


// Own
///////////


// Forward declarations
/////////////////////////
class CV3D;

using namespace std;

/** This class implements a vector class for homogenous coordinates
  *including functionality to vector calculations.
  *
  * @author Markus Janich, Michael Meissner, Rainer Jaeger
  */
class CV4D {
public:
  static double epsilon;

  /** Default constructor.
    * The default value of the instantiated vector will be (0.0,0.0,0.0,0.0).
    */
  CV4D() { m_ard[0] = 0.0;
           m_ard[1] = 0.0;
           m_ard[2] = 0.0;
           m_ard[3] = 0.0; };


  /** Construct new vector.
    * The value of the vector will be (rdX, rdY, rdZ, 1). */
  CV4D(double rdX, double rdY, double rdZ) { m_ard[0] = rdX; 
                                             m_ard[1] = rdY; 
					     m_ard[2] = rdZ;
					     m_ard[3] = 1.0; };

  /** Construct new vector.
    * The value of the vector will be (rdX, rdY, rdZ, rdW). */
  CV4D(double rdX, double rdY, double rdZ, double rdW) { m_ard[0] = rdX; 
                                                         m_ard[1] = rdY; 
					                 m_ard[2] = rdZ;
					                 m_ard[3] = rdW; };

  /** Copy constructor.
    * Initializes the new vector with the vector passed in 'v'. */
  CV4D(const CV4D& Vector) { m_ard[0] = Vector.m_ard[0];
                             m_ard[1] = Vector.m_ard[1];
                             m_ard[2] = Vector.m_ard[2];
                             m_ard[3] = Vector.m_ard[3]; };



  //////////////////////////
  // OVERLOADED OPERATORS //
  //////////////////////////

  /** Cast operator to convert CV4D vectors to CV3D vectors.
    * Each component is devided by the fourth component. */
  operator CV3D() const;

  /** Assign one vector to another.*/
  const CV4D& operator=(const CV4D&);

  /** Compares to vectors.
    * The result will be 'true' if the two vector are indentically
    * in each component. Otherwise 'false' will be returned. */ 
  bool operator==(const CV4D&) const;

  /** Compares to vectors.
    * Same as above. Only the result is negated.
    * @see operator==() */
  bool operator!=(const CV4D&) const;

  /** Adds another vector to this vector. */
  CV4D& operator+=(const CV4D&);

  /** Subtracts another vector from this vector. */
  CV4D& operator-=(const CV4D&);

  /** Adds two vectors. The sum will be returned. */
  CV4D operator+(const CV4D&) const;

  /** Subtracts two vectors. The difference will be returned. */
  CV4D operator-(const CV4D&) const;

  /** Negates the vector. */
  CV4D operator-() const;

  /** Scalar multiplication of two vectors. The result will be returned. */
  double operator*(const CV4D&) const;

  /** Multiplication of a vector with a scalar. */
  CV4D operator*(double) const;

  /** Division of a vector with a scalar. */
  CV4D operator/(double);

  /** crossproduct. (I know that this is only defined in three dimensions).
    * For the calculation only the coefficient for the x-, y- and 
    * z-direction will be used. The fourth coefficient (known as 'w') will
    * be set to 1. */
  CV4D operator|(const CV4D&) const;

  /** Returns the i-th component of the vector.
    * The index goes from 0 to 3, so 0 stands for the
    * x-coordinate, 1 for the y-coordinate and so on. */
  double& operator[](int i) { return m_ard[i]; };

  /** Same as above but for constant vectors. */
  double operator[](int i) const { return m_ard[i]; };

  /** Multiplication of a scalar with a vector. */
  friend CV4D operator*(double, const CV4D&);



  /////////////
  // METHODS //
  /////////////

  /** Returns the x-coordinate of the vector. */
  double getX() const   { return m_ard[0]; };
  
  /** Returns the y-coordinate of the vector. */
  double getY() const   { return m_ard[1]; };
  
  /** Returns the z-coordinate of the vector. */
  double getZ() const   { return m_ard[2]; };
  
  /** Returns the w-coordinate of the vector. */
  double getW() const   { return m_ard[3]; };

  /** Sets the x-coordinate of the vector to 'rdX'. */
  void setX(double rdX) { m_ard[0] = rdX; };

  /** Sets the y-coordinate of the vector to 'rdY'. */
  void setY(double rdY) { m_ard[1] = rdY; };

  /** Sets the z-coordinate of the vector to 'rdZ'. */
  void setZ(double rdZ) { m_ard[2] = rdZ; };

  /** Sets the w-coordinate of the vector to 'rdW'. */
  void setW(double rdW) { m_ard[3] = rdW; };

  /** Set the values of the vector.
    * The value of the vector will be (rdX, rdY, rdZ, rdW). */
  void setCoord(double rdX, double rdY, double rdZ, double rdW) { m_ard[0] = rdX; 
	                                                          m_ard[1] = rdY;
	                                                          m_ard[2] = rdZ;
	                                                          m_ard[3] = rdW; 
                                                                  return; };

  /** Returns the euclidian norm of the vector. */
  double getNorm() const;

  /** Normalizes the vector. */
  void normalize();

  /** Returns the normalized vector. */
  CV4D getNormalized() const;

  /** Prints a vector to the standard output. */ 
  void print() const;
  
  /** Same as above. But more useful for streams. */
  friend ostream& operator<<(ostream&, const CV4D&); 

  /** Reads a vector from the given stream. */
  friend istream& operator>>(istream&, CV4D&); 



protected:
  double m_ard[4];

};



// Function   : operator=
// Parameters : const CV4D& v
// Purpose    : assign another vector to this vector
// Comments   : 
inline const CV4D& CV4D::operator=(const CV4D& v) 
/*******************************************************************/
{
  m_ard[0] = v.m_ard[0];
  m_ard[1] = v.m_ard[1];
  m_ard[2] = v.m_ard[2];
  m_ard[3] = v.m_ard[3];

  return *this;
}

#endif // __CV4D_H_
