/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef POWERSAVE_CONFIG_H
#define POWERSAVE_CONFIG_H

#include <map>
#include <string>
#include "powerlib.h"
#include "event.h"
#include "device.h"

#ifdef CREATE_DEBUG_OUTPUT
#include <sstream>
#include <iostream>
#endif

#define MAX_SCHEMES 50
#define MAX_THERMAL_ZONES 10
#define ACPI_THERMAL_MAX_ACTIVE 10

enum CPUFREQ_MODE { _DYNAMIC, _PERFORMANCE, _POWERSAVE };

/** @brief the cpufreq control mode
 *
 * start with 1 to be used as bitmask
 */
enum CPUFREQ_CONTROL_MODE { CPUFREQ_USERSPACE = 1, CPUFREQ_KERNEL };

enum THERMAL_MANAGEMENT_DELIGATION { OFF, KERNEL, KERNEL_PASSIVE };

class Event;
class GeneralConfig;

using namespace std;

/** @brief Powersave configuration class */
class PS_Config {
public:
	/** @brief constructor */
	PS_Config();

	/** @brief destructor */
	virtual ~ PS_Config();
	
	virtual Event & getEvent(const string &event_name);

	virtual int readConfigFile(const std::string & file = "");

	virtual void assignConfigEntries();

	/** map that contains all EVENT objects. All EVENT objects are
	 * initialised in PM_Interface constructor. */
	typedef map < string, Event >::iterator EventIter;

	/** holds all thermal info coming from powerlib.h. It is
	 * initialised with kernel/BIOS values that could be overridden by
	 * config files */
	ThermalDev thermal_zones[MAX_THERMAL_ZONES];

	/** @brief a scheme name
	 *
	 * this will only be assigned with valid value for schemes
	 */
	string SCHEME_NAME;

	/** @brief the scheme description */
	string SCHEME_DESCRIPTION;

	/* CONFIGURATION VARIABLES FROM CONFIG FILES ********* */
	
	/** @brief cpufreq should be controlled by userspace or kernel */
	CPUFREQ_CONTROL_MODE CPUFREQ_CONTROL;
	
	/** @brief cpu load percentage at which we increase the cpu
	 * speed */
	int CPU_HIGH_LIMIT;

	/** what't that. I can't find word deligation */
	THERMAL_MANAGEMENT_DELIGATION ENABLE_THERMAL_MANAGEMENT;

	/** @brief how long our event loop sleeps (in miliseconds) */
	unsigned long POLL_INTERVAL;

	/** @brief timeout (seconds) after which an additional idle event
	 * is thrown */
	unsigned long CPU_IDLE_TIMEOUT;

	/** @brief cpu limit (in percent) under which after
	 * CPU_IDLE_TIMEOUT the idle event is thrown */
	int CPU_IDLE_LIMIT;

	/** @brief how many CPUs should be only at max
	 *
	 * 0 means all
	 */
	int MAX_CPUS_ONLINE;

	/** @brief should niced processes count for cpufreq calculation,
	 * too? */
	int CONSIDER_NICE;

	/** @brief if suspend to disk is disabled */
	int USER_SUSPEND2DISK_DISABLED;

	/** @brief if suspend to ram is disabled */
	int USER_SUSPEND2RAM_DISABLED;

	/** @brief if standby is disabled */
	int USER_STANDBY_DISABLED;

	/** @brief absolute brightness level (depends on driver) or:
		   -1 ignore -2 low -3 medium -4 high */
	int DISPLAY_BRIGHTNESS;

	/** @brief time an button/power acpi event is delayed until going on */ 
	int POWER_BUTTON_DELAY;

	/* @brief throw a battery warning event if battery reaches this
	 * percent value */
	int BAT_WARN_LIMIT;

	/* @brief throw a battery low event if battery reaches this
	 * percent value */
	int BAT_LOW_LIMIT;

	/* @brief throw a battery critical event if battery reaches this
	 * percent value */
	int BAT_CRIT_LIMIT;

	/** @brief percent on which the cpu frequency is set immediately
	 * to maximum
	 *
	 * Set the CPU load difference limit for which the highest
	 * available frequency is set immediately (e.g. if set to 60, it
	 * switches to full speed if the CPU load jumps from 5% to 66% or
	 * higher instead of ramping up through all available
	 * frequencies). Disable the direct triggering of the highest
	 * frequency by setting a value of 0.
	 */
	int JUMP_CPU_FREQ_MAX_LIMIT;

	/** @brief whether cpufrequency scaling is enabled */
	bool CPUFREQ_ENABLED;

	/** @brief if dpm is enabled */
	bool DPM_ENABLED;

	/** 
	 * The hysteresis for calculating the CPU load limit to switch to
	 * the next lover CPU frequency. Increasing this value lowers the
	 * load percentage at which the deamon switches down one step.
	 * This might be useful if you notice that the frequency
	 * continuously oscilates at a almost constant CPU usage
	 * (e.g. when watching a movie).  5 is default, you should not set
	 * it higher than 40.  Zero is not allowed.
	 */
	int CPUFREQ_HYSTERESIS;

	/** @brief filename of the battery scheme without the leading scheme_*
	 *
	 * Do not mix up with the SCHEME_NAME variable inside the scheme
	 */
	string BATTERY_SCHEME;

	/** @brief filename of the ac scheme without the leading scheme_*
	 *
	 * Do not mix up with the SCHEME_NAME variable inside the scheme
	 */
	string AC_SCHEME;

	/** @brief if throttling is allowed */
	bool ALLOW_THROTTLING;

	/** @brief the maximum percent to which get throttled */
	int MAX_CPU_THROTTLING;

	/** @brief if we should always throttle */
	bool ALWAYS_THROTTLE;

	/** @brief one of dynamic, powersave or performance */
	CPUFREQ_MODE CPUFREQUENCY;

	/** @brief the cooling policy, either active or passive */
	_COOLING_POLICY COOLING_POLICY;

	/** Actions for events as specified in events config file
	   or maybe specialised in a scheme configuration */
	 map < string, Event > eventMap;

	 /** list containing device classes @ref DPM_DEVICE_CLASS which
	  * should get suspended */
	 std::list < DPM_DEVICE_CLASS > dpmClasses;

	/********** end of configuration variables **********/

private:
	/** @brief predefined message for debugging string to int
	 * conversion */
	void strtointErr(char *str, int line, const char *file);

	/** @brief checks a value from a configuration file
	 *
	 * @param what string containing an interger value from configuration file
	 * @param min the minimum value allowed
	 * @param max the maximum value allowed
	 *
	 * @return integer value contained in what, negative value on error
	 * @retval -1 string what is empty
	 * @retval -2 if integer contained in what outside of min or max
	 */
	int checkValue(const std::string &what, int min, int max);

	/** @brief checks a value from a configuration file
	 *
	 * @param what string containing an interger value from configuration file
	 * @param min the minimum value allowed
	 *
	 * @return integer value contained in what, negative value on error
	 * @retval -1 string what is empty
	 * @retval -2 if integer contained in what outside of min or max
	 */
	int checkValue(const std::string &what, int min);

	/** @brief checks a value from a configuration file
	 *
	 * @param def default value returned on error/empty value
	 * @param what string containing an interger value from configuration file
	 * @param min the minimum value allowed
	 * @param max the maximum value allowed
	 *
	 * @return integer value contained in what, default value or negative
	 *         value on error
	 * @retval -1 string what is empty
	 * @retval -2 if integer contained in what outside of min or max
	 */
	int checkValue(int def, const std::string &what, int min, int max);

	/** @brief checks a value from a configuration file
	 *
	 * @param def default value returned on error/empty value
	 * @param what string containing an interger value from configuration file
	 * @param min the minimum value allowed
	 *
	 * @return integer value contained in what, default value or negative
	 *         value on error
	 * @retval -1 string what is empty
	 * @retval -2 if integer contained in what outside of min or max
	 */
	int checkValue(int def, const std::string &what, int min);

	/** @brief checks a value from a configuration file for yes/no
	 *
	 * @param what string containing a value
	 *
	 * @return true if value is 'yes' or empty, false otherwise
	 * @retval -1 if empty value
	 * @retval 1 if value is 'yes'
	 * @retval 0 if value is not yes
	 */
	int checkYes(const std::string &what);

	/** @brief checks a value from a configuration file for yes/no
	 *
	 * @param def default value to return if value is empty
	 * @param what string containing a value
	 *
	 * @return true if value is 'yes', def if empty
	 * @retval def if empty value
	 * @retval 1 if value is 'yes'
	 * @retval 0 if value is not yes
	 */
	int checkYes(int def, const std::string &what);

	/** @brief split a line into single strings
	 *
	 * @param line the line to split delimited by whitespace(s)
	 * @param strings list which gets filled with single values
	 */
	void splitLine(std::string line, std::list< std::string > &strings);

	/** @brief an array that can be accessed by string name */
	typedef std::map < std::string, std::string > StringMap;

	/** @brief holds all configuration variables */
	StringMap data;
};


/** @brief Scheme configuration class */
class SchemeConfig : public PS_Config {
public:
	/** @brief initializes a scheme configuration with default
	 * values */ 
	SchemeConfig(const string &file_name, const GeneralConfig &);

	/** @brief destructor */
	virtual ~SchemeConfig();

	/** @brief reads a scheme configuration file */
	virtual int readConfigFile(const string &file = "");
	
	/** @brief the filename of the scheme given to constructor */
	string file_name;

	/** @brief this is a list... */
	SchemeConfig *next;
	
private:
	/** @brief full path to scheme including the file_name */
	string file_path;
	
#ifdef CREATE_DEBUG_OUTPUT
	friend ostream & operator<<(ostream &, SchemeConfig &);
#endif
};


/** @brief class representing a general configuration (no scheme
 * config) */
class GeneralConfig : public PS_Config {
public:
	/** @brief constructor */
	 GeneralConfig();

	/** @brief destructor */
	virtual ~ GeneralConfig();

	/** @brief read the config files
	 *
	 * @return
	 *     - -1 if no schemes available, have to quit then or we
	 * may access NULL pointer in powersave.cpp:
	 * config->current_scheme, makes no sense to assign a default
	 * scheme
	 */
	int readConfFiles();

	/** @brief get a scheme configuration by scheme name
	 *
	 * @param scheme_name the name of the scheme
	 *
	 * @return SchemeConfig object or NULL
	 */
	SchemeConfig *getScheme(const string &scheme_name);

	/** @brief get a scheme by number
	 *
	 * use for iteration to get all schemes... 
	 *
	 * @param no number of scheme when added
	 *
	 * @return pointer to a SchemeConfig
	 * @retval pointer to requested SchemeConfig
	 * @retval NULL if scheme at position no does not exist
	 */
	SchemeConfig *getSchemeByNumber(int no);

	/** @brief sets the active scheme
	 *
	 * @param scheme_name pass the name of the scheme as it is defined
	 *        in POWERSAVE_SCHEME_NAME
	 *
	 * @return integer representing result of set active scheme
	 * @retval -1 if the scheme does not exist
	 * @retval 0 if scheme is already active
         * @retval 1 if scheme is successfully activated
	 */
	int setActiveScheme(const string &scheme_name);

	/** @brief get the name of the battery scheme
	 *
	 * @return the name of the scheme
	 * @retval "" if no scheme is associated (this would be a bug)
	 */
	string getBatteryScheme();

	/** @brief get the name of the AC scheme
	 *
	 * @return the name of the scheme
	 * @retval "" if no scheme is associated (this would be a bug)
	 */
	string getACScheme();

	/** @brief the current active scheme config */
	SchemeConfig *current_scheme;

	/** @brief the default AC scheme config */
	SchemeConfig *AC_scheme;

	/** @brief the default battery scheme config */
	SchemeConfig *Battery_scheme;

	/** @brief directory where to find config files */
	string config_dir;

	/** @brief directory where to find "internal" scripts */
	string script_dir;

	/** @brief the directory where scheme files are found */
	string scheme_dir;

	/** @brief directory where to find scripts for e.g. events */
	string pub_script_dir;

	/** @brief whether to disable cpu frequency scaling */
	bool disable_CPU_freq;

#ifdef CREATE_DEBUG_OUTPUT
	friend ostream & operator<<(ostream & os, const GeneralConfig & gc);
#endif

private:
	/** @brief six different general config files are used */
	static const string config_files[6];

	/** @brief beginning of scheme list */
	SchemeConfig *scheme_configs;

	/** @brief initializes all schemes reading them from config files
	 *
	 * @return number of initialised schemes or < 0 on error
	 */
	int initSchemes();

	/** @brief parse all scheme config files
	 *
	 * @return true on success, false otherwise
	 */
	bool parseSchemes();
};

#endif // POWERSAVE_CONFIG_H
