/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef PORT_H
#define PORT_H

#include <cstdlib>
#include <string>
#include <ladspa.h>
#include <types.h>
#include "Array.h"
#include "OmObject.h"
#include "PortInfo.h"

using std::string;

namespace Om {

class Node;


/** An audio or control data port.
 *
 * This is basically just a buffer and a bunch of flags and helper methods.
 * All the interesting functionality of ports is in InputPort.
 *
 * \ingroup engine
 */
class Port : public OmObject
{
public:
	virtual ~Port();

	Port* as_port() { return static_cast<Port*>(this); }

	sample get_value(uint voice, size_t offset);
	void   set_value(uint voice, size_t offset, sample val);
	void   set_value(size_t offset, sample val);
	
	sample* const buffer(uint voice) const { return m_buffers.at(voice); }
	
	PortInfo* const port_info() const       { return m_port_info; }
	void            port_info(PortInfo* pi) { m_port_info = pi; }

	/** Called once per process cycle */
	virtual void prepare_buffers();
	
	Node* const parent_node() const { return m_parent->as_node(); }
	
	// Eliminate these entirely?  Not sure the existance of PortInfo
	// is needed at all though..
	bool is_control() const   { return m_port_info->is_control(); }
	bool is_audio() const     { return m_port_info->is_audio(); }
	bool is_input() const     { return m_port_info->is_input(); }
	bool is_output() const    { return m_port_info->is_output(); }

	uint   num() const         { return m_index; }
	uint   poly() const        { return m_poly; }
	size_t buffer_size() const { return m_buffer_size; }

protected:
	enum BufferState { OK, HALF_SET_BLOCK_1, HALF_SET_BLOCK_2 };

	Port(Node* const node, const string& name, uint index, uint poly, PortInfo* port_info, size_t buffer_size);
	
	// Prevent copies
	Port(const Port& copy)            { exit(EXIT_FAILURE); }
	Port& operator=(const Port& copy) { exit(EXIT_FAILURE); }

	void allocate_buffers();

	uint           m_poly;
	uint           m_index;
	Array<sample*> m_local_buffers; // Port's local buffers (always allocated, possibly not used)
	Array<sample*> m_buffers;       // Buffer that will be returned
	size_t         m_buffer_size;
	PortInfo*      m_port_info;
	bool           m_is_local_buffer;
	BufferState    m_buffer_state;
};


} // namespace Om

#endif // PORT_H
