/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "support.h"

#ifdef ENABLE_WBXML
#include <libsyncml/parser/sml_wbxml_internals.h>

#include <libsyncml/sml_command_internals.h>
#include <libsyncml/sml_elements_internals.h>

static SmlAssembler *start_assembler(SmlSession *session, unsigned int limit)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;

	SmlAssembler *assm = smlAssemblerNew(SML_MIMETYPE_WBXML, limit, &error);
	fail_unless(assm != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	if (!smlAssemblerStart(assm, session, &error)) {
		fail_unless(error != NULL, NULL);
		smlAssemblerFree(assm);
		return NULL;
	}
	return assm;
}

START_TEST (check_assembler_message_empty)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_message_only_header)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_message_status)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerReserveStatus(assm, 1, 1, 1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlStatus *status = smlStatusNew(SML_NO_ERROR, 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_message_cmd)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = smlCommandNewAlert(SML_ALERT_TWO_WAY, loc, loc, "next", "last", NULL, &error);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd->cmdID = 1;
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	char *buffer = NULL;
	fail_unless(smlWbxmlConvertFrom(NULL, data, size, &buffer, &error), NULL);
	fail_unless(buffer != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_message_subcmd)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd1 = smlCommandNewSync(loc, loc, 0, &error);
	fail_unless(cmd1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd1->cmdID = 1;
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	fail_unless(cmd2 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd2->cmdID = 2;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_message_several_subcmd)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd1 = smlCommandNewSync(loc, loc, 0, &error);
	fail_unless(cmd1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd1->cmdID = 1;
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 2;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid2", "data2", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 3;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd1 = smlCommandNewSync(loc, loc, 0, &error);
	fail_unless(cmd1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd1->cmdID = 4;
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 5;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid2", "data2", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 6;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_message_flush_open)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd1 = smlCommandNewSync(loc, loc, 0, &error);
	fail_unless(cmd1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd1->cmdID = 1;
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 2;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, FALSE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid2", "data2", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 3;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_out_of_order)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd1 = smlCommandNewSync(loc, loc, 0, &error);
	fail_unless(cmd1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd1->cmdID = 1;
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 2;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd3 = smlCommandNewSync(loc, loc, 0, &error);
	fail_unless(cmd3 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd3->cmdID = 4;
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd3, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid", "data", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 5;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd3, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd3, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid2", "data2", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 3;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd1, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid2", "data2", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 6;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd3, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd3, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	cmd2 = smlCommandNewChange(SML_CHANGE_ADD, "uid2", "data2", 5, SML_ELEMENT_TEXT_VCARD, &error);
	cmd2->cmdID = 7;
	
	fail_unless(smlAssemblerStartCommand(assm, cmd3, cmd2, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, cmd3, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_limit1)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 1);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(!smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_limit2)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 50);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(!smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_limit3)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 100);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerReserveStatus(assm, 1, 1, 1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlStatus *status = smlStatusNew(SML_NO_ERROR, 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_MISMATCH, NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(!smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_limit4)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 380);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerReserveStatus(assm, 1, 1, 1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlStatus *status = smlStatusNew(SML_NO_ERROR, 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_limit5)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 150);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerReserveStatus(assm, 1, 1, 1, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlStatus *status = smlStatusNew(SML_NO_ERROR, 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = smlCommandNewAlert(SML_ALERT_TWO_WAY, loc, loc, "next", "last", NULL, &error);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd->cmdID = 1;
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd, &error) == SML_ASSEMBLER_RESULT_MISMATCH, NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, FALSE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

START_TEST (check_assembler_complex1)
{
	SmlError *error = NULL;
	
	SmlLocation *loc = smlLocationNew("/test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_SERVER, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, "1", 0, &error);

	SmlAssembler *assm = start_assembler(session, 0);
	fail_unless(assm != NULL, NULL);
	
	fail_unless(smlAssemblerReserveStatus(assm, 1, 1, 1, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlAssemblerReserveStatus(assm, 2, 1, 2, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlAssemblerReserveStatus(assm, 3, 1, 3, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlAssemblerReserveStatus(assm, 4, 1, 4, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddHeader(assm, session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlStatus *status = smlStatusNew(SML_NO_ERROR, 1, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	status = smlStatusNew(SML_NO_ERROR, 2, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	status = smlStatusNew(SML_NO_ERROR, 3, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	status = smlStatusNew(SML_NO_ERROR, 4, 1, loc, loc, SML_COMMAND_TYPE_SYNC, &error);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerAddStatus(assm, status, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = smlCommandNewAlert(SML_ALERT_TWO_WAY, loc, loc, "next", "last", NULL, &error);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	cmd->cmdID = 5;
	
	fail_unless(smlAssemblerStartCommand(assm, NULL, cmd, &error) == SML_ASSEMBLER_RESULT_OK, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlAssemblerEndCommand(assm, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	char *data = NULL;
	unsigned int size = 0;
	fail_unless(smlAssemblerRun(assm, &data, &size, NULL, FALSE, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(data != NULL, NULL);
	fail_unless(size != 0, NULL);
	
	fail_unless(smlAssemblerFlush(assm) == 1, NULL);
	fail_unless(error == NULL, NULL);
	
	smlAssemblerFree(assm);
	smlSessionUnref(session);
}
END_TEST

Suite *assembler_suite(void)
{
	Suite *s = suite_create("Assembler");
	//Suite *s2 = suite_create("Assembler");
	
	create_case(s, "check_assembler_message_empty", check_assembler_message_empty);
	create_case(s, "check_assembler_message_only_header", check_assembler_message_only_header);
	create_case(s, "check_assembler_message_status", check_assembler_message_status);
	create_case(s, "check_assembler_message_cmd", check_assembler_message_cmd);
	create_case(s, "check_assembler_message_subcmd", check_assembler_message_subcmd);
	create_case(s, "check_assembler_message_several_subcmd", check_assembler_message_several_subcmd);
	create_case(s, "check_assembler_message_flush_open", check_assembler_message_flush_open);
	create_case(s, "check_assembler_out_of_order", check_assembler_out_of_order);
	create_case(s, "check_assembler_limit1", check_assembler_limit1);
	create_case(s, "check_assembler_limit2", check_assembler_limit2);
	create_case(s, "check_assembler_limit3", check_assembler_limit3);
	create_case(s, "check_assembler_limit4", check_assembler_limit4);
	create_case(s, "check_assembler_limit5", check_assembler_limit5);
	create_case(s, "check_assembler_complex1", check_assembler_complex1);
	
	return s;
}

int main(void)
{
	int nf;

	Suite *s = assembler_suite();
	
	SRunner *sr;
	sr = srunner_create(s);
	srunner_run_all(sr, CK_VERBOSE);
	nf = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (nf == 0) ? 0 : 1;
}
#else


int main(void)
{
	return 1;
}

#endif
