/* -*- mode: fundamental -*- */

%option prefix="rg"

%{

#include "rg.h"

#include "files.h"
#include "readgame-parser.h"
#include "../util/debug.h"
#include "../util/error.h"
#include "../util/Object.h"
#include "../util/String.h"
#include <assert.h>
#include <stdio.h>
#include <string.h>

extern int rg_scanFile(const char*);
extern void rg_registerMacro(const char* name, const char* text);

//--------------------------------------------------------------------------------------------------------------------------------

// some classes to control include files and the like

class RgBufferState : public Object
{
private:
 RgBufferState* pLastBuffer; // holds the buffer state to return to when we are done

protected:
 YY_BUFFER_STATE yyBufferState;

public:
 RgBufferState(RgBufferState* _lastBuffer)
  {
   pLastBuffer = _lastBuffer;
   yyBufferState = NULL;
  }
 virtual ~RgBufferState()
  { if(yyBufferState) rg_delete_buffer(yyBufferState); }
 void switchTo()
  { rg_switch_to_buffer(yyBufferState); }
 RgBufferState* lastBuffer() { return pLastBuffer; }
};

class RgFileBuffer : public RgBufferState
{
private:
 FILE* fp;
public:
 RgFileBuffer(RgBufferState* _lastBuffer)
  : RgBufferState(_lastBuffer)
  { fp = NULL; }
 virtual ~RgFileBuffer()
  { if(fp) fclose(fp); }
 bool open(const char* filename)
  {
   assert(fp == NULL); assert(filename);
   if((fp = fopen(filename, "r")) == NULL) { setError("%s: %s", filename, errnoStr()); return false; }
   yyBufferState = rg_create_buffer(fp, YY_BUF_SIZE);
   return true;
  }
};

// somewhere to keep a copy of the current BufferState
static RgBufferState* currentBuffer = NULL;

//--------------------------------------------------------------------------------------------------------------------------------

// some variables we need whilst scanning

String includeFilename;

//--------------------------------------------------------------------------------------------------------------------------------

%}

%x incl

%%

[[:space:]]		/* nothing (whitespace) */
[#;%!|][^\n]*\n		/* ignore comment */

"hatgame"		{ return k_hatgame; }
"level"			{ return k_level; }
"cutscene"		{ return k_cutscene; }

\<			{
			 includeFilename = "";
			 BEGIN(incl);
			}
<incl>\\\\		{ includeFilename += "\\"; }
<incl>\\\>		{ includeFilename += ">"; }
<incl>[^\>]+		{ includeFilename += rgtext; }
<incl>\>		{
			 VPRINTF("<rg> opening include file \"%s\"\n", (const char*) includeFilename);
			 RgFileBuffer* newBuffer = new RgFileBuffer(currentBuffer);
			 if(newBuffer->open(libFileName(includeFilename)))
			  {
			   currentBuffer = newBuffer;
			   currentBuffer->switchTo();
			  }
			 else
			  {
			   nonFatal("Cannot find include file \"%s\"\n", (const char*) includeFilename);
			   delete newBuffer;
			  }
			 BEGIN(INITIAL);
			}

<<EOF>>			{
			 RgBufferState* nextBuffer = currentBuffer->lastBuffer();
			 if(nextBuffer)
			  {
			   delete currentBuffer;
			   currentBuffer = nextBuffer;
			   currentBuffer->switchTo();
			  }
			 else yyterminate();
			}

[a-zA-Z]+		{ rglval.string = strdup(rgtext); return d_symbol; }

[[:digit:]]+		{ rglval.integer = strtol(rgtext, NULL, 0); return d_number; }
\"[^\"]*\"      	{ rglval.string = strndup(rgtext+1, rgleng-2); return d_string; }

.			{ return rgtext[0]; }

%%

//--------------------------------------------------------------------------------------------------------------------------------

bool rgBeginLex(const char* filename)
{
 VPRINTF("<rg> rgBeginLex(\"%s\")\n", filename);
 if(currentBuffer) fatal("rgBeginLex called twice!\n");
 RgFileBuffer* newBuffer = new RgFileBuffer(NULL);
 if(!newBuffer->open(libFileName(filename))) { delete newBuffer; return false; }
 currentBuffer = newBuffer;
 currentBuffer->switchTo();
 return true;
};

void rgFinishLex()
{
 VPRINTF("<rg> rgFinishLex()\n");
 while(currentBuffer)
  {
   RgBufferState* lastBuffer = currentBuffer->lastBuffer();
   delete currentBuffer;
   currentBuffer = lastBuffer;
  }
}

//--------------------------------------------------------------------------------------------------------------------------------

int rgwrap()
{
 return 1;
}

//--------------------------------------------------------------------------------------------------------------------------------
