#include "gperiodic.h"
#include "gpdata.h"
#include "gparser.h"
#include <stdio.h>
#include <string.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <glib.h>

extern void init_lexer(void);
extern void cleanup_lexer(void);
extern void init_parser(void);
extern void cleanup_parser(void);

extern int debug;
extern struct table_entry table [];

/**
 *  
 * Hydrogen,H,1,1.00794,0.0708,14.01,20.28,79,32,154,14.1,14.267 H-H,0.117 H-H,0.904 H-H,0.1815,2.20,1311.3,1 -1,1s,1,1,0x7FFFFFFF,0x0000AAAA,0x0000FFFF,0x
 *  
 * 
 **/
int parse_table(void)
{
  char *line;
  char *prompt = "";

  while((line = readline(prompt))) {
    printf("read line: %s\n",line);
    free(line);
  }

  return 1;
}

int quit = 0;
int gpparser_quit(const char*cmd,GList*args)
{
  quit = 1;
  return 1;
}

void gpparse_dump_element_data_for_num(int num)
{
  printf("%s,%s,%d,%s,%s,%s,%s,0x%08X,0x%08X,0x%08X\n",
    gpdata_get_element_name(num),
    gpdata_get_element_symbol(num),
    num,
    gpdata_get_element_weight(num),
    gpdata_get_element_density(num),
    gpdata_get_element_melting_point(num),
    gpdata_get_element_boiling_point(num),
    gpdata_get_element_atomic_radius(num),
    gpdata_get_element_covalent_radius(num),
    gpdata_get_element_ionic_radius(num),
    gpdata_get_element_atomic_volume(num),
    gpdata_get_element_specific_heat(num),
    gpdata_get_element_fusion_heat(num),
    gpdata_get_element_evaporation_heat(num),
    gpdata_get_element_termal_conductivity(num),
    gpdata_get_element_pauling(num),
    gpdata_get_element_energy(num),
    gpdata_get_element_oxidation(num),
    gpdata_get_element_configuration(num),
    gpdata_get_element_pixel_depth(num),
    gpdata_get_element_red(num),
    gpdata_get_element_green(num),
    gpdata_get_element_blue(num)
  );
}

void gpparse_print_element_data_for_num(int num)
{
  printf(
    "  Element:                            %s\n"
    "  Symbol:                             %s\n"
    "  Atomic Number:                      %d\n"
    "  Atomic Weight (u.m):                %s\n"
    "  Density (g/cm):                    %s\n"
    "  Melting Point (C):                 %s\n"
    "  Boiling Point (C):                 %s\n"
    "  Atomic Radius (pm):                 %s\n"
    "  Covalent Radius (pm):               %s\n"
    "  Ionic Radius (pm):                  %s\n"
    "  Atomic Volume (cm/mol):            %s\n"
    "  Specific Heat (@20C J/gmol):       %s\n"
    "  Fusion Heat (kJ/mol):               %s\n"
    "  Evaporation Heat (kJ/gmol):         %s\n"
    "  Termal Conductivity (@25C W/m K):  %s\n"
    "  Pauling:                            %s\n"
    "  Energy:                             %s\n"
    "  Oxidation:                          %s\n"
    "  Electronic Configuration:           %s\n",
    gpdata_get_element_name(num),
    gpdata_get_element_symbol(num),
    num,
    gpdata_get_element_weight(num),
    gpdata_get_element_density(num),
    gpdata_get_element_melting_point(num),
    gpdata_get_element_boiling_point(num),
    gpdata_get_element_atomic_radius(num),
    gpdata_get_element_covalent_radius(num),
    gpdata_get_element_ionic_radius(num),
    gpdata_get_element_atomic_volume(num),
    gpdata_get_element_specific_heat(num),
    gpdata_get_element_fusion_heat(num),
    gpdata_get_element_evaporation_heat(num),
    gpdata_get_element_termal_conductivity(num),
    gpdata_get_element_pauling(num),
    gpdata_get_element_energy(num),
    gpdata_get_element_oxidation(num),
    gpdata_get_element_configuration(num)
  );
}

int gpparser_info(const char*cmd,GList*args)
{
  char *arg;
  int num,i;

  if(g_list_length(args) <= 1) {
    printf("error:  %s <atomic number | symbol>\n",cmd);
    return 0;
  }

  for(i = 1; i < g_list_length(args); ++i ) {
    arg = g_list_nth_data(args,i);
    num = atoi(arg);

    if(!num) {
      num = gpdata_get_atomic_number_for_symbol(arg);
    }

    if(!isValidAtomicNumber(num)) {
      printf("error:  invalid atomic number or symbol\n");
      return 0;
    }
  
    if(!strcmp("dump",cmd)) {
      gpparse_dump_element_data_for_num(num);
    }
    else {
      gpparse_print_element_data_for_num(num);
    }
  }

  return 1;
}


int gpparser_table(const char*cmd,GList*args)
{
  char *matrix[18][10];
  int atomicNumber,x,y;

  memset(matrix,0,sizeof(matrix));
  for(atomicNumber = 1; atomicNumber <= 118; ++atomicNumber ) {
    x = gpdata_get_element_x_position(atomicNumber) - 1;
    y = gpdata_get_element_y_position(atomicNumber) - 1;
    matrix[x][y] = (char*)gpdata_get_element_symbol(atomicNumber);
  }

  for(x=0;x<10;++x) {
   printf("   ");
   for(y=0;y<18;++y) {
     printf("%- 3s ",matrix[y][x] ? matrix[y][x] : " ");
   }
   printf("\n");
  }

  return 1;
}

int gpparser_find(const char*cmd,GList*args)
{
  int atomicNumber,matches=0;
  char*str;

  if(g_list_length(args) != 2) {
    printf("error:  %s string\n",cmd);
    return 0;
  }

  str = g_list_nth_data(args,1);
  for(atomicNumber = 1; atomicNumber <= 118; ++atomicNumber ) {
    if(strstr(gpdata_get_element_name(atomicNumber),str)) {
      printf("% 4d %- 3s %s\n",
        atomicNumber,
        gpdata_get_element_symbol(atomicNumber),
        gpdata_get_element_name(atomicNumber)
      );
      ++matches;
    }
  }
  printf("found: %d match%s\n",matches,matches<2?"":"s");

  return 1;
}

int gpparser_help(const char*cmd,GList*args)
{
  printf(
    "\n"
    "Available Commands:\n"
    "  help\n"
    "       Display this help.\n"
    "\n"
    "  info <atomic number | symbol>\n"
    "       Display the data for a given element, or elements.\n"
    "       One or more numbers or symbols can be requested.\n"
    "\n"
    "  table\n"
    "       Display a table of the elements [symbols only].\n"
    "\n"
    "  dump <atomic number | symbol>\n"
    "       Dump elements in cvs format.\n"
    "\n"
    "  find <string>\n"
    "       Prints elements who's name contain string.\n"
    "\n"
    "  quit\n"
    "\n"
  );
  return 1;
}

struct commandStruct {
  char *commandName;
  int (*fp)(const char*,GList*);
};

struct commandStruct commands [] = {
  { "quit",  gpparser_quit },
  { "exit",  gpparser_quit },

  { "help",  gpparser_help },
  { "?",     gpparser_help },

  { "info",  gpparser_info },
  { "dump",  gpparser_info },
  { "find",  gpparser_find },
  { "table", gpparser_table },

  { NULL },
  NULL
};

GHashTable *command_table;
int hashTableBuilt = 0;
void gpparser_build_hash(void)
{
  struct commandStruct *p = commands;
  command_table = g_hash_table_new(g_str_hash,g_str_equal);

  while(p->commandName) {
    g_hash_table_insert(command_table,p->commandName,p->fp);
    p++;
  }
  hashTableBuilt = 1;
}

char* gpparser_completion_entry_function(const char*text,int state)
{
  static GList *completions = NULL;
  struct commandStruct *p = commands;

  if(0 == state) {
    /* destroy the old list and alloc a new one */
    if(completions) { g_list_free(completions); }
    completions = g_list_alloc();
    /* compute the matches */
    while(p->commandName) {
      if(!strncmp(p->commandName,text,strlen(text))) {
        g_list_append(completions,strdup(p->commandName));
      }
      ++p;
    }
    
    /* return the first one */
    return(g_list_nth_data(completions,1));
  }

  /* return the next match, or NULL */
  ++state;
  if(state < g_list_length(completions)) {
    return(g_list_nth_data(completions,state));
  }

  return (char*)NULL;
}

char* gpparser_completion_entry_function2(const char*text,int state)
{
  static GList *completions2 = NULL;
  int i;

  if(0 == state) {
    /* destroy the old list and alloc a new one */
    if(completions2) { g_list_free(completions2); }
    completions2 = g_list_alloc();

    /* compute the matches */
    for(i = 0; i < 118; ++i ) {
      if(!strncasecmp(gpdata_get_element_symbol(i+1),text,strlen(text))) {
        g_list_append(completions2,strdup(gpdata_get_element_symbol(i+1)));
      }
    }

    /* return the first one */
    return(g_list_nth_data(completions2,1));
  }

  /* return the next match, or NULL */
  ++state;
  if(state < g_list_length(completions2)) {
    return(g_list_nth_data(completions2,state));
  }

  return (char*)NULL;
}

char** gpparser_completion(char*text,int start,int end)
{
  struct commandStruct *p = commands;
  char **matches;
  matches = (char**)NULL;

  if(0 == start) {
    matches = completion_matches(text,gpparser_completion_entry_function);
  }
  else {
    matches = completion_matches(text,gpparser_completion_entry_function2);
  }

  return matches;
}

void gpparser_parse_line(char*line,int freeMemory)
{
  if(!hashTableBuilt) { gpparser_build_hash(); }
  init_lexer();
  init_parser();
  yy_scan_string(line);
  yyparse();
  if(strlen(line)) { add_history(line); }
  if(freeMemory)free(line);
  cleanup_lexer();
  cleanup_parser();
}

int interactive_shell(void)
{
  char *line, *prompt = "gperiodic> ";
  int num = 0;

  rl_attempted_completion_function = (CPPFunction*)gpparser_completion;
  /* rl_completion_entry_function = (int(*)())gpparser_completion_entry_function; */
  while((line = readline(prompt))) {
    gpparser_parse_line(line,1);
    if(quit) { break; }
  }
  printf("goodbye\n");
  g_hash_table_destroy(command_table);
  return 1;
}

void gpparser_reinit_parser(void)
{
  cleanup_lexer();
  cleanup_parser();
  init_lexer();
  init_parser();
}

int gpparser_execute_command(char *cmd, GList *args)
{
  int (*fp)(char*,GList*) = g_hash_table_lookup(command_table,cmd);

  if(fp) {
    fp(cmd,args);
  }
  else {
    if(strlen(cmd)) {
      printf("error invalid command: %s\n",cmd);
    }
    // else nothing on the line
  }
  return 1;
}

