/* GNOME-DB Manager component
 * Copyright (C) 1998,1999 Michael Lausch
 * Copyright (C) 2000,2001 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gda-config.h>
#include <gnome-db-dsn-config.h>
#include <gnome-db-dsn-config-druid.h>
#include "gnomedb-mgr.h"
#include "e-hpaned.h"

typedef struct {
	GtkWidget* notebook;
	GtkWidget* dsn_tab;
	GtkWidget* dsn_list;
} MANAGER_DsnInfo;

static void add_dsn_cb      (GtkWidget *w, MANAGER_DsnInfo *dsn_info);
static void edit_dsn_cb     (GtkWidget *w, MANAGER_DsnInfo *dsn_info);
static void refresh_list_cb (GtkWidget *w, MANAGER_DsnInfo *dsn_info);
static void remove_dsn_cb   (GtkWidget *w, MANAGER_DsnInfo *dsn_info);
static void test_dsn_cb     (GtkWidget *w, MANAGER_DsnInfo *dsn_info);

static GnomeUIInfo dsn_toolbar[] = {
	{ GNOME_APP_UI_ITEM, N_("Add"), N_("Add new data source"),
	  add_dsn_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_NEW, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Edit"), N_("Edit selected data source"),
	  edit_dsn_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_PROP, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Remove"), N_("Remove current data source"),
	  remove_dsn_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Refresh"), N_("Refresh Data Sources list"),
	  refresh_list_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_REFRESH, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Test"), N_("Test settings for this datasource"),
	  test_dsn_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_CONVERT, 0, 0, NULL },
	GNOMEUIINFO_END
};

/*
 * Private functions
 */
static void
fill_dsn_list (GtkCList *clist)
{
	GList *dsn_list;
	GList *l;

	g_return_if_fail(GTK_IS_CLIST(clist));

	gnome_db_clear_clist(clist);
	gtk_clist_freeze(clist);
	dsn_list = gda_dsn_list();
	for (l = g_list_first(dsn_list); l != NULL; l = g_list_next(l)) {
		GdaDsn *dsn = (GdaDsn *) l->data;

		if (dsn) {
			const gchar *row[3];

			row[0] = GDA_DSN_GDA_NAME(dsn);
			row[1] = GDA_DSN_PROVIDER(dsn);
			row[2] = GDA_DSN_DESCRIPTION(dsn);
			gtk_clist_append(clist, row);
		}
	}
	gtk_clist_thaw(clist);
	gda_dsn_free_list(dsn_list);
}

/*
 * Callbacks
 */
static void
dsn_druid_finished_cb (GnomeDbDsnConfigDruid *druid, gboolean error, gpointer user_data)
{
	MANAGER_DsnInfo *dsn_info;
	GtkWidget *dialog = (GtkWidget *) user_data;

	g_return_if_fail(GTK_IS_WINDOW(dialog));

	if (!error) {
		GdaDsn *dsn;

		dsn_info = (MANAGER_DsnInfo *) gtk_object_get_data(GTK_OBJECT(druid), "MANAGER_DsnInfo");
		if (dsn_info) {
			dsn = gnome_db_dsn_config_druid_get_dsn(druid);
			if (dsn) {
				gda_dsn_save(dsn);
				fill_dsn_list(GTK_CLIST(dsn_info->dsn_list));
			}
		}
	}

	/* close the dialog box */
	gtk_widget_destroy(dialog);
}

static void
add_dsn_cb (GtkWidget *w, MANAGER_DsnInfo *dsn_info)
{
	GtkWidget *dialog;
	GtkWidget *druid;

	g_return_if_fail(dsn_info != NULL);

	/* create the dialog */
	dialog = gtk_window_new(GTK_WINDOW_DIALOG);
	druid = gnome_db_dsn_config_druid_new();
	gtk_object_set_data(GTK_OBJECT(druid), "MANAGER_DsnInfo", (gpointer) dsn_info);
	gtk_signal_connect(GTK_OBJECT(druid),
			   "finish",
			   GTK_SIGNAL_FUNC(dsn_druid_finished_cb),
			   (gpointer) dialog);
	gtk_widget_show(druid);
	gtk_container_add(GTK_CONTAINER(dialog), druid);

	gtk_widget_show(dialog);
}

static void
clist_selection_cb (GtkCList *clist,
                    gint row,
                    gint col,
                    GdkEvent *event,
                    MANAGER_DsnInfo *dsn_info)
{
	gchar* txt = NULL;
	
	g_return_if_fail(dsn_info != NULL);
	g_return_if_fail(GTK_IS_CLIST(dsn_info->dsn_list));
}

static void
edit_dsn_cb (GtkWidget *w, MANAGER_DsnInfo *dsn_info)
{
	g_return_if_fail(dsn_info != NULL);

	/* get the current selection */
	if (GTK_CLIST(dsn_info->dsn_list)->selection) {
		gchar *txt = NULL;
		GtkWidget *dialog;
		GtkWidget *dsn_config;
		GList *sel = GTK_CLIST(dsn_info->dsn_list)->selection;

		gtk_clist_get_text(GTK_CLIST(dsn_info->dsn_list), GPOINTER_TO_INT(sel->data), 0, &txt);

		/* create dialog */
		dialog = gnome_dialog_new(_("Edit data source"),
								  GNOME_STOCK_BUTTON_OK,
								  GNOME_STOCK_BUTTON_CANCEL,
								  NULL);
		gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);

		dsn_config = gnome_db_dsn_config_new(txt);
		gtk_widget_show(dsn_config);
		gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), dsn_config, 1, 1, 0);

		/* run the dialog */
		if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
			gnome_db_dsn_config_save(GNOME_DB_DSN_CONFIG(dsn_config));
			fill_dsn_list(GTK_CLIST(dsn_info->dsn_list));
		}
		gnome_dialog_close(GNOME_DIALOG(dialog));
	}
}

static void
refresh_list_cb (GtkWidget *w, MANAGER_DsnInfo *dsn_info)
{
	g_return_if_fail(dsn_info != NULL);
	fill_dsn_list(GTK_CLIST(dsn_info->dsn_list));
}

static void
remove_dsn_cb (GtkWidget *w, MANAGER_DsnInfo *dsn_info)
{
	GdaDsn *dsn;

	g_return_if_fail(dsn_info != NULL);
	g_return_if_fail(GTK_IS_CLIST(dsn_info->dsn_list));

	/* get the current selection */
	if (GTK_CLIST(dsn_info->dsn_list)->selection) {
		gchar *txt = NULL;
		GList *sel = GTK_CLIST(dsn_info->dsn_list)->selection;

		gtk_clist_get_text(GTK_CLIST(dsn_info->dsn_list), GPOINTER_TO_INT(sel->data), 0, &txt);
		dsn = gda_dsn_find_by_name(txt);
		if (dsn) {
			gda_dsn_remove(dsn);
			fill_dsn_list(GTK_CLIST(dsn_info->dsn_list));
			gda_dsn_free(dsn);
		}
		else gnome_db_show_error("no DSN found by '%s'", txt);
	}
	else gnome_db_show_error("no selection in GtkCList");
}

static void
test_dsn_cb (GtkWidget *w, MANAGER_DsnInfo *dsn_info)
{
	g_return_if_fail(dsn_info != NULL);
	g_return_if_fail(GTK_IS_CLIST(dsn_info->dsn_list));

	/* get the current selection */
	if (GTK_CLIST(dsn_info->dsn_list)->selection) {
		gchar *txt = NULL;
		GtkWidget *dialog;
		GtkWidget *table;
		GtkWidget *label;
		GtkWidget *entry;
		GtkWidget *user_entry;
		GtkWidget *password_entry;
		GdaDsn *dsn;
		GList *sel = GTK_CLIST(dsn_info->dsn_list)->selection;

		gtk_clist_get_text(GTK_CLIST(dsn_info->dsn_list), GPOINTER_TO_INT(sel->data), 0, &txt);
		dsn = gda_dsn_find_by_name(txt);
		if (!dsn) {
			gnome_db_show_error(_("No data source found with name %s"), txt);
			return;
		}

		/* create the dialog */
		dialog = gnome_dialog_new(_("Test data source"),
								  GNOME_STOCK_BUTTON_OK,
								  GNOME_STOCK_BUTTON_CANCEL,
								  NULL);
		gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
		table = gnome_db_new_table_widget(3, 2, FALSE);
		gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table, 1, 1, 0);

		label = gnome_db_new_label_widget(_("Data source name"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
		entry = gnome_db_new_entry_widget(0, FALSE);
		gtk_entry_set_text(GTK_ENTRY(entry), txt);
		gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

		label = gnome_db_new_label_widget(_("Username"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
		user_entry = gnome_db_new_entry_widget(0, TRUE);
		gtk_entry_set_text(GTK_ENTRY(user_entry), GDA_DSN_USERNAME(dsn));
		gtk_table_attach(GTK_TABLE(table), user_entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

		label = gnome_db_new_label_widget(_("Password"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
		password_entry = gnome_db_new_entry_widget(0, TRUE);
		gtk_entry_set_visibility(GTK_ENTRY(password_entry), FALSE);
		gtk_table_attach(GTK_TABLE(table), password_entry, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);

		/* run the dialog */
		if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
			GdaConnection *cnc = gda_connection_new(gda_corba_get_orb());

			gda_connection_set_provider(cnc, GDA_DSN_PROVIDER(dsn));
			if (gda_connection_open(cnc,
									GDA_DSN_DSN(dsn),
									gtk_entry_get_text(GTK_ENTRY(user_entry)),
									gtk_entry_get_text(GTK_ENTRY(password_entry))) == 0) {
				gnome_db_show_message(_("The connection was successful!"));
				gda_connection_close(cnc);
			}
			else
				gnome_db_show_error(_("There was an error while testing connection"));
			gda_connection_free(cnc);
		}
		gnome_dialog_close(GNOME_DIALOG(dialog));
	}
}

/*
 * Public functions
 */
void
manager_dsn_init (GtkNotebook *notebook)
{
	MANAGER_DsnInfo* dsn_info;
	GtkWidget*       scroll;
	GtkWidget*       button;
	GtkWidget*       toolbar;
	const gchar*     clist_titles[] = { N_("Name"), N_("Provider"), N_("Description") };
	
	dsn_info = g_new0(MANAGER_DsnInfo, 1);
	dsn_info->notebook = GTK_WIDGET(notebook);
	dsn_info->dsn_tab = gnome_db_new_table_widget(4, 2, FALSE);
	
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(dsn_info->dsn_tab), scroll, 0, 1, 1, 2,
	                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
					 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
					 3, 3);
	dsn_info->dsn_list = gnome_db_new_clist_widget(clist_titles, 3);
	gtk_signal_connect(GTK_OBJECT(dsn_info->dsn_list),
	                   "select_row",
	                   GTK_SIGNAL_FUNC(clist_selection_cb),
	                   (gpointer) dsn_info);
	gtk_signal_connect(GTK_OBJECT(dsn_info->dsn_list),
	                   "unselect_row",
	                   GTK_SIGNAL_FUNC(clist_selection_cb),
	                   (gpointer) dsn_info);
	fill_dsn_list(GTK_CLIST(dsn_info->dsn_list));
	gtk_container_add(GTK_CONTAINER(scroll), dsn_info->dsn_list);
	 
	                 
	toolbar = gnome_db_new_toolbar_widget(GTK_ORIENTATION_HORIZONTAL,
	                                      GTK_TOOLBAR_ICONS,
	                                      dsn_toolbar,
	                                      (gpointer) dsn_info);
	gtk_table_attach(GTK_TABLE(dsn_info->dsn_tab), toolbar, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dsn_info->dsn_tab, gtk_label_new(_("Data Sources")));
}

