/* GNOME DB library
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include "gnome-db-combo.h"
#include <gtk/gtk.h>

static void gnome_db_combo_class_init (GnomeDbComboClass *klass);
static void gnome_db_combo_init (GnomeDbCombo *dbcombo);

static void gnome_db_combo_destroy (GtkObject *object);

enum {
	ARG_EDITABLE,
	ARG_ROW_COUNT,
	ARG_STRING,
	ARG_GDA_RECORDSET,
	ARG_COLUMN
};

/*
 * GnomeDbCombo widget signals
 */
enum {
	GNOME_DB_COMBO_ERROR,
	GNOME_DB_COMBO_SELECTION_CHANGED,
	GNOME_DB_COMBO_LAST_SIGNAL
};

static gint gnome_db_combo_signals[GNOME_DB_COMBO_LAST_SIGNAL] = { 0, };

/*
 * Private functions
 */
static void
show_recordset (GnomeDbCombo *dbcombo, gint pos)
{
	GList *strings = NULL;
	
	gtk_list_clear_items(GTK_LIST(GTK_COMBO(dbcombo)->list), 0, -1);
	if (GDA_IS_RECORDSET (dbcombo->recset)) {
		gulong position = gda_recordset_move(dbcombo->recset, 1, 0);
		while (position != GDA_RECORDSET_INVALID_POSITION &&
		       !gda_recordset_eof(dbcombo->recset)) {
			GdaField *field = gda_recordset_field_idx(dbcombo->recset, pos);
			strings = g_list_append(strings, gda_stringify_value(0, 0, field));
			
			/* fetch next row */
			position = gda_recordset_move(dbcombo->recset, 1, 0);
		}
		gtk_combo_set_popdown_strings(GTK_COMBO(dbcombo), strings);
		g_list_free (strings);
	}
}

/*
 * Callbacks
 */
static void
selection_changed_cb (GtkWidget *w, gpointer data)
{
	GnomeDbCombo *dbcombo;
	gchar *str;
	
	g_return_if_fail(GTK_IS_LIST(w));
	g_return_if_fail(GNOME_DB_IS_COMBO(data));
	
	dbcombo = GNOME_DB_COMBO(data);
	
	/* retrieve GtkCombo's current selection */
	str = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(dbcombo)->entry));
	if (str != 0) {
		/* emit GnomeDbCombo's "selection_changed" signal */
		gtk_signal_emit_by_name(GTK_OBJECT(dbcombo), "selection_changed", str);
	}
}

/*
 * GnomeDbCombo widget interface
 */
static void
gnome_db_combo_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GnomeDbCombo *dbcombo = (GnomeDbCombo *) object;

	g_return_if_fail (GNOME_DB_IS_COMBO (dbcombo));

	switch (arg_id) {
	case ARG_EDITABLE :
		GTK_VALUE_INT (*arg) = gnome_db_combo_get_editable (dbcombo);
		break;
	case ARG_ROW_COUNT :
		GTK_VALUE_INT (*arg) = gnome_db_combo_get_row_count (dbcombo);
		break;
	case ARG_STRING :
		GTK_VALUE_STRING (*arg) = gnome_db_combo_get_string (dbcombo);
		break;
	case ARG_GDA_RECORDSET :
		GTK_VALUE_OBJECT (*arg) = (GtkObject *) dbcombo->recset;
		break;
	case ARG_COLUMN :
		GTK_VALUE_INT (*arg) = dbcombo->column;
		break;
	default :
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

static void
gnome_db_combo_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GnomeDbCombo *dbcombo = (GnomeDbCombo *) object;

	g_return_if_fail (GNOME_DB_IS_COMBO (dbcombo));

	switch (arg_id) {
	case ARG_EDITABLE :
		gnome_db_combo_set_editable (dbcombo, GTK_VALUE_INT (*arg));
		break;
	case ARG_GDA_RECORDSET :
		gnome_db_combo_set_recordset (dbcombo,
					      GTK_VALUE_OBJECT (*arg),
					      dbcombo->column);
		break;
	case ARG_COLUMN :
		gnome_db_combo_set_recordset (dbcombo,
					      dbcombo->recset,
					      GTK_VALUE_INT (*arg));
		break;
	default :
		break;
	}
}

static void
gnome_db_combo_class_init (GnomeDbComboClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	
	gnome_db_combo_signals[GNOME_DB_COMBO_ERROR] =
		gtk_signal_new ("error", GTK_RUN_FIRST, object_class->type,
		                GTK_SIGNAL_OFFSET(GnomeDbComboClass, error),
		                gtk_signal_default_marshaller,
		                GTK_TYPE_NONE, 0);
	gnome_db_combo_signals[GNOME_DB_COMBO_SELECTION_CHANGED] =
		gtk_signal_new("selection_changed", GTK_RUN_FIRST, object_class->type,
		               GTK_SIGNAL_OFFSET(GnomeDbComboClass, selection_changed),
		               gtk_signal_default_marshaller,
		               GTK_TYPE_NONE, 0);
	gtk_object_class_add_signals(object_class, gnome_db_combo_signals,
	                             GNOME_DB_COMBO_LAST_SIGNAL);

	object_class->destroy = gnome_db_combo_destroy;
	object_class->get_arg = gnome_db_combo_get_arg;
	object_class->set_arg = gnome_db_combo_set_arg;

	gtk_object_add_arg_type ("GnomeDbCombo:editable",
				 GNOME_DB_TYPE_COMBO,
				 GTK_ARG_READWRITE,
				 ARG_EDITABLE);
	gtk_object_add_arg_type ("GnomeDbCombo:row_count",
				 GNOME_DB_TYPE_COMBO,
				 GTK_ARG_READABLE,
				 ARG_ROW_COUNT);
	gtk_object_add_arg_type ("GnomeDbCombo:string",
				 GNOME_DB_TYPE_COMBO,
				 GTK_ARG_READABLE,
				 ARG_STRING);
	gtk_object_add_arg_type ("GnomeDbCombo:GdaRecordset",
				 GNOME_DB_TYPE_COMBO,
				 GTK_ARG_READWRITE,
				 ARG_GDA_RECORDSET);
	gtk_object_add_arg_type ("GnomeDbCombo:column",
				 GNOME_DB_TYPE_COMBO,
				 GTK_ARG_READWRITE,
				 ARG_COLUMN);

	klass->error = NULL;
	klass->selection_changed = NULL;
}

static void
gnome_db_combo_init (GnomeDbCombo *dbcombo)
{
	gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dbcombo)->entry), FALSE);
	dbcombo->recset = 0;
	dbcombo->total_rows = 0;
	dbcombo->editable = FALSE;
	
	/* connect to GtkList's "selection_changed" signal */
	gtk_signal_connect(GTK_OBJECT(GTK_COMBO(dbcombo)->list),
	                   "selection_changed",
	                   GTK_SIGNAL_FUNC(selection_changed_cb),
	                   (gpointer) dbcombo);
}

guint
gnome_db_combo_get_type (void)
{
	static guint db_combo_type = 0;
	
	if (!db_combo_type) {
		GtkTypeInfo db_combo_info = {
			"GnomeDbCombo",
			sizeof (GnomeDbCombo),
			sizeof (GnomeDbComboClass),
			(GtkClassInitFunc) gnome_db_combo_class_init,
			(GtkObjectInitFunc) gnome_db_combo_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		db_combo_type = gtk_type_unique(gtk_combo_get_type(), &db_combo_info);
	}
	return (db_combo_type);
}

/**
 * gnome_db_combo_new
 * @recset: recordset to be shown
 * @pos: position of the field being shown
 *
 * Creates a new GnomeDbCombo widget. You can optionally specify a GdaRecordset
 * object which will be used by the GnomeDbCombo widget
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget *
gnome_db_combo_new (GdaRecordset *recset, gint pos)
{
	GtkWidget *dbcombo = gtk_type_new(gnome_db_combo_get_type());
	
	if (recset != 0)
		gnome_db_combo_set_recordset(GNOME_DB_COMBO(dbcombo), recset, pos);
	return (dbcombo);
}

static void 
gnome_db_combo_destroy (GtkObject *object)
{
	GnomeDbCombo *dbcombo;
	GtkObjectClass *parent_class;
	
	g_return_if_fail (GNOME_DB_IS_COMBO (object));
	
	dbcombo = GNOME_DB_COMBO (object);
	if (dbcombo->recset != 0)
		gda_recordset_free (dbcombo->recset);

	parent_class = gtk_type_class (gtk_combo_get_type ());
	if (parent_class && parent_class->destroy)
		parent_class->destroy (object);
}

/**
 * gnome_db_combo_get_editable
 * @dbcombo: the GnomeDbCombo widget
 *
 * Gets the editing mode for the given #GnomeDbCombo widget.
 *
 * Returns: TRUE if the combo box is editable , or FALSE if not
 */
gboolean
gnome_db_combo_get_editable (GnomeDbCombo *dbcombo)
{
	g_return_val_if_fail(GNOME_DB_IS_COMBO(dbcombo), FALSE);
	return (dbcombo->editable);
}

/**
 * gnome_db_combo_set_editable
 * @dbcombo: a #GnomeDbCombo widget
 * @editable: whether to let (TRUE) or nor let (FALSE) editing
 *
 * Sets the editing mode for the given #GnomeDbCombo widget. When the
 * editing mode is set to TRUE, it means that users can type on the combo
 * entry widget, thus making this widget useful for data entry screens where
 * you want users to be able to make modifications. 
 */
void
gnome_db_combo_set_editable (GnomeDbCombo *dbcombo, gboolean editable)
{
	g_return_if_fail(GNOME_DB_IS_COMBO(dbcombo));
	
	gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dbcombo)->entry), editable);
	dbcombo->editable = editable;
}

/**
 * gnome_db_combo_refresh
 * @dbcombo: a #GnomeDbCombo widget
 *
 * Refreshes the data being displayed by the given #GnomeDbCombo widget.
 * This means that the same query is executed on the connection associated
 * with this combo widget.
 */
void
gnome_db_combo_refresh (GnomeDbCombo *dbcombo)
{
	g_return_if_fail(dbcombo != NULL);
	g_return_if_fail(GNOME_DB_IS_COMBO(dbcombo));
}

/**
 * gnome_db_combo_get_row_count
 * @dbcombo: a #GnomeDbCombo widget
 *
 * Returns the number of rows being currently displayed by the given
 * #GnomeDbCombo widget.
 *
 * Returns: the number of rows
 */
gint
gnome_db_combo_get_row_count (GnomeDbCombo *dbcombo)
{
	g_return_if_fail(GNOME_DB_IS_COMBO(dbcombo));
	return (dbcombo->total_rows);
}

/**
 * gnome_db_combo_get_string
 * @dbcombo: a #GnomeDbCombo widget
 *
 * Returns the current selection in the given #GnomeDbCombo widget as
 * a string.
 *
 * Returns: the currently selected item's string
 */
gchar *
gnome_db_combo_get_string (GnomeDbCombo *dbcombo)
{
	g_return_val_if_fail (dbcombo != NULL, NULL);
	g_return_val_if_fail (GNOME_DB_IS_COMBO (dbcombo), NULL);
	return gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (dbcombo)->entry));
}

/**
 * gnome_db_combo_get_recordset
 * @dbcombo: a #GnomeDbCombo widget
 *
 * Gets the #GdaRecordset being displayed by the given #GnomeDbCombo widget
 *
 * Returns: the #GdaRecordset object associated with the combo widget, or
 * NULL if none.
 */
GdaRecordset *
gnome_db_combo_get_recordset (GnomeDbCombo *dbcombo)
{
	g_return_val_if_fail (GNOME_DB_IS_COMBO (dbcombo), NULL);
	return dbcombo->recset;
}

/**
 * gnome_db_combo_set_recordset
 * @dbcombo: a #GnomeDbCombo widget
 * @recset: the recordset to be shown
 * @pos: the column to be shown
 *
 * Associates a #GdaRecordset object with the given #GnomeDbCombo widget. This
 * forces the combo widget to get the new set of rows from the database and
 * display them.
 *
 * It's important to note that the #GnomeDbCombo widget keeps a reference
 * to the #GdaRecordset object, so it's safe if you call #gda_recordset_free
 * on the same recordset you used in the call to this function. Note however
 * that if you close the recordset (by calling #gda_recordset_close), this
 * will indeed affect the combo widget if it's keeping a reference to that
 * recordset.
 */
void
gnome_db_combo_set_recordset (GnomeDbCombo *dbcombo, GdaRecordset *recset, gint pos)
{
	g_return_if_fail(GNOME_DB_IS_COMBO(dbcombo));
	g_return_if_fail(GDA_IS_RECORDSET(recset));
	g_return_if_fail(pos >= 0);
	
	if (recset != NULL && recset == dbcombo->recset)
		gnome_db_combo_refresh(dbcombo);
	else {
		if (dbcombo->recset != 0) {
			gda_recordset_free(dbcombo->recset);
		}
		gtk_object_ref (GTK_OBJECT (recset));
		dbcombo->recset = recset;
		dbcombo->column = pos;
		show_recordset(dbcombo, pos);
	}
}

/**
 * gnome_db_combo_sync
 * @dbcombo: a #GnomeDbCombo widget
 *
 * Synchronizes the underlying #GdaRecordset object associated with the
 * given #GnomeDbCombo widget so that its current row is set to the row
 * selected in the #GnomeDbCombo widget.
 *
 * This is specially useful for data entry screens, where you can have a list
 * of all primary keys in a #GnomeDbCombo widget, and, when the user selects
 * an entry in the combo pop-down list, other data entry widgets (such as
 * #GnomeDbTable, for instance) update themselves with the contents of the
 * new selected row.
 */
void
gnome_db_combo_sync (GnomeDbCombo *dbcombo)
{
	GList*     selected;
	GtkWidget* item;
	gint       row_index;
	
	g_return_if_fail(GNOME_DB_IS_COMBO(dbcombo));
	g_return_if_fail(GDA_IS_RECORDSET(dbcombo->recset));
	
	/* find the index of the current combo selection */
	selected = GTK_LIST(GTK_COMBO(dbcombo)->list)->selection;
	if (selected) {
		item = GTK_WIDGET(selected->data);
		row_index = gtk_list_child_position(GTK_LIST(GTK_COMBO(dbcombo)->list),  item);
		
		/* reposition the record set */
		gda_recordset_move_first(dbcombo->recset);
		gda_recordset_move(dbcombo->recset, row_index, NULL);
	}
}
