/* gnomad2.c
   Main loop and GTK+ GUI setup file and callback functions
   Copyright (C) 2001-2003 Linus Walleij

This file is part of the GNOMAD package.

GNOMAD is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

You should have received a copy of the GNU General Public License
along with GNOMAD; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. 

*/

#include "common.h"
#include "jukebox.h"
#include "xfer.h"
#include "data.h"
#include "playlists.h"
#include "prefs.h"
#include "util.h"
#if !GTK_CHECK_VERSION(2,6,0)
#include <libgnomeui/libgnomeui.h>
#endif
#ifdef HAVE_DBUSGLIB
#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus-glib.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>
#endif
#ifdef HAVE_HAL
#include <hal/libhal.h>
#endif

/* This one should be global really */
GtkWidget *main_window;

/* Local variables */
#if GTK_CHECK_VERSION(2,6,0)
static GdkPixbuf *icon_pixbuf;
#endif
#if !GTK_CHECK_VERSION(2,4,0)
static GtkWidget *jukeboxowner_dialog;
static GtkWidget *jukeboxowner_entry;
static GtkWidget *select_jukebox_dialog;
static GtkWidget *jukebox_menu;
#endif
// The expose() event will be treated JUST ONCE at the first window creation
static gboolean justonce = TRUE;
#if defined(HAVE_DBUSGLIB) && defined(HAVE_HAL)
DBusConnection *dbus_session_conn;
DBusConnection *dbus_system_conn;
DBusError dbus_error;
LibHalContext *hal_context;
char *global_udistr = NULL;
#endif


/* Expect these functions */
void scan_jukebox(gpointer data);

/* Response to DELETE event (window destruction) */
static gint delete(GtkWidget *widget,
		   GtkWidget *event,
		   gpointer   data)
{
  if (jukebox_locked) {
    create_error_dialog(_("Cannot quit! Jukebox is busy!"));
    return TRUE;
  }
  write_prefs_file();
  jukebox_release();

  gtk_main_quit();
  return FALSE;
}

static GtkWidget *create_jukebox_menu(gchar **jukeboxes)
{
  GtkWidget *option_menu;
  GSList *group = NULL;
  GtkWidget *menu;
  GtkWidget *menu_item;
  gint i;

  /* Create the option menu */
  // FAILS WITH GTK+ 2.3.x/2.4
  option_menu = gtk_option_menu_new();
  menu = gtk_menu_new();
  /* Add a row to the menu */
  for (i = 0; i < vectorlength(jukeboxes); i++) {
    menu_item = gtk_radio_menu_item_new_with_label(group, jukeboxes[i]);
    group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM (menu_item));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
    /* Check the first item */
    if (i == 0) {
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item), TRUE);
    }
    gtk_widget_show(menu_item);
  }
  gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu), 0);

  return option_menu;
}

#if !GTK_CHECK_VERSION(2,4,0)
static GCallback jukebox_ok_signal(GtkButton *button,
				   gpointer data)
{
  guint selection = gtk_option_menu_get_history(GTK_OPTION_MENU(jukebox_menu));
  // Select the first jukebox
  gtk_widget_destroy(select_jukebox_dialog);
  if (jukebox_select(selection)) {
    // Initially scan the jukebox
    scan_jukebox(NULL);
#ifdef HAVE_DBUSGLIB
    // Transmit jukebox connection message on D-BUS
#endif
  }
}
#endif

static void create_select_jukebox_dialog(void)
{
  gchar **jukeboxes;
  jukeboxes = jukebox_discover();
  if (jukeboxes != NULL) {
    if (vectorlength(jukeboxes) > 1) {
      int i;
      GtkWidget *label;
      // Build a selection menu
#if GTK_CHECK_VERSION(2,4,0)
      GtkWidget *select_jukebox_dialog;
      GtkWidget *jukebox_menu;

      select_jukebox_dialog = gtk_message_dialog_new (GTK_WINDOW(main_window),
						      GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_MESSAGE_QUESTION,
						      GTK_BUTTONS_OK_CANCEL,
						      _("Select jukebox"));
      jukebox_menu = create_jukebox_menu(jukeboxes);
      gtk_window_set_position (GTK_WINDOW(select_jukebox_dialog), GTK_WIN_POS_MOUSE);
      gtk_box_set_spacing (GTK_BOX(GTK_DIALOG(select_jukebox_dialog)->vbox), 5);

      label = gtk_label_new(_("Select jukebox to operate:"));
      gtk_box_pack_start(GTK_BOX(GTK_DIALOG(select_jukebox_dialog)->vbox), 
			 label, TRUE, TRUE, 0);
      gtk_widget_show(label);
      gtk_box_pack_start(GTK_BOX(GTK_DIALOG(select_jukebox_dialog)->vbox), 
			 jukebox_menu, TRUE, TRUE, 0);
      gtk_widget_show(jukebox_menu);
      gtk_widget_show_all(select_jukebox_dialog);
      i = gtk_dialog_run(GTK_DIALOG(select_jukebox_dialog));
      if (i == GTK_RESPONSE_OK) {
	guint selection = gtk_option_menu_get_history(GTK_OPTION_MENU(jukebox_menu));
	gtk_widget_destroy(select_jukebox_dialog);
	// Select the first jukebox
	if (jukebox_select(selection)) {
	  // Initially scan the jukebox
	  scan_jukebox(NULL);
	}
      } else {
	gtk_widget_destroy(select_jukebox_dialog);
      }
#else
      select_jukebox_dialog = gnome_dialog_new(_("Select jukebox"),
					       GNOME_STOCK_BUTTON_CANCEL,
					       GNOME_STOCK_BUTTON_OK,
					       NULL);
      gnome_dialog_button_connect_object(GNOME_DIALOG(select_jukebox_dialog),
					 0,
					 G_CALLBACK(gtk_widget_destroy),
					 GTK_OBJECT(select_jukebox_dialog));
      gnome_dialog_button_connect(GNOME_DIALOG(select_jukebox_dialog),
				  1,
				  G_CALLBACK(jukebox_ok_signal),
				  NULL);
      jukebox_menu = create_jukebox_menu(jukeboxes);
      gtk_window_set_position (GTK_WINDOW(select_jukebox_dialog), GTK_WIN_POS_MOUSE);
      g_signal_connect_object(GTK_OBJECT (select_jukebox_dialog),
			      "delete_event",
			      G_CALLBACK(gtk_widget_destroy),
			      GTK_OBJECT(select_jukebox_dialog),
			      0);
      label = gtk_label_new(_("Select jukebox to operate:"));
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(select_jukebox_dialog)->vbox), 
			 label, TRUE, TRUE, 0);
      gtk_widget_show(label);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(select_jukebox_dialog)->vbox), 
			 jukebox_menu, TRUE, TRUE, 0);
      gtk_widget_show(jukebox_menu);
      gtk_widget_show(select_jukebox_dialog);
      g_strfreev(jukeboxes);
#endif
    } else {
      // Select the first jukebox
      if (jukebox_select(0)) {
	// Initially scan the jukebox
	scan_jukebox(NULL);
#ifdef HAVE_DBUSGLIB
    // Transmit jukebox connection message on D-BUS
#endif
      }
    }
  }
}

/* Response to EXPOSE event (window creation) */
static gint expose(GtkWidget *widget,
		   GtkWidget *event,
		   gpointer   data)
{
  if (justonce) {
    justonce = FALSE;
    if (get_prefs_autoscan()) {
      // This will be treated JUST ONCE at the first window creation
      create_select_jukebox_dialog();
    }
    if (get_prefs_set_time()) {
      if (jukebox_selected() && !jukebox_locked) {
	jukebox_synchronize_time();
      }
    }
  }
  return(FALSE);
}

/* The about box */
static void about_box(gpointer data)
{
  const gchar *authors[] = {"Linus Walleij - main code, parts of libnjb and libmtp", 
			    "John Mechalas (Seagull) - libnjb",
			    "César Talón - Jukebox 3 examinations", 
			    "Richard A. Low - initial MTP work and libmtp",
			    NULL};
  const gchar *copystring = "Copyright (C) 2006 Linus Walleij";
  gchar *aboutstring = _("A program to communicate with\nCreative Jukeboxes\nand MTP devices");
  gchar *translator_credits = _("translator_credits");

  GtkWidget *about;
  /*
   * For a modern GTK+ we remove the use of libgnomeui stuff.
   */
#if GTK_CHECK_VERSION(2,6,0)
  about = gtk_about_dialog_new();
  gtk_about_dialog_set_name(GTK_ABOUT_DIALOG(about), PACKAGE);
  gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(about), VERSION);
  gtk_about_dialog_set_copyright(GTK_ABOUT_DIALOG(about), copystring);
  gtk_about_dialog_set_comments(GTK_ABOUT_DIALOG(about), aboutstring);
  gtk_about_dialog_set_license(GTK_ABOUT_DIALOG(about), "GPL 2+");
  gtk_about_dialog_set_website(GTK_ABOUT_DIALOG(about), "http://gnomad2.sourceforge.net/");
  gtk_about_dialog_set_website_label(GTK_ABOUT_DIALOG(about), "Gnomad Homepage");
  gtk_about_dialog_set_authors(GTK_ABOUT_DIALOG(about), authors);
  if (strcmp(translator_credits, "translator_credits")) {
    gtk_about_dialog_set_translator_credits(GTK_ABOUT_DIALOG(about), translator_credits);
  }
#else
  const gchar *thanks[] = {"ALL YOUR BASE ARE", "BELONG TO US", NULL};

  about = gnome_about_new(PACKAGE,
			  VERSION,
			  copystring,
			  aboutstring,
			  authors,
			  thanks,
			  strcmp(translator_credits, "translator_credits") != 0 ? translator_credits : NULL,
			  NULL);
#endif
  gtk_dialog_run(GTK_DIALOG(about));
  gtk_widget_destroy(about);
}

/*
 * Scan the jukebox for files
 */

void scan_jukebox(gpointer data)
{
  GtkWidget *label, *dialog;
  static scan_thread_arg_t scan_thread_args;

  if (!jukebox_selected()) {
    create_select_jukebox_dialog();
    return;
  }
  if (jukebox_locked) {
    create_error_dialog(_("Jukebox busy"));
    return;
  }
  jukebox_locked = TRUE;
  cancel_jukebox_operation = FALSE;

#if GTK_CHECK_VERSION(2,4,0)
  dialog = gtk_message_dialog_new (GTK_WINDOW(main_window),
				   GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
				   GTK_MESSAGE_INFO,
				   GTK_BUTTONS_CLOSE,
				   _("Scanning jukebox library"));
  gtk_window_set_title (GTK_WINDOW (dialog), _("Retrieving metadata from jukebox"));
  g_signal_connect_object(GTK_OBJECT(dialog),
			  "delete_event",
			  G_CALLBACK(cancel_jukebox_operation_click),
			  NULL,
			  0);
  g_signal_connect_object(GTK_OBJECT(dialog), 
			  "response",
			  G_CALLBACK(cancel_jukebox_operation_click), 
			  NULL,
			  0);
  /* A blank label for progress messages */
  label = gtk_label_new("");
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, TRUE, TRUE, 0);
  gtk_widget_show_all(dialog);
#else
  label = gtk_label_new(_("Retrieveing metadata from jukebox"));
  dialog = gnome_dialog_new(_("Scanning jukebox library"),
			    GNOME_STOCK_BUTTON_CANCEL,
			    NULL);
  gnome_dialog_button_connect(GNOME_DIALOG(dialog),
			      0,
			      G_CALLBACK(cancel_jukebox_operation_click),
			      NULL);
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), label, TRUE,
		      TRUE, 0);
  /* A blank label for progress messages */
  label = gtk_label_new("");
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), label, TRUE,
		      TRUE, 0);
  gtk_widget_show_all(dialog);
#endif

  /* A bunch of arguments needed by the scanning thread */
  scan_thread_args.dialog = dialog;
  scan_thread_args.label = label;
  scan_thread_args.pltreestore = playlist_widgets.pltreestore;

  g_thread_create(scan_thread,(gpointer) &scan_thread_args, FALSE, NULL);
  return;
}

#if !GTK_CHECK_VERSION(2,4,0)
static void set_owner_ok_button (GtkButton *button,
				 gpointer data)
{
  gchar *owner;

  if (jukebox_locked) {
    create_error_dialog(_("Jukebox busy - change discarded"));
    return;
  }
  owner = g_strdup(gtk_entry_get_text(GTK_ENTRY(jukeboxowner_entry)));
  jukebox_set_ownerstring(owner);
  g_free(owner);
  gtk_widget_destroy(jukeboxowner_dialog);
}
#endif

/* A dialog to select the owner */
static void set_jukeboxowner_dialog(gpointer data)
{
  GtkWidget *label;
  gchar *ownerstring;

#if GTK_CHECK_VERSION(2,4,0)
  GtkWidget *jukeboxowner_dialog;
  GtkWidget *jukeboxowner_entry;
  gint i;
  gchar *owner;

  if (!jukebox_selected()) {
    create_select_jukebox_dialog();
    return;
  }
  jukeboxowner_dialog = gtk_message_dialog_new (GTK_WINDOW(main_window),
						GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
						GTK_MESSAGE_QUESTION,
						GTK_BUTTONS_OK_CANCEL,
						_("Set owner string"));
  gtk_box_set_spacing (GTK_BOX(GTK_DIALOG(jukeboxowner_dialog)->vbox), 5);
  
  label = gtk_label_new(_("Edit the owner of this jukebox:"));
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(jukeboxowner_dialog)->vbox), 
		     label, TRUE, TRUE, 0);
  gtk_widget_show(label);
  jukeboxowner_entry = gtk_entry_new();
  ownerstring = jukebox_get_ownerstring();
  if (ownerstring != NULL)
    gtk_entry_set_text(GTK_ENTRY(jukeboxowner_entry), ownerstring);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(jukeboxowner_dialog)->vbox), jukeboxowner_entry, TRUE, TRUE, 0);
  gtk_widget_show(jukeboxowner_entry);
  gtk_widget_show_all(jukeboxowner_dialog);
  i = gtk_dialog_run(GTK_DIALOG(jukeboxowner_dialog));
  if (i == GTK_RESPONSE_OK) {
    if (jukebox_locked) {
      create_error_dialog(_("Jukebox busy - change discarded"));
      return;
    }
    owner = g_strdup(gtk_entry_get_text(GTK_ENTRY(jukeboxowner_entry)));
    jukebox_set_ownerstring(owner);
    g_free(owner);
    gtk_widget_destroy(jukeboxowner_dialog);    
  } else {
    gtk_widget_destroy(jukeboxowner_dialog);
  }
#else
  if (!jukebox_selected()) {
    create_select_jukebox_dialog();
    return;
  }
  jukeboxowner_dialog = gnome_dialog_new(_("Set owner string"),
					 GNOME_STOCK_BUTTON_CANCEL,
					 GNOME_STOCK_BUTTON_OK,
					 NULL);
  gnome_dialog_button_connect_object(GNOME_DIALOG(jukeboxowner_dialog),
				     0,
				     GTK_SIGNAL_FUNC(gtk_widget_destroy),
				     GTK_OBJECT(jukeboxowner_dialog));
  gnome_dialog_button_connect(GNOME_DIALOG(jukeboxowner_dialog),
			      1,
			      GTK_SIGNAL_FUNC(set_owner_ok_button),
			      NULL);
  gtk_window_set_position (GTK_WINDOW (jukeboxowner_dialog), GTK_WIN_POS_MOUSE);
  label = gtk_label_new(_("Edit the owner of this jukebox:"));
  gtk_box_pack_start (GTK_BOX(GNOME_DIALOG(jukeboxowner_dialog)->vbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);
  jukeboxowner_entry = gtk_entry_new();
  ownerstring = jukebox_get_ownerstring();
  if (ownerstring != NULL)
    gtk_entry_set_text(GTK_ENTRY(jukeboxowner_entry), ownerstring);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(jukeboxowner_dialog)->vbox), jukeboxowner_entry, TRUE, TRUE, 0);
  gtk_widget_show(jukeboxowner_entry);
  gtk_widget_show(jukeboxowner_dialog);
#endif
}


static void information_dialog(gpointer data)
{
  GtkWidget *dialog;

  if (jukebox_locked) {
    create_error_dialog(_("Cannot view information! Jukebox is busy!"));
    return;
  }

  dialog = jukebox_get_deviceinfo_dialog();
  if (dialog != NULL) {
    gtk_widget_show(dialog);  
  } else {
    create_error_dialog(_("Cannot retrieve jukebox information!"));
  }
}

#if defined(HAVE_DBUSGLIB) && defined(HAVE_HAL)
static void
hal_device_added (LibHalContext *context, const char *udi)
{
  int vendor_id;
  int product_id;
  char *protocol;

  vendor_id = libhal_device_get_property_int (context, udi, "usb.vendor_id", NULL);
  product_id = libhal_device_get_property_int (context, udi, "usb.product_id", NULL);
  protocol = libhal_device_get_property_string (context, udi, "portable_audio_player.type", NULL);
  if (protocol == NULL) {
    protocol = libhal_device_get_property_string (context, udi, "portable_audio_player.libnjb.protocol", NULL);
  }

  if (protocol != NULL) {
    g_print("Detected connected device VID=0x%04x/PID=0x%04x, protocol: %s\n", vendor_id, product_id, protocol);
    g_print("UDI: %s\n", udi);
    if (!strcmp(protocol, "mtp") || !strcmp(protocol, "pde") || !strcmp(protocol, "oasis")) {
      // If we haven't yet selected a jukebox, take this one!
      if (!jukebox_selected()) {
	global_udistr = strdup(udi);
	scan_jukebox(NULL);
      }
    }
  }

  libhal_free_string(protocol);
}

static void
hal_device_removed (LibHalContext *context, const char *udi)
{
  int vendor_id;
  int product_id;

  vendor_id = libhal_device_get_property_int (context, udi, "usb.vendor_id", NULL);
  product_id = libhal_device_get_property_int (context, udi, "usb.product_id", NULL);

  if (global_udistr != NULL && !strcmp(udi, global_udistr)) {
    g_print("Device VID=0x%04x/PID=0x%04x detached.\n", vendor_id, product_id);
    g_print("UDI: %s\n", udi);
    jukebox_release();
    free(global_udistr);
    global_udistr = NULL;
  }
}
#endif

/***********************************************************************************/
/* And this is where everything starts, everything graphic is initialized etc      */
/***********************************************************************************/
int main(int argc,
	 char *argv[])
{
  const gchar cProgramName[]= PACKAGE " " VERSION;
#if !GTK_CHECK_VERSION(2,6,0)
  GnomeProgram *gnomad;
#endif
  /* For the menus */
  GtkWidget *menu;
  GtkWidget *menu_bar;
  GtkWidget *root_menu;
  GtkWidget *menu_item;
  GtkWidget *music_panel;
  GtkWidget *data_panel;

  GtkWidget *vbox, *prefshbox;
  // GtkWidget *hbox;
  GtkWidget *notebook;
  GtkWidget *label;
  GtkAccelGroup *accel_group;
  extern char *optarg;
  gint opt;
  GError *icon_load_error = NULL;

  /* Binds for internationalization (i18n) */
  bindtextdomain (GETTEXT_PACKAGE, GNOMADLOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  g_thread_init(NULL);

#if GTK_CHECK_VERSION(2,6,0)
  gtk_init(&argc, &argv);
#else
  gnomad = gnome_program_init(PACKAGE, VERSION, LIBGNOMEUI_MODULE, argc, argv, 
			      GNOME_PARAM_POPT_TABLE, NULL, 
			      GNOME_PROGRAM_STANDARD_PROPERTIES, NULL);
#endif
  /* Parse extra arguments */
  gnomad_debug = 0;
  while ((opt = getopt(argc, argv, "D:")) != -1 ) {
    switch (opt) {
    case 'D':
      gnomad_debug = atoi(optarg);
      break;
    default:
      break;
    }
  }
  
#if GTK_CHECK_VERSION(2,6,0)
  main_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (main_window), cProgramName);
  icon_pixbuf = gdk_pixbuf_new_from_file(PIXMAPSDIR "/gnomad2-logo.png", &icon_load_error);
  if (icon_pixbuf != NULL) {
    gtk_window_set_default_icon(icon_pixbuf);
  }
#else
  main_window = gnome_app_new(PACKAGE, cProgramName);
#endif
  gtk_window_set_wmclass(GTK_WINDOW(main_window), PACKAGE, "main");
  gtk_widget_set_usize(GTK_WIDGET(main_window), 640, 480);
  
  g_signal_connect(GTK_OBJECT(main_window), 
		   "delete_event",
		   G_CALLBACK(delete),
		   NULL);

  g_signal_connect(GTK_OBJECT(main_window), 
		   "expose_event",
		   G_CALLBACK(expose),
		   NULL);
  
  gtk_container_set_border_width (GTK_CONTAINER (main_window), 0);

  /* Create metadata list stores */
  create_list_stores();

  /* Create a GtkAccelGroup and add it to the window. */
  accel_group = gtk_accel_group_new();
  gtk_window_add_accel_group (GTK_WINDOW (main_window), accel_group);

  /* Stuff everything into a vertical box */
  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), 0);
#if GTK_CHECK_VERSION(2,6,0)
  gtk_container_add(GTK_CONTAINER(main_window), vbox);
#else
  gnome_app_set_contents(GNOME_APP(main_window), vbox);
#endif
  gtk_widget_show(vbox);

  /* Create the menu bar */
  menu_bar = gtk_menu_bar_new ();
  gtk_widget_show (menu_bar);
  gtk_box_pack_start (GTK_BOX (vbox), menu_bar, FALSE, FALSE, 0);

  menu = gtk_menu_new (); // Shall not be shown!

  menu_item = gtk_menu_item_new_with_label(_("Quit"));
  gtk_menu_shell_append (GTK_MENU_SHELL(menu), menu_item);
  gtk_widget_add_accelerator (menu_item, "activate", accel_group,
                              GDK_q, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
  g_signal_connect(GTK_OBJECT(menu_item), 
		   "activate",
		   G_CALLBACK(delete),
		   NULL);
  gtk_widget_show (menu_item);
  root_menu = gtk_menu_item_new_with_label (_("File"));
  gtk_widget_show (root_menu);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
  gtk_menu_shell_append (GTK_MENU_SHELL(menu_bar), root_menu);

  /* Create a new notebook, place the position of the tabs */
  notebook = gtk_notebook_new ();
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
  gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);
  gtk_widget_show(notebook);

  /***********************************************************************************/
  /* Prefs tab                                                                       */
  /***********************************************************************************/
  
  prefshbox = gtk_hbox_new(FALSE, 0);
  gtk_widget_show (prefshbox);
  /* Put preferences in the hbox */
  create_prefs_widgets(prefshbox);
  // gtk_box_pack_start(GTK_BOX(hbox), transfer_widgets.playlistmenu, TRUE, TRUE, 0);
  
  /***********************************************************************************/
  /* Music transfer tab                                                              */
  /***********************************************************************************/

  music_panel = create_xfer_widgets();
  label = gtk_label_new (_("Music transfer"));
  gtk_notebook_append_page (GTK_NOTEBOOK(notebook), music_panel, label);

  /***********************************************************************************/
  /* Playlists tab                                                                   */
  /***********************************************************************************/

  /* This is an option menu wih playlist that is filled in 
   * by the scanning function */
  
  vbox=gtk_vbox_new(FALSE, 0);
  create_playlist_widgets(vbox);
  gtk_widget_show(vbox);
  label = gtk_label_new (_("Playlists"));
  gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

  /***********************************************************************************/
  /* Data file transfer tab                                                          */
  /***********************************************************************************/

  data_panel = create_data_widgets();
  label = gtk_label_new (_("Data transfer"));
  gtk_notebook_append_page (GTK_NOTEBOOK(notebook), data_panel, label);
  
  /* Add prefs tab */

  label = gtk_label_new (_("Preferences"));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), prefshbox, label);
  
  /* Set what page to start at (page 4) */
  gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), 0);

  menu = gtk_menu_new (); // Shall not be shown!
  menu_item = gtk_menu_item_new_with_label (_("Rescan contents"));
  gtk_menu_shell_append (GTK_MENU_SHELL(menu), menu_item);
  gtk_widget_add_accelerator (menu_item, "activate", accel_group,
                              GDK_r, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
  g_signal_connect(GTK_OBJECT (menu_item),
		   "activate",
		   G_CALLBACK(scan_jukebox),
		   NULL);
  gtk_widget_show(menu_item);
  menu_item = gtk_menu_item_new_with_label(_("Set owner string"));
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
  g_signal_connect(GTK_OBJECT (menu_item),
		   "activate",
		   G_CALLBACK(set_jukeboxowner_dialog),
		   NULL);
  gtk_widget_show (menu_item);
  menu_item = gtk_menu_item_new_with_label (_("Jukebox information"));
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
  gtk_widget_add_accelerator (menu_item, "activate", accel_group,
                              GDK_i, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
  g_signal_connect(GTK_OBJECT(menu_item), 
		   "activate",
		   G_CALLBACK(information_dialog),
		   NULL);
  gtk_widget_show (menu_item);
  root_menu = gtk_menu_item_new_with_label (_("Jukebox library"));
  gtk_widget_show (root_menu);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
  gtk_menu_shell_append(GTK_MENU_SHELL(menu_bar), root_menu);

  menu = gtk_menu_new(); // Shall not be shown!
  menu_item = gtk_menu_item_new_with_label (_("About"));
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
  g_signal_connect(GTK_OBJECT (menu_item), 
		   "activate",
		   G_CALLBACK(about_box),
		   NULL);
  gtk_widget_show (menu_item);
  root_menu = gtk_menu_item_new_with_label (_("Help"));
  gtk_widget_show (root_menu);
  gtk_menu_item_set_right_justified(GTK_MENU_ITEM(root_menu), TRUE);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(root_menu), menu);
  gtk_menu_shell_append(GTK_MENU_SHELL(menu_bar), root_menu);

#ifdef HAVE_DBUSGLIB
  // Connect to the SESSION bus to broadcast events and stuff.
  dbus_error_init(&dbus_error);
  dbus_session_conn = dbus_bus_get(DBUS_BUS_SESSION, &dbus_error);
  if (dbus_session_conn == NULL) {
    g_printerr ("Failed to open connection to D-BUS SESSION bus\n");
  } else {
    DBusMessage *message;

    dbus_connection_setup_with_g_main(dbus_session_conn, NULL);
    message = dbus_message_new_signal ("/net/sourceforge/gnomad2",
				       "net.sourceforge.gnomad2", 
				       "DeviceStatus");
    // Append the string "Online" to the signal
    //dbus_message_append_args(message,DBUS_TYPE_STRING,"Online",DBUS_TYPE_INVALID);
    dbus_connection_send(dbus_session_conn, message, NULL);
    dbus_message_unref(message);
  }
  dbus_error_free(&dbus_error);

#endif

#if defined(HAVE_DBUSGLIB) && defined(HAVE_HAL)
  // Connect to the SYSTEM bus to detect devices and stuff.
  dbus_error_init(&dbus_error);
  dbus_system_conn = dbus_bus_get(DBUS_BUS_SYSTEM, &dbus_error);
  if (dbus_system_conn == NULL) {
    g_printerr ("Failed to open connection to D-BUS SYSTEM bus\n");
  } else {
    dbus_connection_setup_with_g_main(dbus_system_conn, NULL);
    // dbus_system_proxy = dbus_g_proxy_new_for_name (dbus_system_conn,
    //					   "org.freedesktop.Hal",
    //					   "/",
    //					   "org.freedesktop.Hal.Device");
    // g_object_unref (dbus_system_proxy);
    hal_context = libhal_ctx_new ();
    if (hal_context == NULL) {
      g_print("Couldn't create HAL context!\n");
    } else {
      //libhal_ctx_set_user_data (hal_context, (void *) NULL);
      libhal_ctx_set_dbus_connection (hal_context, dbus_system_conn);
      libhal_ctx_set_device_added (hal_context, hal_device_added);
      libhal_ctx_set_device_removed (hal_context, hal_device_removed);
      libhal_device_property_watch_all (hal_context, &dbus_error);
      if (!libhal_ctx_init (hal_context, &dbus_error)) {
	g_print("error: libhal_ctx_init: %s: %s\n", dbus_error.name, dbus_error.message);
      }
    }
  }
  dbus_error_free(&dbus_error);
#endif
  
  gtk_widget_show(main_window);

  gtk_main();
  return 0;
}
