/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  File-Roller
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>

#include <gnome.h>
#include <glade/glade.h>

#include "bookmarks.h"
#include "file-utils.h"
#include "window.h"
#include "typedefs.h"


#define GLADE_FILE "file_roller.glade"


typedef struct {
	FRWindow *   window;
	GladeXML *   gui;

	GtkWidget *  dialog;
	GtkWidget *  e_extract_to_fileentry;
	GtkWidget *  e_extract_to_entry;

	GtkWidget *  e_fav_clist;
	GtkWidget *  e_add_fav_button;
	GtkWidget *  e_remove_fav_button;

	GtkWidget *  e_all_radiobutton;
	GtkWidget *  e_selected_radiobutton;
	GtkWidget *  e_files_radiobutton;
	GtkWidget *  e_files_entry;
	GtkWidget *  e_recreate_dir_checkbutton;
	GtkWidget *  e_overwrite_checkbutton;
	GtkWidget *  e_not_newer_checkbutton;

	gboolean     extract_clicked;
} DialogData;


/* called when the main dialog is closed. */
static void
destroy_cb (GtkWidget *widget,
            DialogData *data)
{
	if (! data->extract_clicked)
		window_batch_mode_stop (data->window);

        gtk_object_unref (GTK_OBJECT (data->gui));
        g_free (data);
}


/* called when the "ok" button is pressed. */
static void
ok_clicked_cb (GtkWidget *widget, 
	       DialogData *data)
{
	char *extract_to_dir;
	gboolean overwrite;
	gboolean skip_newer;
	gboolean selected_files;
	gboolean pattern_files;
	gboolean junk_paths;
	GList *file_list;
	FRWindow *window = data->window;

	data->extract_clicked = TRUE;

	/* collect extraction options. */

	extract_to_dir = gnome_file_entry_get_full_path (GNOME_FILE_ENTRY (data->e_extract_to_fileentry), FALSE);

	overwrite = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->e_overwrite_checkbutton));
	skip_newer = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->e_not_newer_checkbutton));
	junk_paths = ! gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->e_recreate_dir_checkbutton));

	selected_files = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->e_selected_radiobutton));
	pattern_files = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->e_files_radiobutton));

	/* create the file list. */

	file_list = NULL;
	if (selected_files) 
		file_list = window_get_file_list_selection (window, TRUE, NULL);
	else if (pattern_files) {
		gchar *pattern;
		pattern = gtk_entry_get_text (GTK_ENTRY (data->e_files_entry));
		file_list = window_get_file_list_pattern (window, pattern);
	}

	/* close the dialog. */

	gtk_widget_destroy (data->dialog);

	/* extract ! */

	window_archive_extract (window,
				file_list,
				extract_to_dir,
				skip_newer,
				overwrite,
				junk_paths);
	
	path_list_free (file_list);
	g_free (extract_to_dir);
}


static void
files_entry_changed_cb (GtkWidget *widget, 
			DialogData *data)
{
	if (! GTK_TOGGLE_BUTTON (data->e_files_radiobutton)->active)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->e_files_radiobutton), TRUE);
}


static void
path_entry_changed (GtkWidget *widget, 
		    DialogData *data)
{
	char *path;
	gboolean can_add;

	path = gtk_entry_get_text (GTK_ENTRY (data->e_extract_to_entry));
	can_add = strlen (path) > 0;
	gtk_widget_set_sensitive (data->e_add_fav_button, can_add);
}


static void
update_bookmark_list (DialogData *data)
{
	Bookmarks *bookmarks;
	GList *scan;

	gtk_clist_freeze (GTK_CLIST (data->e_fav_clist));
	gtk_clist_clear (GTK_CLIST (data->e_fav_clist));

	bookmarks = bookmarks_new (RC_BOOKMARKS_FILE);
	bookmarks_load_from_disk (bookmarks);

	for (scan = bookmarks->list; scan; scan = scan->next) {
		char *buf[2];
		gint row;

		buf[0] = scan->data;
		buf[1] = NULL;

		row = gtk_clist_append (GTK_CLIST (data->e_fav_clist), buf);
		gtk_clist_set_row_data_full (GTK_CLIST (data->e_fav_clist), 
					     row,
					     g_strdup (scan->data),
					     g_free);
	}

	bookmarks_free (bookmarks);

	gtk_clist_thaw (GTK_CLIST (data->e_fav_clist));
}


/* called when the "add to favorites" button is pressed. */
static void
add_fav_cb (GtkWidget *widget, 
	    DialogData *data)
{
	Bookmarks *bookmarks;
	char *path;

	path = gtk_entry_get_text (GTK_ENTRY (data->e_extract_to_entry));
	if (path == NULL) 
		return;

	bookmarks = bookmarks_new (RC_BOOKMARKS_FILE);
	bookmarks_load_from_disk (bookmarks);
	bookmarks_add (bookmarks, path);
	bookmarks_write_to_disk (bookmarks);
	bookmarks_free (bookmarks);

	update_bookmark_list (data);
}


/* called when the "remove favorite" button is pressed. */
static void
remove_fav_cb (GtkWidget *widget, 
	       DialogData *data)
{
	Bookmarks *bookmarks;
	GList *scan;

	bookmarks = bookmarks_new (RC_BOOKMARKS_FILE);
	bookmarks_load_from_disk (bookmarks);

	scan = GTK_CLIST (data->e_fav_clist)->selection;
	for (; scan; scan = scan->next) {
		char *path;
		gint row;

		row = GPOINTER_TO_INT (scan->data);
		path = gtk_clist_get_row_data (GTK_CLIST (data->e_fav_clist),
					       row);
		bookmarks_remove (bookmarks, path);
	}

	bookmarks_write_to_disk (bookmarks);
	bookmarks_free (bookmarks);
	update_bookmark_list (data);
}


static int
fav_button_press_cb (GtkWidget *widget, 
		     GdkEventButton *event,
		     gpointer callback_data)
{
        DialogData *data = callback_data;
	gint row, col;

	if ((event->type == GDK_2BUTTON_PRESS) && (event->button == 1)) {
		char *path;

		if (!gtk_clist_get_selection_info (GTK_CLIST (widget),
						   event->x, 
						   event->y, 
						   &row, 
						   &col))
			return FALSE;
		
		path = gtk_clist_get_row_data (GTK_CLIST (widget), row);
		gtk_entry_set_text (GTK_ENTRY (data->e_extract_to_entry),
				    path);
		return TRUE;
	}

	return FALSE;
}


void
dlg_extract (GtkWidget *widget,
	     gpointer callback_data)
{
        DialogData *data;
	FRWindow *window = callback_data;
	GtkWidget *cancel_button;
	GtkWidget *ok_button;
	gchar *path;

        data = g_new (DialogData, 1);

        data->window = window;
	data->extract_clicked = FALSE;

	data->gui = glade_xml_new (GLADEDIR "/" GLADE_FILE , NULL);
	if (!data->gui) {
                g_warning ("Could not find " GLADE_FILE "\n");
                return;
        }

        /* Get the widgets. */

        data->dialog = glade_xml_get_widget (data->gui, "extract_dialog");
        data->e_extract_to_fileentry = glade_xml_get_widget (data->gui, "e_extract_to_fileentry");
        data->e_extract_to_entry = glade_xml_get_widget (data->gui, "e_extract_to_entry");

	data->e_fav_clist = glade_xml_get_widget (data->gui, "e_fav_clist");
	data->e_add_fav_button = glade_xml_get_widget (data->gui, "e_add_fav_button");
	data->e_remove_fav_button = glade_xml_get_widget (data->gui, "e_remove_fav_button");

        data->e_all_radiobutton = glade_xml_get_widget (data->gui, "e_all_radiobutton");
        data->e_selected_radiobutton = glade_xml_get_widget (data->gui, "e_selected_radiobutton");
        data->e_files_radiobutton = glade_xml_get_widget (data->gui, "e_files_radiobutton");
        data->e_files_entry = glade_xml_get_widget (data->gui, "e_files_entry");
        data->e_overwrite_checkbutton = glade_xml_get_widget (data->gui, "e_overwrite_checkbutton");
        data->e_not_newer_checkbutton = glade_xml_get_widget (data->gui, "e_not_newer_checkbutton");
        data->e_recreate_dir_checkbutton = glade_xml_get_widget (data->gui, "e_recreate_dir_checkbutton");

	ok_button = glade_xml_get_widget (data->gui, "e_ok_button");
	cancel_button = glade_xml_get_widget (data->gui, "e_cancel_button");

	/* Set widgets data. */

	path = g_strconcat (window->extract_default_dir, "/", NULL);
	gtk_entry_set_text (GTK_ENTRY (data->e_extract_to_entry), path);
	g_free (path);
	
	if (GTK_CLIST (data->window->clist)->selection == NULL)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->e_all_radiobutton), TRUE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->e_selected_radiobutton), TRUE);

	gtk_clist_set_auto_sort (GTK_CLIST (data->e_fav_clist), TRUE);
	update_bookmark_list (data);

	/* Set the signals handlers. */

        gtk_signal_connect (GTK_OBJECT (data->dialog), "destroy",
                            (GtkSignalFunc) destroy_cb,
                            data);
	gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
                                   GTK_SIGNAL_FUNC (gtk_widget_destroy),
                                   GTK_OBJECT (data->dialog));
	gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
			    GTK_SIGNAL_FUNC (ok_clicked_cb),
			    data);
	gtk_signal_connect (GTK_OBJECT (data->e_add_fav_button), "clicked",
			    GTK_SIGNAL_FUNC (add_fav_cb),
			    data);
	gtk_signal_connect (GTK_OBJECT (data->e_remove_fav_button), "clicked",
			    GTK_SIGNAL_FUNC (remove_fav_cb),
			    data);
	gtk_signal_connect (GTK_OBJECT (data->e_extract_to_entry), "changed",
			    GTK_SIGNAL_FUNC (path_entry_changed),
			    data);
	gtk_signal_connect (GTK_OBJECT (data->e_fav_clist), 
			    "button_press_event",
			    (GtkSignalFunc) fav_button_press_cb, 
			    data);
	gtk_signal_connect (GTK_OBJECT (data->e_files_entry), "changed",
			    GTK_SIGNAL_FUNC (files_entry_changed_cb),
			    data);
	/* Run dialog. */

        gtk_window_set_transient_for (GTK_WINDOW (data->dialog), 
				      GTK_WINDOW (window->app));
        gtk_window_set_modal         (GTK_WINDOW (data->dialog), TRUE);

	gtk_widget_show (data->dialog);
}
