
/*
 * LIB/SEQ.C	- Sequence space control
 *
 *	The sequence space is based on directories split into 10 minute
 *	segments.  The main diablo process updates the sequence space by
 *	scanning /news/spool/news and creates a new directory every 10
 *	minutes.
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

Prototype void InitSeqSpace(void);
Prototype uint32 ExpireToSeq(uint16 exp);
Prototype void LoadSlotDir(time_t t, int forceCreate);

typedef struct SeqSpace {
    uint32	ss_Beg;
    uint32	ss_End;
} SeqSpace;

SeqSpace	*SSeq;
uint32		LSTime;

void
InitSeqSpace(void)
{
#if USE_SPAM_SHM
    int sid = shmget(IPC_PRIVATE, sizeof(SeqSpace), SHM_R|SHM_W);
    struct shmid_ds ds;

    if (sid < 0) {
        syslog(LOG_CRIT, "sysv shared memory alloc failed");
        exit(1);
    }

    SSeq = (SeqSpace *)shmat(sid, NULL, SHM_R|SHM_W);

    if (shmctl(sid, IPC_STAT, &ds) < 0 || shmctl(sid, IPC_RMID, &ds) < 0) {
        syslog(LOG_CRIT, "sysv shmctl stat/rmid failed");
        exit(1);
    }
    if (SSeq == (SeqSpace *)-1) {
	SSeq = NULL;
        syslog(LOG_CRIT, "sysv shared memory map failed");
        exit(1);
    }
    bzero(SSeq, sizeof(SeqSpace));
#else
#error "lib/seq.c currently requires USE_SPAM_SHM to be set in lib/config.h or lib/vendor.h"
#endif
}

void
LoadSlotDir(time_t t, int forceCreate)
{
    uint32 gmt = t / 60;

    gmt = gmt - gmt % 10;

    if (LSTime == 0 || gmt != LSTime) {
	char path[256];

	/*
	 * create directory, fixup end sequence
	 *
	 * XXX ERROR 2030 BUG XXX gmt will rollover because t will rollover.
	 */

	snprintf(path, sizeof(path), "%s", PatExpand(SpoolHomePat));
	snprintf(path + strlen(path), sizeof(path) - strlen(path), "/D.%08x", gmt);
	mkdir(path, 0755);

	/*
	 * Kludge - check for 'hicups'.  We have to maintain the D.XXXXXXXX
	 * directories at 10 minute intervals, with diablo adding articles
	 * to the end (feeder expire) or anywhere (reader expire), and 
	 * dexpire removing whole directories from the beginning.
	 */
	if (forceCreate == 0) {
	    int cnt = ((int)(gmt - LSTime) / 10);
	    if (cnt > 1 && cnt < 64) {
		syslog(LOG_INFO, "SeqSpaceSkipped! %08x - %08x", LSTime, gmt);
		while (LSTime < gmt) {
		    snprintf(path, sizeof(path), "%s", PatExpand(SpoolHomePat));
		    snprintf(path + strlen(path), sizeof(path) - strlen(path), "/D.%08x", LSTime);
		    mkdir(path, 0755);
		    LSTime += 10;
		}
	    }
	}

	LSTime = gmt;
	SSeq->ss_End = gmt;

	/*
	 * Scan the spool to obtain the beginning sequence
	 */

	{
	    DIR *dir;

	    snprintf(path, sizeof(path), "%s", PatExpand(SpoolHomePat));
	    if ((dir = opendir(path)) != NULL) { 
		den_t *den;
		uint32 beg = gmt;

		while ((den = readdir(dir)) != NULL) { 
		    int32 v;

		    if (den->d_name[0] != 'D' || den->d_name[1] != '.')
			continue;
		    if (sscanf(den->d_name + 2, "%x", &v) == 1) {
			if ((uint32)v < beg)
			    beg = (uint32)v;
		    }
		}   
		closedir(dir);

		/*
		 * trim based on DiabloFeederMaxSpool.  ss_Beg/ss_End are in
		 * minutes, there is a 10 minute interval per directory.
		 */

		if ((SSeq->ss_End - beg) / 10 > DiabloFeederMaxSpool)
		    beg = SSeq->ss_End - (DiabloFeederMaxSpool * 10);

		SSeq->ss_Beg = beg;
		/*syslog(LOG_INFO, "SeqSpace: %08x - %08x", SSeq->ss_Beg, SSeq->ss_End);*/
	    }
	}
	/* SSeq->ss_End = gmt; */

	/*
	 * If diablo is restarted, we have to create an directories
	 * that might have gone missing between the sequence range.
	 * If we don't, we hit a sleep failsafe in the mainline diablo
	 * code and lose articles.
	 */

	if (forceCreate) {
	    uint32 sgmt = SSeq->ss_Beg - SSeq->ss_Beg % 10;

	    while (sgmt < SSeq->ss_End) {
		snprintf(path, sizeof(path), "%s", PatExpand(SpoolHomePat));
		snprintf(path + strlen(path), sizeof(path) - strlen(path), "/D.%08x", sgmt);
		mkdir(path, 0755);
		sgmt += 10;
	    }
	}
    }
}

uint32
ExpireToSeq(uint16 exp)
{
    uint32 slot;

    if (DiabloExpire == EXPIRE_FEEDER) {
	slot = SSeq->ss_End;
    } else {
	/* EXPIRE_READER */
	slot = SSeq->ss_Beg + (SSeq->ss_End - SSeq->ss_Beg) * (uint32)exp / 100;
    }
    slot = slot - slot % 10;	/* 10 minute boundries */
    return(slot);
}

