
/*
 * LIB/HOSTAUTH.C
 *
 * (c)Copyright 1997-1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution 
 *    for specific rights granted.
 *
 */

#include "defs.h"

Prototype int LoadHostAccess(time_t t, int force, int rebuildtime);
Prototype char *Authenticate(int fd, struct sockaddr_in *asin, char *label);

#define	TESTTYPE_NONE	0
#define	TESTTYPE_ANY	1
#define	TESTTYPE_IP	2
#define	TESTTYPE_FQDN	4

typedef struct HostAccess {
    char		entry[256];
    int			testtype;
    struct sockaddr_in 	ipaddr;
    int			iplen;
    char		label[256];
} HostAccess;

time_t HAGmtMin = (time_t)-1;
time_t HAMTime = 0;
time_t HACMTime = 0;

/*
 * LoadAccess() - [re]load diablo.hosts file
 *
 * This load is to create a cache file of the diablo.hosts file
 * that contains the IP addresses of all the hosts found. This
 * makes authentication a lot quicker and easier.
 */

int
LoadHostAccess(time_t t, int force, int rebuildtime)
{
    int timeChanged = 0;
    struct stat st = { 0 };
    pid_t pid;

    /*
     * check for diablo.hosts file modified once a minute
     */

    if (!force  && (t / 60 == HAGmtMin))
	return(0);

    timeChanged = 1;

    HAGmtMin = t / 60;

    /* Don't do this if diablo.hosts hasn't been modified  or we are
     * within the rebuild time limit
     */
    if (!force && HACMTime + rebuildtime > t &&
		stat(PatDbExpand(DHostsCachePat), &st) == 0 &&
		stat(PatLibExpand(DiabloHostsPat), &st) == 0 &&
		st.st_mtime == HAMTime)
	return(0);

    if (stat(PatLibExpand(DiabloHostsPat), &st) == 0)
	HAMTime = st.st_mtime;
     HACMTime = t;

    /*
     * Do the DNS lookups in the background
     * The main loop does a wait to cleanup the children
     */
    if ((pid = fork()) == 0) {
	FILE *fi;
	FILE *fo;
	char buf[512];
	HostAccess HA;
	struct hostent *he;
	int i;
	char *s;
	char *l;
	char *p;

	fi = fopen(PatLibExpand(DiabloHostsPat), "r");

	if (fi == NULL) {
	    syslog(LOG_EMERG, "%s: %s", PatLibExpand(DiabloHostsPat), strerror(errno));
	    exit(1);
	}

	snprintf(buf, sizeof(buf), "%s.new", PatDbExpand(DHostsCachePat));

	if (stat(buf, &st) == 0) {
	    syslog(LOG_INFO, "%s : Rebuild in progress", buf);
	    exit(0);
	}

	fo = fopen(buf, "w");
	if (fo == NULL) {
		syslog(LOG_EMERG, "%s: %s", buf, strerror(errno));
		exit(1);
	}

	while (fgets(buf, sizeof(buf), fi) != NULL) {

	    p = buf;
	    if (*p == '\n' || *p == '#')
		continue;
	    HA.testtype = TESTTYPE_NONE;
	    if (strncmp(p, "IP:", 3) == 0) {
		HA.testtype |= TESTTYPE_IP;
		p += 3;
	    } else if (strncmp(p, "FQDN:", 5) == 0) {
		HA.testtype |= TESTTYPE_FQDN;
		p += 5;
	    }
	    s = strtok(p, ": \t\r\n");	/* hostname	*/
	    if (s == NULL || !*s)
		continue;
            l = strtok(NULL, ": \t\r\n");	/* label	*/
	    if (s == NULL || !*l)
		continue;

	    if (strlen(s) > 255 || strlen(l) > 255)
		continue;

	    /*
	     * If we start with a digit, then don't match against domainname
	     * as it could be an IP only match
	     */
	    if ((strchr(s, '*') != NULL || strchr(s, '?') != NULL) && !isdigit(*s))
		HA.testtype |= TESTTYPE_ANY;

	    strcpy(HA.entry, s);
	    strcpy(HA.label, l);
	    memset(&HA.ipaddr, 0, sizeof(HA.ipaddr));
	    HA.iplen = 0;

	    if (strchr(s, '*') != NULL || strchr(s, '?') != NULL) {
		fwrite(&HA, sizeof(HA), 1, fo);
		continue;
	    }

	    if ((he = gethostbyname(s)) != NULL) 
		for (i = 0; he->h_addr_list[i] != NULL; ++i) {
		    memcpy(&HA.ipaddr, he->h_addr_list[i], he->h_length);
		    HA.iplen = he->h_length;
		    fwrite(&HA, sizeof(HA), 1, fo);
		}
	}
	fclose(fi);
	fclose(fo);

	/*
	 * Now rename the newly created cache to its correct name
	 */
	snprintf(buf, sizeof(buf), "%s.new", PatDbExpand(DHostsCachePat));
	rename(buf, PatDbExpand(DHostsCachePat));
	exit(0);
    }
    return(pid);
}

/*
 * AUTHENTICATION()	- authenticate a new connection
 *
 *	If the forward and reverse dns match, look for the
 *	host or ip in diablo.hosts.  If they do not, only
 *	look for the ip in diablo.hosts.
 */

char *
Authenticate(int fd, struct sockaddr_in *asin, char *label)
{
    char *hname = NULL;		/* authenticated hostname or NULL */
    char *pname = NULL;		/* IP address (as string)	  */
    int isSecure = 0;
    int isSpoof = 0;
    int isValid = 0;

    /*
     * check reverse lookup verses forward lookup, set either hname or uname
     * or neither, but never both.
     */

    {
	struct hostent *he;

	he = gethostbyaddr(
	    (char *)&asin->sin_addr,
	    sizeof(asin->sin_addr),
	    AF_INET
	);

	if (he != NULL) {
	    hname = zallocStr(&SysMemPool, he->h_name);
	    if ((he = gethostbyname(hname)) != NULL) {
		if (strcasecmp(he->h_name, hname) == 0) {
		    isSecure = 1;
		} else {
		    isSpoof = 1;
		}
	    }
	}
    }

    /*
     * set pname
     */
    pname = zallocStr(&SysMemPool, inet_ntoa(asin->sin_addr));

    /*
     * Check IP against cache file of diablo.hosts. 
     */
    {
	FILE *fi = fopen(PatDbExpand(DHostsCachePat), "r");

	if (fi) {
	    HostAccess HA;

	    while (fread(&HA, sizeof(HA), 1, fi) == 1) {
		if (HA.iplen && memcmp(&asin->sin_addr, &HA.ipaddr, HA.iplen) == 0) {
		    isValid = 1;
		    break;
		} else if (hname && HA.testtype & TESTTYPE_ANY && WildCaseCmp(HA.entry, hname) == 0) {
		    isValid = 1;
		    break;
		} else if (pname && WildCaseCmp(HA.entry, pname) == 0) {
		    isValid = 1;
		    break;
		} else if (hname && strcasecmp(HA.entry, hname) == 0) {
		    isValid = 1;
		    break;
		}
	    }
	    if (isValid) {
		/* syslog(LOG_INFO, "Matched %s (%s) to %s", hname, pname, HA.entry); */
		if (HA.label[0])
		    strcpy(label, HA.label);
		else
		    label[0] = 0;
	    }
	    fclose(fi);
	} else {
	    syslog(LOG_ERR, "%s file not found", PatDbExpand(DHostsCachePat));
	}
    }

    if (isValid) {
	/*
	 * If we have a valid connection, but we were
	 * unable to resolve the FQDN, we make the FQDN
	 * the IP address.
	 */
	if (hname == NULL) {
	    hname = pname;
	    pname = NULL;
	}
    } else {
	if (isSpoof)
	    syslog(LOG_EMERG, "DNS failure or spoof from %s", pname);
        zfreeStr(&SysMemPool, &hname);
	hname = NULL;
    }
    zfreeStr(&SysMemPool, &pname);
    return(hname);
}

