
/*
 * POST.C
 *
 *	Post an article.  The article is posted to one or several of the
 *	upstream servers depending on prioritization and the existence of
 *	the 'P'ost flag.  
 *
 *	The article is loaded into memory prior to posting.  If no servers
 *	are available, the article is written to a local posting queue and
 *	the post is retried later.
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

Prototype void NNPostBuffer(Connection *conn);
Prototype void NNPostCommand1(Connection *conn);

void NNPostResponse1(Connection *conn);
void NNPostArticle1(Connection *conn);
void NNPostResponse2(Connection *conn);
int MailIfModerated(char *groups, Connection *conn);
int groupAlready(int v, int *save, int saveCount);
void MailToModerator(const char *group, Connection *conn, const char *modMail);
int CheckGroupModeration(const char *group);
void GenerateDistribution(MBufHead *mbuf, char *newsGrps);;

/*
 * NNPostBuffer() - called when client connection post buffer contains the
 *		    entire article.  We need to check the article headers
 *		    and post it as appropriate.  If we are unable to post it
 *		    immediately, we queue it.
 *
 * STEPS:
 *	(1) remove headers otherwise generated internally (for security)
 *	(2) add internally generated headers
 *	
 * Internally generated headers:
 *	Path:			(removed, generated)
 *	Date:			(removed, generated)
 *	Lines:			(removed, generated)
 *	Message-ID:		(checked, generated)
 *	NNTP-Posting-Host:	(removed, generated)
 *	X-Trace:		(removed, generated)
 *	Xref:			(removed, RIP)
 *	Distribution:		(checked, generated)
 *
 * Errors:
 *	441 Article has no body -- just headers
 *	441 Required "XXX" header is missing (Subject, From, Newsgroups)
 *	441 435 Bad Message-ID
 */

void
NNPostBuffer(Connection *conn)
{
    int artLen;
    int haveMsgId = 0;
    int haveSubject = 0;
    int haveFrom = 0;
    int haveDate = 0;
    int haveApproved = 0;
    int haveNewsgroups = 0;
    int haveDist = 0;
    const char *errMsg = NULL;
    char *newsGrps = NULL;
    char *base = MBNormalize(&conn->co_ArtBuf, &artLen);
    char *msgid = NULL;
    int	msgidAllocLen = 0;
    MBufHead	tmbuf;

    MBInit(&tmbuf, -1, &conn->co_MemPool, &conn->co_BufPool);

    /*
     * Output Path: line
     */

    MBPrintf(&tmbuf, "Path: %s%snot-for-mail\r\n",
	PathHost,
	(PathHost[0] ? "!" : "")
    );

    /*
     * Scan Headers, check for required headers
     */

    while (artLen > 1 && !(base[0] == '\n') && !(base[0] == '\r' && base[1] == '\n')) {
	int l;
	int keep = 1;

	/*
	 * extract header.  Headers can be multi-line so deal with that.
	 */

	for (l = 0; l < artLen; ++l) {
	    if (base[l] == '\r' && 
		base[l+1] == '\n' &&
		(l + 2 >= artLen || (base[l+2] != ' ' && base[l+2] != '\t'))
	    ) {
		l += 2;
		break;
	    }
	}

	/*
	 * check against specials
	 */
	if (strncasecmp(base, "Subject:", 8) == 0) {
	    haveSubject = 1;
	} else if (strncasecmp(base, "From:", 5) == 0) {
	    haveFrom = 1;
	} else if (strncasecmp(base, "Approved:", 9) == 0) {
	    haveApproved = 1;
	} else if (strncasecmp(base, "Newsgroups:", 11) == 0) {
	    haveNewsgroups = 1;
	    newsGrps = zallocStrTrim(&conn->co_MemPool, base + 11, l - 11);
	} else if (strncasecmp(base, "Date:", 5) == 0) {
	    haveDate = 1;
	} else if (strncasecmp(base, "Distribution:", 13) == 0) {
	    /*
	     * Only keep the header and set haveDist if it has something in 
	     * it, otherwise we (potentially) override it.
	     */
	    {
		char *ptr = zallocStrTrim(&conn->co_MemPool, base+13, l - 13);
		if (ptr && *ptr)
		    haveDist = 1;
		zfreeStr(&conn->co_MemPool, &ptr);
	    }
	    if (haveDist == 0)
		keep = 0;
	} else if (strncasecmp(base, "Path:", 5) == 0) {
	    keep = 0;	/* delete */
	} else if (strncasecmp(base, "Date:", 5) == 0) {
	    keep = 0;	/* delete */
	} else if (strncasecmp(base, "Lines:", 6) == 0) {
	    keep = 0;	/* delete */
	} else if (strncasecmp(base, "Message-ID:", 11) == 0) {
	    int save = base[l-2];
	    const char *msgmsgid;

	    base[l-2] = 0;
	    if (strcmp((msgmsgid = MsgId(base + 11)), "<>") == 0 ||
		strchr(base, '@') == NULL
	    ) {
		keep = 0;	/* delete */
		errMsg = "441 435 Bad Message-ID\r\n";
		msgid = NULL;
	    } else {
		keep = 1;
		haveMsgId = 1;
		msgidAllocLen = strlen(msgmsgid) + 1;
		msgid = strcpy(zalloc(&conn->co_MemPool, msgidAllocLen), msgmsgid);
	    }
	    base[l-2] = save;
	} else if (strncasecmp(base, "NNTP-Posting-Host:", 18) == 0) {
	    keep = 0;	/* delete */
	} else if (strncasecmp(base, "X-Trace:", 8) == 0) {
	    keep = 0;	/* delete */
	} else if (strncasecmp(base, "Xref:", 5) == 0) {
	    keep = 0;	/* delete */
	}
	if (keep)
	    MBWrite(&tmbuf, base, l);
	base += l;
	artLen -= l;
    }

    /*
     * Check errors
     */

    if (haveNewsgroups == 0)
	errMsg = "441 Required \"Newsgroups\" header is missing\r\n";
    if (haveFrom == 0)
	errMsg = "441 Required \"From\" header is missing\r\n";
    if (haveSubject == 0)
	errMsg = "441 Required \"Subject\" header is missing\r\n";
    if (artLen <= 2)
	errMsg = "441 Article has no body -- just headers\r\n";

    /*
     * If no error, add our own headers
     */
    if (errMsg == NULL) {
	time_t t = time(NULL);

	/*
	 * Add Date:
	 */
	if (haveDate == 0) {
	    struct tm *tp = gmtime(&t);
	    char buf[64];

	    strftime(buf, sizeof(buf), "%d %b %Y %H:%M:%S GMT", tp);
	    MBPrintf(&tmbuf, "Date: %s\r\n", buf);
	}
	/*
	 * Add Lines:
	 */
	{
	    int lines = 0;
	    int i;

	    for (i = 2; i < artLen; ++i) {
		if (base[i] == '\n')
		    ++lines;
	    }
	    if (base[i-1] != '\n')	/* last line not terminated? */
		++lines;
	    MBPrintf(&tmbuf, "Lines: %d\r\n", lines);
	}
	/*
	 * Add Message-ID:
	 */
	if (haveMsgId == 0) {
	    static time_t LastT;
	    static int MsgIdCounter;

	    if (LastT != t) {
		LastT = t;
		MsgIdCounter = 0;
	    }
	    msgidAllocLen = strlen(ReportedHostName) + 40;
	    msgid = zalloc(&conn->co_MemPool, msgidAllocLen);

	    sprintf(msgid, "<%08lx$%d$%d@%s>",
		(long)t,
		MsgIdCounter,
		(int)getpid(),
		ReportedHostName
	    );
	    MBPrintf(&tmbuf, "Message-ID: %s\r\n", msgid);
	    ++MsgIdCounter;
	}
	/*
	 * Add Distribution:
	 */
	if (haveDist == 0 && newsGrps) {
	    GenerateDistribution(&tmbuf, newsGrps);
	}

	/*
	 * Add NNTP-Posting-Host:
	 */
	MBPrintf(&tmbuf, "NNTP-Posting-Host: %s\r\n", conn->co_Auth.dr_Host);

	/*
	 * Add X-Trace:
	 */
	MBPrintf(&tmbuf, "X-Trace: %s %lu %d %s%s%s\r\n",
	    (PathHost[0] ? PathHost : ReportedHostName),
	    (unsigned long)t,
	    (int)getpid(),		/* should probably be slot id */
	    (conn->co_Auth.dr_User[0] ? conn->co_Auth.dr_User : ""),
	    (conn->co_Auth.dr_User[0] ? "@" : ""),
	    inet_ntoa(conn->co_Auth.dr_Addr)
	);
    }

    /*
     * blank line and article body
     */

    if (errMsg == NULL) {
	MBWrite(&tmbuf, base, artLen);
    }
    MBFree(&conn->co_ArtBuf);

    /*
     * If no error, pump out to ready server or queue for transmission, 
     * else write the error out and throw away.
     *
     * When POSTing to a moderated group, the article is mailed to the
     * moderator and NOT otherwise posted, even if crossposted to non-moderated
     * groups.
     */

    if (errMsg == NULL) {
	/*
	 * Copy into co_ArtBuf for transmission/mailing
	 */
	MBCopy(&tmbuf, &conn->co_ArtBuf);
	MBFree(&tmbuf);

	syslog(
	    LOG_NOTICE,
	    "%s%s%s post ok %s", 
	    (conn->co_Auth.dr_User[0] ? conn->co_Auth.dr_User : ""),
	    (conn->co_Auth.dr_User[0] ? "@" : ""),
	    inet_ntoa(conn->co_Auth.dr_Addr),
	    msgid
	);

	if (newsGrps && 
	    haveApproved == 0 && 
	    MailIfModerated(newsGrps, conn) == 0
	) {
	    MBFree(&conn->co_ArtBuf);
	    MBPrintf(&conn->co_TMBuf, "240 Article mailed to moderator\r\n");
	    NNCommand(conn);
	} else {
	    NNServerRequest(conn, NULL, msgid, NULL, SREQ_POST);
	}
    } else {
	MBWrite(&conn->co_TMBuf, errMsg, strlen(errMsg));
	MBFree(&tmbuf);
	NNCommand(conn);
    }
    if (newsGrps)
	zfreeStr(&conn->co_MemPool, &newsGrps);
    if (msgid)
	zfree(&conn->co_MemPool, msgid, msgidAllocLen);
}

/*
 * NNPostCommand1() - called when article has been queued to a server 
 *		      connection, in the server's context, to post the
 *		      article.
 *
 * SEQUENCE:
 */

void
NNPostCommand1(Connection *conn)
{
    ServReq *sreq = conn->co_SReq;

    MBPrintf(&conn->co_TMBuf, "ihave %s\r\n", sreq->sr_MsgId);
    NNPostResponse1(conn);
}

void
NNPostResponse1(Connection *conn)
{
    /*ServReq *sreq = conn->co_SReq;*/
    char *buf;
    int len;

    conn->co_Func = NNPostResponse1;
    conn->co_State = "pores1";

    if ((len = MBReadLine(&conn->co_RMBuf, &buf)) > 0) {
	if (strtol(buf, NULL, 10) == 335) {
	    NNPostArticle1(conn);
	} else {
	    char errMsg[256];

	    MBFree(&conn->co_ArtBuf);

	    if (len >= 1 && buf[len-1] == 0)
		--len;
	    if (len >= 1 && buf[len-1] == '\r')
		--len;
	    if (len > sizeof(errMsg) - 32)
		len = sizeof(errMsg) - 32;
	    strcpy(errMsg, "441 ");
	    bcopy(buf, errMsg + 4, len);
	    errMsg[4+len] = '\r';
	    errMsg[5+len] = '\n';
	    errMsg[6+len] = 0;
	    NNFinishSReq(conn, errMsg, 0);
	}
    } else if (len < 0) {
	NNServerTerminate(conn);
    } /* else we haven't gotten the reponse yet */
}

void
NNPostArticle1(Connection *conn)
{
    ServReq *sreq = conn->co_SReq;

    conn->co_Func = NNPostArticle1;
    conn->co_State = "poart1";

    while (conn->co_TMBuf.mh_Bytes < 800) {
	char *buf;
	int len;

	if ((len = MBReadLine(&sreq->sr_CConn->co_ArtBuf, &buf)) > 0) {
	    buf[len-1] = '\n';
	    MBWrite(&conn->co_TMBuf, buf, len);
	} else if (len <= 0) {
	    MBPrintf(&conn->co_TMBuf, ".\r\n");
	    NNPostResponse2(conn);
	    return;
	}
    }
}

void
NNPostResponse2(Connection *conn)
{
    /*ServReq *sreq = conn->co_SReq;*/
    char *buf;
    int len;

    conn->co_Func = NNPostResponse2;
    conn->co_State = "pores2";

    if ((len = MBReadLine(&conn->co_RMBuf, &buf)) > 0) {
	if (strtol(buf, NULL, 10) == 235) {
	    NNFinishSReq(conn, "240 Article posted\r\n", 0);
	} else {
	    char errMsg[256];

	    MBFree(&conn->co_ArtBuf);

	    if (len >= 1 && buf[len-1] == 0)
		--len;
	    if (len >= 1 && buf[len-1] == '\r')
		--len;
	    if (len > sizeof(errMsg) - 32)
		len = sizeof(errMsg) - 32;
	    strcpy(errMsg, "441 ");
	    bcopy(buf, errMsg + 4, len);
	    errMsg[4+len] = '\r';
	    errMsg[5+len] = '\n';
	    errMsg[6+len] = 0;
	    NNFinishSReq(conn, errMsg, 0);
	}
    } else if (len < 0) {
	NNServerTerminate(conn);
    } /* else we haven't gotten the reponse yet */
}

/*
 * MailIfModerated() -	if posting to moderated groups and the article
 *			was not approved, mail the message to the moderators 
 *			instead
 *
 *	Return 0 if article was mailed, -1 otherwise
 *
 */

int
MailIfModerated(char *groups, Connection *conn)
{
    FILE *fi;
    int r = -1;
    int saveCount = 0;
    char buf[256];
    int save[256];

    if ((fi = fopen(PatLibExpand(ModeratorsPat), "r")) != NULL) {
	while (fgets(buf, sizeof(buf), fi) != NULL) {
	    char *modGroupWild = strtok(buf, ": \t\r\n");
	    char *modMail = (modGroupWild) ? strtok(NULL, ": \t\r\n") : NULL;
	    int i = 0;

	    if (modGroupWild == NULL || modMail == NULL)
		continue;

	    while (groups[i]) {
		int j;
		char c;

		for (j = i; groups[j] && groups[j] != ','; ++j)
		    ;
		c = groups[j];
		groups[j] = 0;

		if (i != j && 
		    groupAlready(i, save, saveCount) < 0 &&
		    saveCount != arysize(save)
		) {
		    if (CheckGroupModeration(groups + i) < 0)
			save[saveCount++] = i;
		}

		if (i != j && 
		    groupAlready(i, save, saveCount) < 0 &&
		    saveCount != arysize(save) &&
		    WildCmp(modGroupWild, groups + i
		) == 0) {
		    r = 0;
		    MailToModerator(groups + i, conn, modMail);
		    save[saveCount++] = i;
		}
		i = j;
		if ((groups[j] = c) == ',')
		    ++i;
	    }
	}
	fclose(fi);
    }
    return(r);
}

int
groupAlready(int v, int *save, int saveCount)
{
    int i;
    int r = -1;

    for (i = 0; i < saveCount; ++i) {
	if (save[i] == v) {
	    r = 0;
	    break;
	}
    }
    return(r);
}

/*
 * MailToModerator() -  send mail to the moderator.  
 *
 *	Note: we have to unescape the posted article and we cannot pass
 *	certain headers.  We cannot pass Path: because the moderator may
 *	forward it, causing the posting news machine to miss the approved
 *	posting made by the moderator (because it will have locked itself
 *	out due to the Path: ).  We also do not pass To:, Cc:, or Bcc:
 *	headers for two reasons:  First, because the news client is 
 *	supposed to handle those fields and we don't parse them for
 *	non-moderated postings, and second, for security reasons.
 */

void
MailToModerator(const char *group, Connection *conn, const char *modMail)
{
    char mailDest[256];
    pid_t pid;
    int fds[3];
    char *argv[5] = { SENDMAIL_PATH, SENDMAIL_ARG0, "-t", "-i", NULL };

    {
	char *mgroup = zallocStr(&conn->co_MemPool, group);
	int i;

	for (i = 0; mgroup[i]; ++i) {
	    if (mgroup[i] == '.')			/* dots to dashes */
		mgroup[i] = '-';
	    if (mgroup[i] == '@' || mgroup[i] == ',')	/* shouldn't happen */
		mgroup[i] = '-';
	}
	snprintf(mailDest, sizeof(mailDest), modMail, mgroup);
	zfreeStr(&conn->co_MemPool, &mgroup);
    }

    if ((pid = RunProgramPipe(fds, RPF_STDOUT, argv)) > 0) {
	FILE *fo;

	if ((fo = fdopen(fds[1], "w")) != NULL) {
	    int artLen = 0;
	    int lastLineOk = 1;
	    int inHeaders = 1;
	    int b = 0;
	    char *base = MBNormalize(&conn->co_ArtBuf, &artLen);

	    fprintf(fo, "To: %s\n", mailDest);

	    while (b < artLen) {
		int i;
		int l;
		int lineOk = 1;

		for (i = b; i < artLen && base[i] != '\n'; ++i)
		    ;
		if (i < artLen && base[i] == '\n')
		    ++i;

		/*
		 * Length of line including CR+LF
		 */
		l = i - b;

		/*
		 * Remove CR+LF or LF (LF will be added back in the email)
		 */
		if (l > 0 && base[i-1] == '\n') {
		    --l;
		    if (l > 1 && base[i-2] == '\r')
			--l;
		}

		/*
		 * Get rid of leading dot if dot-escaping
		 */
		if (l > 0 && base[b] == '.') {
		    --l;
		    ++b;
		}

		/*
		 * Process headers, output line.   The lastLineOk stuff is
		 * to handle header line continuation.
		 */

		if (inHeaders) {
		    if (l == 0) {
			inHeaders = 0;
		    } else if (base[b] == ' ' || base[b] == '\t') {
			lineOk = lastLineOk;
		    } else if (l >= 3 && !strncasecmp(base + b, "To:", 3)) {
			lineOk = 0;
		    } else if (l >= 3 && !strncasecmp(base + b, "Cc:", 3)) {
			lineOk = 0;
		    } else if (l >= 4 && !strncasecmp(base + b, "Bcc:", 4)) {
			lineOk = 0;
		    } else if (l >= 5 && !strncasecmp(base + b, "Path:", 5)) {
			lineOk = 0;
		    } 
		}
		if (lineOk) {
		    fwrite(base + b, l, 1, fo);
		    fputc('\n', fo);
		}
		lastLineOk = lineOk;
		b = i;
	    }
	    fclose(fo);
	} else {
	    close(fds[1]);
	}
	waitpid(pid, NULL, 0);
    }
}

int
CheckGroupModeration(const char *group)
{
    const char *rec;
    int recLen;
    int r = -1;

    rec = KPDBReadRecord(KDBActive, group, KP_LOCK, &recLen);
    if (rec != NULL) {
        int slen;
        const char *s = KPDBGetField(rec, recLen, "S", &slen, NULL);
        if (s) {
            int i;

	    for (i = 0; i < slen; ++i) {
		if (s[i] == 'm')
                    r = 0;
            }
        }
        KPDBUnlock(KDBActive, rec);
    }
    return(r);
}

/*
 * GenerateDistribution(): generate a distribution header.  At the moment
 *			   we only generate one distribution tag, even if
 *			   crossposting, using the earliest highest-weight
 *			   tag found in distrib.pats.
 *
 *			   If we cannot find any matches, no Distribution:
 *			   header is generated.
 */

void
GenerateDistribution(MBufHead *mbuf, char *newsGrps)
{
    FILE *fi; 
    int bestWeight = -1;
    char *bestDist = NULL;

    if ((fi = fopen(PatLibExpand(DistribDotPatsPat), "r")) != NULL) {
        char buf[256];
    
        while (fgets(buf, sizeof(buf), fi) != NULL) {
	    char *weight = strtok(buf, ":\r\n");
	    char *pattern = weight ? strtok(NULL, ":\r\n") : NULL;
	    char *value = pattern ? strtok(NULL, ":\r\n") : NULL;
	    int i = 0;

	    if (value == NULL || 
	        weight == NULL || 
		weight[0] == '#' ||
		weight[0] == 0
	    ) {
		continue;
	    }
	    while (newsGrps[i]) {
		int j;
		char c;

		for (j = i; newsGrps[j] && newsGrps[j] != ','; ++j)
		    ;
		c = newsGrps[j];
		newsGrps[j] = 0;
		if (WildCmp(pattern, newsGrps + i) == 0) {
		    if (strtol(weight, NULL, 0) > bestWeight) {
			zfreeStr(&SysMemPool, &bestDist);
			bestWeight = strtol(weight, NULL, 0);
			bestDist = zallocStr(&SysMemPool, value);
		    }
		}
		newsGrps[j] = c;
		if (newsGrps[j] != ',')
		    break;
		i = j + 1;
	    }
        }
        fclose(fi);
    }
    if (bestDist) {
	MBPrintf(mbuf, "Distribution: %s\r\n", bestDist);
	zfreeStr(&SysMemPool, &bestDist);
    }
}

