
#include <lib/defs.h>
#include <sys/uio.h>

#define DRHSIZE		256
#define DRHMASK		(DRHSIZE-1)

/*
 * MSG_ flags that may not be implemented on a particular OS
 */

#ifndef MSG_WAITALL
#define MSG_WAITALL	0
#endif
#ifndef MSG_EOR
#define MSG_EOR		0
#endif

struct ServReq;
struct Timer;

#define THREAD_LISTEN	1	/* listening thread		*/
#define THREAD_DNS	2	/* dns resolver thread		*/
#define THREAD_READER	3	/* reader thread		*/
#define THREAD_SDNS	4	/* server dns resolver thread	*/

#define THREAD_NNTP	5	/* nntp connection		*/
#define THREAD_SPOOL	6	/* spool connection		*/
#define THREAD_POST	7	/* outgoing post		*/
#define THREAD_FEEDER	8	/* feeder thread		*/

#define OVERVIEW_FMT	"Subject:\r\nFrom:\r\nDate:\r\nMessage-ID:\r\nReferences:\r\nBytes:\r\nLines:\r\nXref:full\r\n"
#define DEFMAXARTSINGROUP	1024
#define DEFARTSINGROUP		512
#define OV_CACHE_MAX			8	/* open fd's for N groups */
#define MAX_OVERVIEW_CACHE_REGIONS      2       /* 256x2 = 512 articles   */
#define MAX_CROSS			64

/*
 * THREAD_QSIZE is the number of spool requests that can be queued to
 * a spool thread before we back off to a higher d_Pri.
 *
 * THREAD_PSIZE is the number of post requests that can be queued to
 * a spool thread before we back off to a higher d_Pri.
 */

#define THREAD_QSIZE	4	/* for spool access		*/
#define THREAD_PSIZE	2	/* for posting			*/
#define THREAD_LIMSIZE	(((THREAD_QSIZE > THREAD_PSIZE) ? THREAD_QSIZE : THREAD_PSIZE) * 2)

/*
 * ForkDesc - structure used to manage descriptors for select() threads
 */

typedef struct ForkDesc {
    struct Timer *d_Timer;	/* active timer		*/
    struct ForkDesc *d_Next;	/* linked list @ pri	*/ 
    struct ForkDesc *d_Prev;	/* linked list @ pri	*/ 
    char	*d_Id;		/* host name 		*/
    pid_t	d_Pid;
    int		d_Slot;		/* status reporting slot*/
    int		d_Fd;
    int		d_Type;
    int		d_FdPend;	/* save pending fd	*/
    int		d_Count;	/* active subthreads	*/
    int		d_Pri;
    void	*d_Data;
    struct sockaddr_in d_SaveRSin;
} ForkDesc;

/*
 * DnsReq/DnsRes - structures used to manage DNS lookups
 */

typedef struct DnsReq {
    struct sockaddr_in dr_LSin;
    struct sockaddr_in dr_RSin;
    char dr_AuthUser[16];
    char dr_AuthPass[16];
    int performident;
} DnsReq;

typedef struct DnsRes {
    struct DnsRes *dr_HNext;	/* main process hash link */
    int dr_PerArtRateLim;
    int dr_GlobalRateLim;
    int dr_MaxFromMatch;
    int dr_MaxFromHost;
    int dr_MaxFromUserHost;
    int dr_Code;
    int dr_Flags;
    int dr_FetchPri;
    int dr_PushPri;
    int dr_GroupId;
    int dr_GrpCount;
    struct in_addr dr_Addr;
    int dr_Port;
    char dr_User[16];
    char dr_Host[64];
    char dr_Match[64];
} DnsRes;

#define dr_ReaderPid	dr_MaxFromHost	/* master process pid	*/
#define dr_ArtCount	dr_FetchPri	/* reader process return stats */
#define dr_ByteCount	dr_PushPri	/* reader process return stats */

#define DF_FEED		0x00000001	/* f		*/
#define DF_READ		0x00000002	/* r		*/
#define DF_POST		0x00000004	/* p/w		*/
#define DF_FETCHSERVER	0x00000008	/* s#		*/
#define DF_PUSHSERVER	0x00000010	/* o#		*/
#define DF_GROUPID	0x00000020	/* g#		*/
#define DF_AUTHREQUIRED	0x00000040	/* a user=user pass=pass */
#define DF_HOSTISIP	0x00000080	/* host auth is IP address */
#define DF_QUIET	0x00000100	/* q: do not log*/
#define DF_FEEDONLY	0x00000200	/* feed-only thread	*/
#define DF_VALID	0x40000000

/*
 * MBuf - structure used to manage read and write buffers
 */

typedef struct MBuf {
    struct MBuf *mb_Next;
    char	*mb_Buf;
    int		mb_Index;	/* finished index		*/
    int		mb_NLScan;	/* newline scan index 		*/
    int		mb_Size;
    int		mb_Max;
} MBuf;

typedef struct MBufHead {
    MBuf	*mh_MBuf;
    MemPool	**mh_MemPool;
    MemPool	**mh_BufPool;
    int		mh_Bytes;
    int		mh_Wait;
    int		mh_Fd;
    int		mh_TotalBytes;
    char	mh_REof;
    char	mh_WEof;
    char	mh_RError;
    char	mh_WError;
} MBufHead;

/*
 * Timer
 */

typedef struct Timer {
    struct Timer   *ti_Next;
    struct Timer   *ti_Prev;
    struct ForkDesc *ti_Desc;
    struct timeval ti_To;	/* requested timeout		*/
    struct timeval ti_Tv;	/* absolute time of timeout	*/
    int		ti_Flags;
} Timer;

#define TIF_READ	0x01
#define TIF_WRITE	0x02

typedef struct TimeRestrict {
    time_t	     tr_Time;
} TimeRestrict;

/*
 * Connection - structure used to manage an NNTP connection
 *
 * note(1): Server requests are attached in two places:  The co_SReq in a 
 *	    THREAD_NNTP connection and the co_SReq in a THREAD_SPOOL or
 *	    THREAD_POST connection.
 */

typedef struct Connection {
    ForkDesc	*co_Desc;
    void	(*co_Func)(struct Connection *conn);
    int		(*co_ArtFuncHead)(struct Connection *conn, char *buf, int len);
    int		(*co_ArtFuncBody)(struct Connection *conn, char *buf, int len);
    const char	*co_State;
    MemPool	*co_MemPool;
    MemPool	*co_BufPool;
    MBufHead	co_TMBuf;
    MBufHead	co_RMBuf;
    int		co_FCounter;
    int		co_ByteCounter;
    int		co_BytesHeader;
    int		co_Flags;
    TimeRestrict co_TimeRestrict;
    DnsRes	co_Auth;
    char	*co_AuthInfoUser;
    char	*co_AuthInfoPass;
    char	*co_GroupName;	/* current group or NULL	*/
    char	*co_IHaveMsgId;
    Control	*co_Ctl;	/* control lookup cache		*/
    struct ServReq *co_SReq;	/* see note(1)			*/
    int		co_ListRec;
    int		co_ListRecLen;
    char	*co_ListPat;
    char	*co_ListHdrs;
    int		co_ListBegNo;
    int		co_ListEndNo;
    int		co_ArtMode;	/* current article mode		*/
    int		co_ArtNo;	/* current article number	*/
    int		co_ArtBeg;
    int		co_ArtEnd;

    struct timeval co_RateTv;
    int		co_RateCounter;

    MBufHead	co_ArtBuf;	/* article buffer		*/
} Connection;

#define COF_SERVER	0x00000001
#define COF_HEADFEED	0x00000002
#define COF_STREAM	0x00000004
#define COF_IHAVE	0x00000008	/* temporary ihave->takethis */
#define COF_POST	0x00000010	/* post command, else feed   */
#define COF_INHEADER	0x00000020	/* post/feed, reading headers*/
#define COF_DORANGE	0x00000040	/* do article range, else msgid */
#define COF_WASCONTROL	0x00000080
#define COF_MODEREADER  0x00000100	/* spool fetch, do mode reader	*/
#define COF_CLOSESERVER 0x00010000	/* close server ASAP	     	*/
#define COF_PATISWILD	0x00020000	/* list pattern is wildcard	*/
#define COF_MAYCLOSESRV 0x00040000	/* maybe close server 		*/
#define COF_INPROGRESS	0x00080000	/* operation in progress	*/
#define COF_ININIT	0x00100000	/* see server.c			*/

#define MBUF_SIZE	1024
#define MBUF_HIWAT	(MBUF_SIZE*2-256)

#define COM_ARTICLE	0
#define COM_BODY	1
#define COM_HEAD	2
#define COM_STAT	3
#define COM_BODYNOSTAT	4
#define COM_FUNC	5

#define COM_ACTIVE	6		/* list active			*/
#define COM_GROUPDESC	7		/* list newsgroups grouppat	*/

#define COM_XHDR	8
#define COM_XOVER	9
#define COM_XPAT	10

#define COM_NEWGROUPS	11		/* newgroups			*/
#define COM_ARTICLEWVF	12		/* verify body before output	*/
#define COM_BODYWVF	13		/* verify body before output	*/

typedef struct ServReq {
    struct ServReq *sr_Next;	/* linked list of requests	*/
    Connection *sr_CConn;	/* client making request	*/
    Connection *sr_SConn;	/* server handling request	*/
    char	*sr_Group;	/* request related to group	*/
    char	*sr_MsgId;	/* request related to messageid	*/
    time_t	sr_Time;	/* time of request for timeout calc	*/
    FILE	*sr_Cache;	/* cache write (locked for duration)	*/
    int		sr_Rolodex;	/* see server.c		*/
    int		sr_NoPass;	/* see server.c		*/
} ServReq;

#define SREQ_RETRIEVE	1
#define SREQ_POST	2


/*
 * Overview record.	over.groupname	(overview information - headers)
 *			numa.groupname	(article number assignment file)
 */

typedef union OverHead {
    struct {
	int	reserved[32];
    } u;
    struct {
	int	version;
	int	byteOrder;
	int	headSize;
	int	maxArts;
    } v;
} OverHead;

#define oh_Version	v.version
#define oh_HeadSize	v.headSize
#define oh_MaxArts	v.maxArts	/* adjusted by dexpireover -s */
#define oh_ByteOrder	v.byteOrder

#define OH_VERSION	1
#define OH_BYTEORDER	((int)0xF1E2D3C4)

typedef struct OverArt {
    int         oa_ArtNo;       /* article number                       */
    int         oa_SeekPos;     /* seek in data.grouphash file          */
    int		oa_Bytes;	/* bytes of headers in data.grphash file*/
    hash_t	oa_MsgHash;	/* locate message-id (used by cancel)	*/
    int		oa_ArtSize;	/* used for xover Bytes: header		*/
    int		oa_TimeRcvd;	/* time received			*/
    int		oa_Unused2;
} OverArt;

typedef struct OverData {
    struct OverData *od_Next;
    int		od_HFd;
    int		od_ArtBase;
    int		od_HMapPos;
    int		od_HMapBytes;
    const char *od_HMapBase;
} OverData;

typedef struct OverInfo {
    struct OverInfo *ov_Next;
    int		ov_Refs;
    char        *ov_Group;
    int         ov_MaxArts;     /* maximum number of articles in group  */
    OverData	*ov_HData;	/* hdata file reference linked list	*/
    OverData	*ov_HCache;	/* last accessed file reference		*/
    int         ov_OFd;
    OverHead	*ov_Head;
    int		ov_Size;
} OverInfo;

typedef struct ArtNumAss {
    struct ArtNumAss *an_Next;
    const char	     *an_GroupName;	/* NOT TERMINATED	*/
    int	 	     an_GroupLen;
    int		     an_ArtNo;
} ArtNumAss;

#include <obj/dreaderd-protos.h>


