#include "MainDialog.h"
#define Inherited MainDialogData

#include "images/prev.xpm"
#include "images/stop.xpm"
#include "images/play.xpm"
#include "images/pause.xpm"
#include "images/ff.xpm"
#include "images/next.xpm"

#include "images/stop_p.xpm"
#include "images/pause_p.xpm"
#include "images/ff_p.xpm"

#include "images/prev_l.xpm"
#include "images/stop_l.xpm"
#include "images/play_l.xpm"
#include "images/pause_l.xpm"
#include "images/ff_l.xpm"
#include "images/next_l.xpm"

#include "images/prev_l_p.xpm"
#include "images/stop_l_p.xpm"
#include "images/play_l_p.xpm"
#include "images/next_l_p.xpm"

// --------------------------------------------------------------------------

MainDialog::MainDialog(QWidget* parent,	const char* name) : Inherited(parent,name)
{
    setCaption( "SIDPLAY" );
    setMinimumSize(width(),70);
    setIcon(*myMainIcon);

    infoField->setFont(QFont("Courier",12,QFont::Normal,false,QFont::Latin1));
    infoField->setText("");
	
    QPopupMenu *file = new QPopupMenu;
    CHECK_PTR( file );
    file->insertItem( "Open", this, SLOT(open()), CTRL+Key_O );
    file->insertItem( "Save as", this, SLOT(saveAs()), CTRL+Key_S );
    file->insertSeparator();
    file->insertItem( "Exit", this, SLOT(quit()), CTRL+Key_Q );

    QPopupMenu *config = new QPopupMenu;
    CHECK_PTR( config );
    config->insertItem( "Audio", this, SLOT(openAudioDialog()) );
    config->insertItem( "Emulator", this, SLOT(openEmulatorDialog()) );
    config->insertItem( "SID Filter", this, SLOT(openFilterDialog()) );
    config->insertItem( "HVSC Info", this, SLOT(openHVSCDialog()) );

    extra = new QPopupMenu;
    CHECK_PTR( extra );
    extra->setCheckable(true);
    stilID = extra->insertItem( "STIL View", this, SLOT(toggleStilDialog()) );
	extra->insertItem( "History", this, SLOT(openHistoryDialog()) );
    extra->insertItem( "Mixer", this, SLOT(openMixerDialog()) );
    waveViewID = extra->insertItem( "Oscilloscope", this, SLOT(openWaveViewDialog()) );

    QPopupMenu *help = new QPopupMenu;
    CHECK_PTR( help );
    help->insertItem( "SIDPLAY", this, SLOT(about()), CTRL+Key_H );
    help->insertItem( "Qt", this, SLOT(aboutQt()) );

    menu = new QMenuBar( this );
    CHECK_PTR( menu );
    menu->insertItem( "&File", file );
    menu->insertItem( "&Config", config );
    menu->insertItem( "&Extra", extra );
    menu->insertSeparator();
    menu->insertItem( "&About", help );
	
    myFileDlg = new QFileDialog(0,0,false);
    myFileDlg->setMode(QFileDialog::ExistingFile);
    myFileDlg->setCaption("Sidtune Selection");
    myFileDlg->setIcon(*myMainIcon);
    connect(myFileDlg,SIGNAL(fileHighlighted(const char*)),this,SLOT(playFile(const char*)));
	
    myAudioDlg = new AudioDialog();

    myEmuDlg = new EmuDialog();
    
    myFilterDlg = new FilterDialog();
    
    myMixerDlg = new MixerDialog();
	
    myStilDlg = new StilDialog();

    myHVSCDlg = new HVSCDialog();

    myWaveViewDlg = new WaveViewDialog();

	myHistoryDlg = new HistoryDialog();
    
    this->connect(myAudioDlg,SIGNAL(changed(const AudioConfig&)),this,SLOT(receiveAudioConfig(const AudioConfig&)));
    myFilterDlg->connect(myEmuDlg,SIGNAL(changed(const emuConfig&)),myFilterDlg,SLOT(receiveConfig(const emuConfig&)));
    myEmuDlg->connect(myFilterDlg,SIGNAL(changed(const emuConfig&)),myEmuDlg,SLOT(receiveConfig(const emuConfig&)));
    this->connect(myEmuDlg,SIGNAL(changed(const emuConfig&)),this,SLOT(receiveEmuConfig(const emuConfig&)));
    this->connect(myFilterDlg,SIGNAL(changed(const emuConfig&)),this,SLOT(receiveEmuConfig(const emuConfig&)));
    this->connect(myHVSCDlg,SIGNAL(changed(const HVSCconfig&)),this,SLOT(receiveHVSCconfig(const HVSCconfig&)));
    this->connect(myMixerDlg,SIGNAL(changed(const MixerConfig&)),this,SLOT(receiveMixerConfig(const MixerConfig&)));
    this->connect(myHistoryDlg,SIGNAL(playHistoryEntry(const char*)),this,SLOT(playHistoryFile(const char*)));
    
    connect( prevBtn, SIGNAL(pressed()), SLOT(prevSongPressed()) );
    connect( prevBtn, SIGNAL(released()), SLOT(prevSongReleased()) );
    prevBtn->setAutoRepeat( true );
    prevBtn->setAutoResize( false );
    QToolTip::add(prevBtn,"Previous song");

    connect( stopBtn, SIGNAL(pressed()), SLOT(stopSongPressed()) );
    connect( stopBtn, SIGNAL(released()), SLOT(stopSongReleased()) );
    stopBtn->setAutoRepeat( false );
    stopBtn->setAutoResize( false );
    QToolTip::add(stopBtn,"Stop playing");

    connect( playBtn, SIGNAL(pressed()), SLOT(playSongPressed()) );
    connect( playBtn, SIGNAL(released()), SLOT(playSongReleased()) );
    playBtn->setAutoRepeat( false );
    playBtn->setAutoResize( false );
    QToolTip::add(playBtn,"Play/Pause");

    connect( ffBtn, SIGNAL(pressed()), SLOT(forwardSongPressed()) );
    connect( ffBtn, SIGNAL(released()), SLOT(forwardSongReleased()) );
    ffBtn->setAutoRepeat( false );
    ffBtn->setAutoResize( false );
    QToolTip::add(ffBtn,"Fast forward");

    connect( nextBtn, SIGNAL(pressed()), SLOT(nextSongPressed()) );
    connect( nextBtn, SIGNAL(released()), SLOT(nextSongReleased()) );
    nextBtn->setAutoRepeat( true );
    nextBtn->setAutoResize( false );
    QToolTip::add(nextBtn,"Next song");

    subSongSlider = new QSlider(this,0);
    subSongSlider->setOrientation(QSlider::Horizontal);
    subSongSlider->setGeometry(140,60,136+64+12,10);
    subSongSlider->setTracking(false);
    subSongSlider->setValue(1);
    subSongSlider->setRange(1,1);
    subSongSlider->setSteps(1,1);
    subSongSlider->setFocus();
    connect(subSongSlider,SIGNAL(sliderMoved(int)),songLCD,SLOT(display(int)));
    connect(subSongSlider,SIGNAL(valueChanged(int)),this,SLOT(newSubSong(int)));
    QToolTip::add(subSongSlider,"Quick song selection");

    songLCD->display(1);
    QToolTip::add(songLCD,"Song number");

    timeLCD->setFrameStyle( 50 );
    timeLCD->setSmallDecimalPoint( FALSE );
    timeLCD->setNumDigits( 5 );
    timeLCD->setMode( QLCDNumber::DEC );
    timeLCD->setSegmentStyle( QLCDNumber::Filled );
    {
        QColorGroup normal( QColor( QRgb(0) ), QColor( QRgb(11513755) ), QColor( QRgb(16777215) ), QColor( QRgb(6316128) ), QColor( QRgb(10789024) ), QColor( QRgb(0) ), QColor( QRgb(16777215) ) );
        QColorGroup disabled( QColor( QRgb(8421504) ), QColor( QRgb(12632256) ), QColor( QRgb(16777215) ), QColor( QRgb(6316128) ), QColor( QRgb(10789024) ), QColor( QRgb(8421504) ), QColor( QRgb(12632256) ) );
        QColorGroup active( QColor( QRgb(0) ), QColor( QRgb(12632256) ), QColor( QRgb(16777215) ), QColor( QRgb(6316128) ), QColor( QRgb(10789024) ), QColor( QRgb(0) ), QColor( QRgb(16777215) ) );
        QPalette palette( normal, disabled, active );
        timeLCD->setPalette( palette );
	}
	
    // Init pixmap pointers.
	
    prevPic = new QPixmap((const char**)prev_xpm);
    stopPic = new QPixmap((const char**)stop_xpm);
    playPic = new QPixmap((const char**)play_xpm);
    pausePic = new QPixmap((const char**)pause_xpm);
    ffPic = new QPixmap((const char**)ff_xpm);
    nextPic = new QPixmap((const char**)next_xpm);

    stop_pPic = new QPixmap((const char**)stop_p_xpm);
    pause_pPic = new QPixmap((const char**)pause_p_xpm);
    ff_pPic = new QPixmap((const char**)ff_p_xpm);
	
    prev_lPic = new QPixmap((const char**)prev_l_xpm);
    stop_lPic = new QPixmap((const char**)stop_l_xpm);
    play_lPic = new QPixmap((const char**)play_l_xpm);
    pause_lPic = new QPixmap((const char**)pause_l_xpm);
    ff_lPic = new QPixmap((const char**)ff_l_xpm);
    next_lPic = new QPixmap((const char**)next_l_xpm);
	
    prev_l_pPic = new QPixmap((const char**)prev_l_p_xpm);
    stop_l_pPic = new QPixmap((const char**)stop_l_p_xpm);
    play_l_pPic = new QPixmap((const char**)play_l_p_xpm);
    next_l_pPic = new QPixmap((const char**)next_l_p_xpm);
	
    buttonsOff();

    // Instantiate sidtune loader.
    mySidTune = new sidTune(0);

    myEmuEngine = new emuEngine();
    myEmuEngine->getConfig(myEmuConfig);

    myEmuDlg->setConfig(myEmuConfig);
    myFilterDlg->setConfigDefault(myEmuConfig);  // *not* redundant
    myFilterDlg->setConfig(myEmuConfig);
    
    // The currently active file name.
    sidFileNameFull = sidFileNameHVSC = "";

    buffers = 2;
    multiBufferSize = 2048;
    initAudioBufferSystem();
    
    playbackTimer.stop();
    connect(&playbackTimer,SIGNAL(timeout()),this,SLOT(timerJob()));
    playbackState = IS_STOPPED;
    isReadyToPlay = false;
}

MainDialog::~MainDialog()
{
    playbackTimer.stop();
    freeAudioBufferSystem();
    
    delete mySidTune;
    delete myEmuEngine;
    
    delete prevPic;
    delete stopPic;
    delete playPic;
    delete pausePic;
    delete ffPic;
    delete nextPic;

    delete stop_pPic;
    delete pause_pPic;
    delete ff_pPic;
	
    delete prev_lPic;
    delete stop_lPic;
    delete play_lPic;
    delete pause_lPic;
    delete ff_lPic;
    delete next_lPic;
}

void MainDialog::resizeEvent(QResizeEvent*)
{
    int w = width();
    int h = height();
    
    // rel. 365,195
	sidInfoFrame->setGeometry(10,75,w-20,h-85);
    infoField->setGeometry(15,80,w-30,h-95);
}


void MainDialog::enableWaveViewDlg(bool val)
{
    menu->setItemEnabled(waveViewID,val);
}

void MainDialog::done(int r)
{
    stopPlaying();
    writeConfigFile();     // ignore return value
    myHistoryDlg->save();  //
    QDialog::done(r);
}

void MainDialog::adjustCommandButtons()
{
	if (!isReadyToPlay)
	{
		prevBtn->setPixmap(*prevPic);
		stopBtn->setPixmap(*stopPic);
		playBtn->setPixmap(*playPic);
		ffBtn->setPixmap(*ffPic);
		nextBtn->setPixmap(*nextPic);
		return;
	}
	else
	{
		if ( mySidTune->getStatus() )
		{
			mySidTune->getInfo(mySidTuneInfo);
		}
		else
		{
			mySidTuneInfo.currentSong = 1;
			mySidTuneInfo.songs = 1;
		}
		showSidTuneInfo();
	}
}

void MainDialog::buttonsOff()
{
	prevBtn->setPixmap(*prevPic);
	stopBtn->setPixmap(*stopPic);
	playBtn->setPixmap(*playPic);
	ffBtn->setPixmap(*ffPic);
	nextBtn->setPixmap(*nextPic);
}

void MainDialog::buttonsForceInitialState()
{
	// Make them look like when playing.
	setPrevBtnState();
	stopBtn->setPixmap(*stop_lPic);
	playBtn->setPixmap(*pause_lPic);
	ffBtn->setPixmap(*ff_lPic);
	setNextBtnState();
}

void MainDialog::showSidTuneInfo()
{
	if (mySidTune->getStatus())
    {
		mySidTune->getInfo(mySidTuneInfo);
		
		QString tmp;
        if ( mySidTuneInfo.numberOfInfoStrings == 3 )
        {
            tmp = "Name      : ";
            tmp += mySidTuneInfo.infoString[0];
            tmp += "\nAuthor    : ";
            tmp += mySidTuneInfo.infoString[1];
            tmp += "\nCopyright : ";
            tmp += mySidTuneInfo.infoString[2];
        }
        else
        {
            for ( int infoi = 0; infoi < mySidTuneInfo.numberOfInfoStrings; infoi++ )
            {
                if (infoi > 0)
                    tmp += "\n";
                tmp += "Info      : ";
                tmp += mySidTuneInfo.infoString[infoi];
            }
        }
		QString tmp2;
		tmp2.sprintf("\nSongs     : %d (Startsong: %d)",
					 mySidTuneInfo.songs,
					 mySidTuneInfo.startSong);
		tmp += tmp2;
		tmp2.sprintf("\nSong speed: %s",mySidTuneInfo.speedString);
		tmp += tmp2;
		tmp2.sprintf("\nAddresses : $%04x, $%04x, $%04x",
					 mySidTuneInfo.loadAddr,
					 mySidTuneInfo.initAddr,
					 mySidTuneInfo.playAddr);
		tmp += tmp2;
		tmp2.sprintf("\nFormat    : %s",mySidTuneInfo.formatString);
		tmp += tmp2;
		infoField->setText(tmp);
			
		songLCD->display(mySidTuneInfo.currentSong);
	}
	else
	{
		// error (can this ever happen?)
	}
}

void MainDialog::clearSidTuneInfo()
{
    QString tmp;
    tmp  = "Name      : \n";
    tmp += "Author    : \n";
    tmp += "Copyright : \n";
    tmp += "Addresses : \n";
    tmp += "Songs     : \n";
    tmp += "Song speed: \n";
    tmp += "Format    : \n";
    infoField->setText(tmp);
}

void MainDialog::commandLinePlay(const char* fileName, const int song)
{
    if (fileName == 0)
        return;
    if (strcmp(fileName,"-") != 0)
    {
        // Who would use '-' for something else than stdin?
        QFile testFile(fileName);
        if (!testFile.exists())
            return;
    }
    isHistoryFile = true;  // don`t put in history
    loadSidTune(fileName,song);
    loadStilData(fileName);
}

void MainDialog::playFile(const char* fileName)
{
    isHistoryFile = false;
	QFile testFile(fileName);
	if (testFile.exists())
	{
		loadSidTune(fileName,0);
		loadStilData(fileName);
	}
}

void MainDialog::playHistoryFile(const char* fileName)
{
    isHistoryFile = true;
	QFile testFile(fileName);
	if (testFile.exists())
	{
		loadSidTune(fileName,0);
		loadStilData(fileName);
	}
    else
    {
        ; // file no longer exists
    }
}

void MainDialog::quit()
{
	stopPlaying();
    writeConfigFile();     // ignore return value
    myHistoryDlg->save();  //
	qApp->quit();
}

void MainDialog::open()
{
    if (myVarConfig.geomDirDlg.good())
    {
        myFileDlg->setGeometry(myVarConfig.geomDirDlg.x,myVarConfig.geomDirDlg.y,
                               myVarConfig.geomDirDlg.w,myVarConfig.geomDirDlg.h);
    }
    myFileDlg->show();
}

void MainDialog::saveAs()
{
	if (mySidTune->getStatus())
    {
		if ( !mySidTune->savePSIDfile(QFileDialog::getSaveFileName(),true) )
		{			
			QMessageBox error;
			error.setIcon(QMessageBox::Critical);
			error.setText("Could not save file.");
			error.adjustSize();
			error.show();
		}
	}
}

void MainDialog::about()
{
	QString verCat = "SIDPLAY/X11   Version ";
    verCat += VERSION;
    verCat += "\n\n";
	verCat += "Music player and SID chip emulator ";
	verCat += myEmuEngine->getVersionString();
	verCat += "\nby Michael Schwendt <sidplay@geocities.com>\n\n";
	
	verCat += mySTIL.getVersion();
		
	verCat += "\nListening mileage: ";
	udword mileage = usage+myEmuEngine->getSecondsTotal();
	int days = mileage/(3600L*24);
	mileage -= days*(3600L*24);
	int hours = mileage/3600L;
	mileage -= hours*3600L;
	int minutes = mileage/60;
	mileage -= minutes*60;
	QString timeCat;
	timeCat.sprintf("%d days, %d hours, %d minutes",days,hours,minutes);
	verCat += timeCat;
	QMessageBox::about(this,"About SIDPLAY",verCat);
}

void MainDialog::aboutQt()
{
	QMessageBox::aboutQt(this);
}

void MainDialog::openAudioDialog()
{
	myAudioDlg->setButtons();
	myAudioDlg->show();
}

void MainDialog::openEmulatorDialog()
{
	myEmuDlg->show();
}

void MainDialog::openFilterDialog()
{
	myFilterDlg->show();
}

void MainDialog::openMixerDialog()
{
	myMixerDlg->show();
}

void MainDialog::openWaveViewDialog()
{
	if ( !myWaveViewDlg->isShown() )
		myWaveViewDlg->show();
}

void MainDialog::openHVSCDialog()
{
    myHVSCDlg->show();
}

void MainDialog::openHistoryDialog()
{
    myHistoryDlg->show();
}


/*
 * [ms]: If this way of receiving config data from dialogs (and sending
 *       data to other dialogs) on the fly turns out to be buggy, it
 *       could be changed at any time by turning the dialogs into modal
 *       dialogs and adding common OK/Apply/Cancel buttons.
 *       Currently the worst thing that can happen is that Qt signals
 *       pass on to the next dialog until all dialogs have the same
 *       config.
 */

void MainDialog::setAudioConfig(const AudioConfig& inConfig)
{
    playerState playbackStateOld = playbackState;
    playbackState = IS_INTERRUPTED;
    if (driverIsOpen)
        myAudio.reset();

    myAudioDlg->setConfig(inConfig);
    // Try to open/init audio driver. This will take the config
    // from the audio dialog.
    if (!openAudioDriver())
    {  
        // Set system to inoperational state.
        // Fix this! Buttons?
        playbackStateOld = IS_STOPPED;
    }
    if ((playbackStateOld==IS_INTERRUPTED) ||
        (playbackStateOld==IS_STOPPED))
    {
        closeAudioDriver();
    }

    // Now retrieve the possibly changed config from the driver.
    AudioConfig goodAudioConfig = myAudio.getConfig();
    myAudioDlg->setConfig(goodAudioConfig);
    
    // Translate config to EMU. It is assumed that the emulator
    // engine can output in all possible audio configs.
    
    if (goodAudioConfig.precision == AudioConfig::BITS_16)
        myEmuConfig.bitsPerSample = SIDEMU_16BIT;
    else
        myEmuConfig.bitsPerSample = SIDEMU_8BIT;
    
    if (goodAudioConfig.encoding == AudioConfig::SIGNED_PCM)
        myEmuConfig.sampleFormat = SIDEMU_SIGNED_PCM;
    else
        myEmuConfig.sampleFormat = SIDEMU_UNSIGNED_PCM;
    
    if (goodAudioConfig.channels == AudioConfig::STEREO)
        myEmuConfig.channels = SIDEMU_STEREO;
    else
    {
        myEmuConfig.channels = SIDEMU_MONO;
        myEmuConfig.autoPanning = SIDEMU_NONE;
        // Max. mono can do is VOLCONTROL mode.
        if (myEmuConfig.volumeControl==SIDEMU_FULLPANNING ||
            myEmuConfig.volumeControl==SIDEMU_STEREOSURROUND)
            myEmuConfig.volumeControl = SIDEMU_NONE;
    }
    myEmuConfig.frequency = goodAudioConfig.frequency;
    // Set the emuEngine to supported settings.
    myEmuEngine->setConfig(myEmuConfig);
    myEmuEngine->getConfig(myEmuConfig);
    
    // Set dialogs that directly depend on the emuConfig.
    myEmuDlg->setConfig(myEmuConfig);
    myFilterDlg->setConfig(myEmuConfig);
    
    // Here update the dialog buttons according to the settings the
    // audio driver accepted.
    myAudioDlg->setButtons();

    myWaveViewDlg->setConfig(goodAudioConfig);
    
    MixerConfig newMixerConfig = myMixerDlg->getConfig();
    // The mixer knows the emuConfig constants.
    newMixerConfig.channels = myEmuConfig.channels;
    newMixerConfig.mixingMode = myEmuConfig.volumeControl;
    newMixerConfig.panningMode = myEmuConfig.autoPanning;
    myMixerDlg->setConfig(newMixerConfig);
    // Make sure volume levels get set as well.
    receiveMixerConfig(newMixerConfig);
    
    if (inConfig.bufSize != 0)
        multiBufferSize = inConfig.bufSize;
    else
        multiBufferSize = goodAudioConfig.blockSize;
    openAudioBufferSystem();

	// Restore old state.
	playbackState = playbackStateOld;
}

void MainDialog::receiveAudioConfig(const AudioConfig& inConfig)
{
    setAudioConfig(inConfig);
}

void MainDialog::setEmuConfig(const emuConfig& inConfig)
{
    emuConfig tmp = inConfig;  // to escape compiler warning <g>
    myEmuEngine->setConfig(tmp);
    //  myEmuEngine->setConfig(inConfig);  // (alt.)
    myEmuEngine->getConfig(myEmuConfig);  // get a good copy
    
    // Set dialogs that directly depend on the emuConfig.
    // Not: myMixerDlg.
    myEmuDlg->setConfig(myEmuConfig);
    myFilterDlg->setConfig(myEmuConfig);
}

const emuConfig& MainDialog::getEmuConfig() const
{
    return myEmuConfig;
}

void MainDialog::receiveEmuConfig(const emuConfig& inConfig)
{
    emuConfig tmp = inConfig;  // to escape compiler warning <g>
    myEmuEngine->setConfig(tmp);
    //  myEmuEngine->setConfig(inConfig);  // (alt.)
    myEmuEngine->getConfig(myEmuConfig);  // get a good copy

    // Do not write back emuConfig to any dialog.
    // But:
    myFilterDlg->paintFilterFunc();
    
    MixerConfig newMixerConfig = myMixerDlg->getConfig();
    // The mixer knows the emuConfig constants.
    newMixerConfig.channels = myEmuConfig.channels;
    newMixerConfig.mixingMode = myEmuConfig.volumeControl;
    newMixerConfig.panningMode = myEmuConfig.autoPanning;
    myMixerDlg->setConfig(newMixerConfig);
}

void MainDialog::setHVSCconfig(const HVSCconfig& inConfig)
{
    myHVSCDlg->setConfig(inConfig);
    myStilDlg->setConfig(inConfig);
}

void MainDialog::receiveHVSCconfig(const HVSCconfig& inConfig)
{
    myStilDlg->setConfig(inConfig);
    initStilViewClass();
    if (!isStilInfoEnabled())
        myStilDlg->hide();
}

void MainDialog::receiveMixerConfig(const MixerConfig& inConfig)
{
    if ((myEmuConfig.volumeControl!=inConfig.mixingMode) ||
        (myEmuConfig.autoPanning!=inConfig.panningMode))
    {
        myEmuConfig.volumeControl = inConfig.mixingMode;
        myEmuConfig.autoPanning = inConfig.panningMode;
        myEmuEngine->setConfig(myEmuConfig);
        myEmuEngine->getConfig(myEmuConfig);
        myEmuDlg->setConfig(myEmuConfig);
    }

    for (int v=0; v < MixerDialog::voices; v++)
    {
		// In mono mode, set voice volume regardless of whether mixing
		// mode is active.
		if (myEmuConfig.channels == SIDEMU_MONO)
		{
			myEmuEngine->setVoiceVolume(v+1,255,0,
                                       inConfig.volTotal[v]&inConfig.muteMask[v]);
		}
		else
		{
			if (myEmuConfig.volumeControl == SIDEMU_VOLCONTROL)
			{
				myEmuEngine->setVoiceVolume(v+1,inConfig.volHQ[v].l,inConfig.volHQ[v].r,
										   inConfig.volTotal[v]&inConfig.muteMask[v]);
			}
			else if ((myEmuConfig.volumeControl==SIDEMU_FULLPANNING) &&
                     (myEmuConfig.autoPanning==SIDEMU_NONE))
            {
                myEmuEngine->setVoiceVolume(v+1,inConfig.volFP[v].l,inConfig.volFP[v].r,
                                           inConfig.volTotal[v]&inConfig.muteMask[v]);
            }
            else  // STEREOSURROUND or AUTOPANNING
            {
                // Adjust only the master volume level.
                // This is not clean because we don't use the emuEngine-defaults.
                if (myEmuConfig.volumeControl == SIDEMU_STEREOSURROUND)
                {
                    myEmuEngine->setVoiceVolume(v+1,255,255,inConfig.volTotal[v]&inConfig.muteMask[v]);
                }
                else
                {
                    if ((v&1)==0)
                        myEmuEngine->setVoiceVolume(v+1,255,0,inConfig.volTotal[v]&inConfig.muteMask[v]);
                    else
                        myEmuEngine->setVoiceVolume(v+1,0,255,inConfig.volTotal[v]&inConfig.muteMask[v]);
                }
            }
        }
    }
}

void MainDialog::setVariousConfig(const VariousConfig& inConfig)
{
    myVarConfig = inConfig;
    usage = inConfig.usage;
    
    // Since we don't save the entire MixerConfig to disk we copy
    // relevant values from the emuConfig.
    // The mixer knows the emuConfig constants.
    MixerConfig newMixerConfig = inConfig.myMixerConfig;
    newMixerConfig.channels = myEmuConfig.channels;
    newMixerConfig.mixingMode = myEmuConfig.volumeControl;
    newMixerConfig.panningMode = myEmuConfig.autoPanning;
    myMixerDlg->setConfig(newMixerConfig);
    // Make sure volume levels get set as well.
    receiveMixerConfig(newMixerConfig);
    
    if (inConfig.enableSTILview)
    {
        setStilInfoEnabled(initStilViewClass());
    }

    if (inConfig.geomMainDlg.good())
    {
        if (inConfig.geomMainDlg.w < 0)
            setGeometry(inConfig.geomMainDlg.x,inConfig.geomMainDlg.y,
                        width(),height());
        else
            setGeometry(inConfig.geomMainDlg.x,inConfig.geomMainDlg.y,
                        inConfig.geomMainDlg.w,inConfig.geomMainDlg.h);
    } 
    myFileDlg->setDir(inConfig.currentDir);
    QFileDialog* myBakFileDlg = myFileDlg;
    myFileDlg = new QFileDialog(myBakFileDlg->dirPath(),inConfig.nameFilter,0,0,false);
    myFileDlg->setMode(QFileDialog::ExistingFile);
    myFileDlg->setCaption("Sidtune Selection");
    myFileDlg->setIcon(*myMainIcon);
    delete myBakFileDlg;
    connect(myFileDlg,SIGNAL(fileHighlighted(const char*)),this,SLOT(playFile(const char*)));
    if (inConfig.showDirDlg)
        open();
}

bool MainDialog::writeConfigFile()
{
    ConfigFile myConfigFile(".sidplayrc");
    
    AudioConfig thisAudioConfig = myAudioDlg->getConfig();
    // Always save user`s sane fragmentation settings.
    extern AudioConfig bakAudioConfig;
    thisAudioConfig.bufSize = bakAudioConfig.bufSize;
    thisAudioConfig.maxFrags = bakAudioConfig.maxFrags;
    thisAudioConfig.fragSize = bakAudioConfig.fragSize;
    
    myConfigFile.setAudioConfig(thisAudioConfig);
    myConfigFile.setEmuConfig(myEmuConfig);
    myConfigFile.setHVSCconfig(myHVSCDlg->getConfig());
    myVarConfig.usage = usage+myEmuEngine->getSecondsTotal();
    myVarConfig.enableSTILview = isStilInfoEnabled();
    myVarConfig.currentDir = myFileDlg->dirPath();
    myVarConfig.nameFilter = myFileDlg->dir()->nameFilter();
    myVarConfig.showDirDlg = myFileDlg->isVisible();
    myVarConfig.geomMainDlg.x = x();
    myVarConfig.geomMainDlg.y = y();
    myVarConfig.geomMainDlg.w = width();
    myVarConfig.geomMainDlg.h = height();
    myVarConfig.geomDirDlg.x = myFileDlg->x();
    myVarConfig.geomDirDlg.y = myFileDlg->y();
    myVarConfig.geomDirDlg.w = myFileDlg->width();
    myVarConfig.geomDirDlg.h = myFileDlg->height();
    myVarConfig.myMixerConfig = myMixerDlg->getConfig();

    myConfigFile.setVariousConfig(myVarConfig);
    bool ret = myConfigFile.save();
    
    if (!ret)
    {
        QMessageBox error;
        error.setIcon(QMessageBox::Warning);
        error.setText("Could not save configuration file.");
        error.adjustSize();
        error.show();
    }
    return ret;
}

// --------------------------------------------------------------------------

void MainDialog::newSubSong(int newSong)
{
    if (mySidTune->getStatus())
    {
        playerState playbackStateOld = playbackState;
        playbackState = IS_INTERRUPTED;

        mySidTune->getInfo(mySidTuneInfo);
		if (newSong>=1 && newSong<=mySidTuneInfo.songs)
		{
			sidEmuInitializeSong(*myEmuEngine,*mySidTune,newSong);

            resetTimeDisplay();
            
			mySidTune->getInfo(mySidTuneInfo);
			if ( !mySidTune->getStatus() )
			{
				playbackStateOld = (playbackState = IS_STOPPED);
			}
			showSidTuneInfo();
			showStilInfo();
		}
		playbackState = playbackStateOld;
	}
	adjustCommandButtons();
	setPrevBtnState();
	setNextBtnState();
}

void MainDialog::updateSubSongSlider()
{
    if (mySidTune->getStatus())
    {
        mySidTune->getInfo(mySidTuneInfo);
        subSongSlider->setValue(mySidTuneInfo.currentSong);
        subSongSlider->setRange(1,mySidTuneInfo.songs);
        subSongSlider->setSteps(1,1);
    }
}

void MainDialog::prevSongPressed()
{
	if (mySidTune->getStatus())
    {
		playerState playbackStateOld = playbackState;
		playbackState = IS_INTERRUPTED;
		
		mySidTune->getInfo(mySidTuneInfo);
		if (mySidTuneInfo.currentSong != 1)
		{
			prevBtn->setPixmap(*prev_l_pPic);
	
			sidEmuInitializeSong(*myEmuEngine,*mySidTune,mySidTuneInfo.currentSong-1);
            
            resetTimeDisplay();
            updateSubSongSlider();
            
			mySidTune->getInfo(mySidTuneInfo);
			if ( !mySidTune->getStatus() )
			{
				playbackStateOld = (playbackState = IS_STOPPED);
			}
			showSidTuneInfo();
			showStilInfo();
		}
		playbackState = playbackStateOld;
	}
	adjustCommandButtons();
	setNextBtnState();
}

void MainDialog::prevSongReleased()
{
	setPrevBtnState();
}

void MainDialog::setPrevBtnState()
{
	// PREV possible?
	if (mySidTuneInfo.currentSong == 1)
	{
		prevBtn->setPixmap(*prevPic);
	}
	else
	{
		prevBtn->setPixmap(*prev_lPic);
	}
}

void MainDialog::nextSongPressed()
{
	if (mySidTune->getStatus())
    {
		playerState playbackStateOld = playbackState;
		playbackState = IS_INTERRUPTED;

		mySidTune->getInfo(mySidTuneInfo);
		if (mySidTuneInfo.currentSong != mySidTuneInfo.songs)
		{
			nextBtn->setPixmap(*next_l_pPic);
		
			sidEmuInitializeSong(*myEmuEngine,*mySidTune,mySidTuneInfo.currentSong+1);

            resetTimeDisplay();
            updateSubSongSlider();
            
			mySidTune->getInfo(mySidTuneInfo);
			if ( !mySidTune->getStatus() )
			{
				playbackStateOld = (playbackState = IS_STOPPED);
			}
			showSidTuneInfo();
			showStilInfo();
		}
		playbackState = playbackStateOld;
	}
	adjustCommandButtons();
	setPrevBtnState();
}

void MainDialog::nextSongReleased()
{
	setNextBtnState();
}

void MainDialog::setNextBtnState()
{
	// NEXT possible?
	if (mySidTuneInfo.currentSong == mySidTuneInfo.songs)
	{
		nextBtn->setPixmap(*nextPic);
	}
	else
	{
		nextBtn->setPixmap(*next_lPic);
	}
}

void MainDialog::stopSongPressed()
{
	if (isReadyToPlay && playbackState!=IS_STOPPED)
	{
		stopBtn->setPixmap(*stop_l_pPic);
		playBtn->setPixmap(*play_lPic);
		ffBtn->setPixmap(*ffPic);
		stopPlaying();
	}
}

void MainDialog::stopSongReleased()
{
	stopBtn->setPixmap(*stopPic);
}

void MainDialog::playSongPressed()
{
	if (playbackState == IS_STOPPED)
	{
		stopBtn->setPixmap(*stop_lPic);
		playBtn->setPixmap(*play_l_pPic);
		//
		startPlaying();
	}
	else if (playbackState == IS_PLAYING)
	{
		playBtn->setPixmap(*pause_pPic);
		//
		pausePlaying();
	}
	else if (playbackState == IS_PAUSED)
	{
		playBtn->setPixmap(*play_l_pPic);
		//
		resumePlaying();
	}
}

void MainDialog::playSongReleased()
{
	if (playbackState == IS_PLAYING)
	{
		playBtn->setPixmap(*pause_lPic);
		ffBtn->setPixmap(*ff_lPic);
	}
	else if (playbackState == IS_PAUSED)
	{
		playBtn->setPixmap(*play_lPic);
		ffBtn->setPixmap(*ffPic);
	}
}

void MainDialog::forwardSongPressed()
{
	extern bool sidEmuFastForwardReplay(int);
	if (playbackState == IS_FORWARDING)
	{
		playBtn->setPixmap(*pause_lPic);
		ffBtn->setPixmap(*ff_lPic);
		sidEmuFastForwardReplay(100);  // calc 100%
		playbackState = IS_PLAYING;
	}
	else if (playbackState == IS_PLAYING)
	{
		playBtn->setPixmap(*pausePic);
		ffBtn->setPixmap(*ff_pPic);
		sidEmuFastForwardReplay(10);   // calc 10%
		playbackState = IS_FORWARDING;
	}
}

void MainDialog::forwardSongReleased()
{
	if (playbackState == IS_FORWARDING)
	{
		ffBtn->setPixmap(*ff_pPic);
	}
	else if (playbackState == IS_PLAYING)
	{
		ffBtn->setPixmap(*ff_lPic);
	}
}

// -------------------------------------------------------------- STIL stuff.

void MainDialog::loadStilData(const char* fileName)
{
    // If the input filename contains a part which is relative to
    // the HVSC root, this will be that part.
    sidFileNameHVSC = "";
    
    // STIL View not enabled?
    if ( !isStilInfoEnabled() )
        return;

    QString fileNameStr = fileName;
    // Is filename anywhere in HVSC tree?
    if (fileNameStr.find(myHVSCDlg->getConfig().hvscRootPath.absPath()) == 0)
    {
        // Create filename which is relative to the HVSC root.
        sidFileNameHVSC = 
            fileNameStr.right(fileNameStr.length()-
                              myHVSCDlg->getConfig().hvscRootPath.absPath().length());
        showStilInfo();  // if any
    }
    else
    {
        // Nothing to show. Hence hide a possibly open STIL View dialog.
        myStilDlg->hide();
    }
}

void MainDialog::showStilInfo()
{
    // STIL View not enabled or no HVSC-relative file active?
	if ( !isStilInfoEnabled() || sidFileNameHVSC.isEmpty())
		return;

	int song = mySidTuneInfo.currentSong;
	if (myHVSCDlg->getConfig().showForFile)
		song = 0;              // 0 means: file-global STIL entry
    
	myStilDlg->clearLabels();  // drop previous entry
    
    bool stilIsOkay = true;    // we want to avoid successive errors
    char* globalComment;
    char* stilEntry;
    char* bugEntry;
    // Now retrieve the individual STIL entries.
    if (stilIsOkay)
    {
        globalComment = mySTIL.getGlobalComment(sidFileNameHVSC);
        stilIsOkay = !mySTIL.hasCriticalError();
    }
    if (stilIsOkay)
    {
        stilEntry = mySTIL.getEntry(sidFileNameHVSC,song);
        stilIsOkay = !mySTIL.hasCriticalError();
    }
    if (stilIsOkay)
    {
        bugEntry = mySTIL.getBug(sidFileNameHVSC,song);
        stilIsOkay = !mySTIL.hasCriticalError();
    }
    if (stilIsOkay)
    {
        // Show the STIL entry.
        myStilDlg->setLabels(globalComment,stilEntry,bugEntry);
    }
    else
    {
		QMessageBox error;
		error.setIcon(QMessageBox::Critical);
		error.setText(mySTIL.getErrorStr());
		error.adjustSize();
		error.show();
        // Disable STIL View.
        // Protect the user from any further critical errors.
		setStilInfoEnabled(false);
    }
}

void MainDialog::toggleStilDialog()
{
    if ( !isStilInfoEnabled() )
    {
        // Enable STIL View.
        setStilInfoEnabled(initStilViewClass());
        // Try to show STIL entry of currently active file.
        loadStilData(sidFileNameFull);
    }
    else
    {
        // Disable STIL View.
        setStilInfoEnabled(false);
        // Hide a possibly open STIL View dialog.
        myStilDlg->hide();
    }
}

bool MainDialog::initStilViewClass()
{
    // Let STIL View class do check for STIL database file.
    if ( !mySTIL.setBaseDir(myHVSCDlg->getConfig().hvscRootPath.absPath()) )
    {
        QMessageBox error;
        error.setIcon(QMessageBox::Warning);
        error.setText(mySTIL.getErrorStr());
        error.adjustSize();
        error.show();
        // Disable STIL View.
        setStilInfoEnabled(false);
        return false;
    }
    // STIL View class initialized.
    else
    {
        return true;
    }
}

// ------------------------------------------------------------------- Player

void MainDialog::loadSidTune(const char* fileName, const int commandLineStartSong)
{
	// Stop any kind of audio playback.
	stopPlaying();
    resetTimeDisplay();

	bool loadSucceeded = false;
	
    // Make a copy for later use.
    sidFileNameFull = fileName;
    
	// sidTune::open() does not support standard input.
	if (strcmp(fileName,"-") == 0)
	{
		ubyte* fileBuf = new ubyte[maxSidtuneFileLen];
		if (fileBuf == 0)
		{
			// If we are out of memory at this point,
			// it is very unlikely that anything else works.
			QMessageBox error;
			error.setIcon(QMessageBox::Critical);
			error.setText("Out of memory.");
			error.adjustSize();
			error.show();
			return;
		}
		udword fileLen = 0;
		ubyte datb;
		while (cin.get(datb) && fileLen<maxSidtuneFileLen)
			fileBuf[fileLen++] = datb;
		loadSucceeded = mySidTune->load(fileBuf,fileLen);
		delete[] fileBuf;
	}
	else
	{
		loadSucceeded = mySidTune->open(fileName);
	}

	mySidTune->getInfo(mySidTuneInfo);
    
	if (!loadSucceeded)
    {
		buttonsOff();
		clearSidTuneInfo();
		
		infoField->setText(mySidTuneInfo.statusString);
        // With too many errors one has to click too much.
        //
		//QMessageBox error;
		//error.setIcon(QMessageBox::Warning);
		//error.setText(mySidTuneInfo.statusString);
		//error.adjustSize();
		//error.show();
    }
	else
    {
		if ( mySidTune->getStatus() )
		{
			if (commandLineStartSong != 0)
				sidEmuInitializeSong(*myEmuEngine,*mySidTune,commandLineStartSong);
			else
				sidEmuInitializeSong(*myEmuEngine,*mySidTune,mySidTuneInfo.startSong);
            updateSubSongSlider();
			mySidTune->getInfo(mySidTuneInfo);
            if (!isHistoryFile)
            {
                QString tmp = mySidTuneInfo.nameString;
                tmp += " (";
                tmp += mySidTuneInfo.authorString;
                tmp += ")";
                myHistoryDlg->add(fileName,tmp);
            }
			isReadyToPlay = true;
			startPlaying();
            if (playbackState == IS_PLAYING)
                buttonsForceInitialState();
		}
    }
}

void MainDialog::resetTimeDisplay()
{
    myEmuEngine->resetSecondsThisSong();
    timeLCD->resetDisplay();
}

void MainDialog::timerJob()
{
    static bool isActive = false;  // semaphore
    if (!isActive)
    {
        isActive = true;
        if (playbackState==IS_PLAYING ||
            playbackState==IS_FORWARDING)
		{
			int secs = myEmuEngine->getSecondsThisSong();

			// Multi-buffering system unavailable.
			//
			// Due to a bug in the OSS sound driver in Linux 2.0.x it is
            // impossible to determine the current playback position in an
            // accurate way. Normally, this would allow synchronization of
            // audio and graphics output.
            //
            // Let's don't care about this...
			
			sidEmuFillBuffer(*myEmuEngine,*mySidTune,pBuffer[currentBuffer],multiBufferSize);
			myAudio.play(pBuffer[currentBuffer],multiBufferSize);
			bufferFlag[currentBuffer] = true;
			lastBufferCount += multiBufferSize;

//				bufferCount[nextBuffer] = lastBufferCount;
//				if (++nextBuffer >= buffers)
//					nextBuffer = 0;
//				if (++currentBuffer >= buffers)
//					currentBuffer = 0;
			
			timeLCD->updateDisplay(secs);
			
			if (myWaveViewDlg->isVisible())
			{
				if ( bufferFlag[currentBuffer] )
				{
					bufferFlag[currentBuffer] = false;
					myWaveViewDlg->paintWaveformFunc(pBuffer[currentBuffer]);
				}
			}
		}
		isActive = false;
    }
}

bool MainDialog::openAudioDriver()
{
    closeAudioDriver();
    if (!myAudio.open(myAudioDlg->getConfig()))
    {
        driverIsOpen = false;
        //QMessageBox error;
        //error.setIcon(QMessageBox::Warning);
        //error.setText(myAudio.getErrorString());
        //error.adjustSize();
        //error.show();
        // Below looks better.
		infoField->setText(myAudio.getErrorString());
    }
    else
    {
		driverIsOpen = true;
    }
	return driverIsOpen;
}

void MainDialog::closeAudioDriver()
{
    if (driverIsOpen)
    {
        myAudio.close();
        driverIsOpen = false;
    }
}

void MainDialog::stopPlaying()
{
    playbackState = IS_STOPPED;
    playbackTimer.stop();
    myAudio.reset();
    closeAudioDriver();
}

void MainDialog::pausePlaying()
{
    playbackState = IS_PAUSED;
    myAudio.reset();
}

void MainDialog::resumePlaying()
{
    playbackState = IS_PLAYING;
}

void MainDialog::startPlaying()
{
    stopPlaying();
    resetTimeDisplay();
	
	if (!openAudioDriver())
    {
        buttonsOff();
    }
    else
    {
		sidEmuInitializeSong(*myEmuEngine,*mySidTune,mySidTuneInfo.currentSong);
		extern bool sidEmuFastForwardReplay(int percent);
		sidEmuFastForwardReplay(100);  // reset to normal speed
		if ( !mySidTune->getStatus() )
		{
			buttonsOff();
		}
		else
		{
            updateSubSongSlider();
			setPrevBtnState();
			setNextBtnState();
			showSidTuneInfo();
				
            // preFillBuffers
            currentBuffer = 0;
            nextBuffer = currentBuffer+1;
	
            lastBufferCount = 0;
            for (int i = 0; i < buffers; i++)
            {
                sidEmuFillBuffer(*myEmuEngine,*mySidTune,pBuffer[i],multiBufferSize);
                bufferFlag[i] = true;
                lastBufferCount += multiBufferSize;
                bufferCount[i] = lastBufferCount;
            }
            for (int i = 0; i < buffers; i++)
            {
                myAudio.play(pBuffer[i],multiBufferSize);
            }
            
			// Enable timer callback.
			playbackTimer.changeInterval(0);
			playbackTimer.start(playbackTimer.interval,false);
			playbackState = IS_PLAYING;
		}
    }
}

// Multi-buffering system is not fully implemented.
// See timerJob().

void MainDialog::initAudioBufferSystem()
{
    // Clear multi-buffering system.
    lastBufferCount = 0;
    for (int i = 0; i < maxBuffers; i++)
    {
        bufferFlag[i] = false;
        bufferCount[i] = 0;
        pBuffer[i] = 0;
    }
}

void MainDialog::freeAudioBufferSystem()
{
    for (int i = 0; i < maxBuffers; i++)
    {
        if (pBuffer[i] != 0)
            delete[] pBuffer[i];
    }
    initAudioBufferSystem();
}

void MainDialog::openAudioBufferSystem()
{
    freeAudioBufferSystem();
    
    bool success = true;
    for (int i = 0; i < buffers; i++)
    {
        if ((pBuffer[i] = new ubyte[multiBufferSize]) == 0)
        {
            success = false;
            break;
        }
    }
    if ( !success )
    {
        QMessageBox error;
        error.setIcon(QMessageBox::Critical);
        error.setText("Out of memory.");
        error.adjustSize();
        error.show();
	}
}
