/*
 * aflog.c
 *
 * This module supports ordinary file channel for appending, no truncation
 *
 * $Id: aflog.c,v 1.5 1996/08/27 23:10:14 ron Exp $
 *
 */
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include "afos.h"

int log_init ( AF_CHANNEL *chan, void (*read_callback)() ) ;
int log_write ( AF_CHANNEL *chan, unsigned char *dat, int size ) ;
int log_getchar ( AF_CHANNEL *chan ) ;

typedef struct _LOG_STRUCT_ {
	char file_name[AFCHAN_NAME_SIZE];
	} LOG_STRUCT ;

/*
	parse_log parses out the 'log' line in the config file
*/
int
parse_log( char *line )
{
	static char args[8][80] ;
	AF_CHANNEL *chan ;
	LOG_STRUCT *mystuff ;
	int count ;

	count = sscanf ( line,  "%[^ 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]",
		args[0], args[1], args[2]) ;

	if ( count < 3 )
	{
		fprintf ( stderr, "\r\nAFOSD: Unable to parse the following line:\n%s\n"
			"Insufficient parameters for log definition\n"
			"Usage:\n"
	"log <CHANNEL NAME> <FILE NAME> \n", line);
		exit (-1);
	}

	if (!verify_unique_portname( args[1] ) )
	{
		fprintf(stderr,"\r\nERROR in port configuration file on LINE:\n%s\n"
					   "Port '%s' already defined\n",
						line, args[1] ) ;
		exit (-1);
	}

	chan = (AF_CHANNEL *)malloc( sizeof(*chan) ) ;
	if (!chan)
	{
		fprintf(stderr, "Unable to allocate new channel definition space\n");
		exit (-1);
	}
	
	/* Put new channel on the channel list */
	chan->next = channel_list ;
	channel_list = chan ;
	strncpy ( chan->name, args[1], AFCHAN_NAME_SIZE ) ;

	/* Allocate our private data structure and fill it out with our
	   configuration info */
	mystuff = (LOG_STRUCT *)malloc( sizeof(*mystuff) ) ;
	if (!mystuff)
	{
		fprintf(stderr, "Unable to allocate new channel definition space\n");
		exit (-1);
	}
	strncpy ( mystuff->file_name, args[2], AFCHAN_NAME_SIZE ) ;

	chan->private = mystuff ;

	/* Fill out the rest of the channel info */
	chan->fd = -1 ;
	chan->init = log_init ;
	chan->chanwrite = log_write ;
	chan->changetchar = log_getchar ;
#if 1
	printf( "config line: '%s'\n"
			"channel name: '%s'\n"
			"file name: '%s'\n",
			line, chan->name, mystuff->file_name ) ;
#endif
}


/*
	log_init initializes a file channel
*/
int
log_init ( AF_CHANNEL *chan, void (*read_callback)() )
{
	LOG_STRUCT *mystuff ;

	mystuff = chan->private ;

	/* Check to see if previously initialized */
	if ( chan->fd != -1 )
	{
		/* Already initialized, just return */
		return 0 ;
	}

	/* 
		open the file for creation, append
	*/
	chan->fd = open ( mystuff->file_name, O_CREAT|O_WRONLY|O_APPEND, 0664 ) ;
	if (chan->fd < 0)
	{
		fprintf(stderr, "AFOSD: Unable to open file '%s' for channel '%s'\n",
			mystuff->file_name, chan->name ) ;
		return -1 ;
	}

}

/*
	file_write writes a message out to the file channel
*/
int
log_write ( AF_CHANNEL *chan, unsigned char *dat, int size )
{
	int ret ;

	chan->write_count++ ;
	chan->bytes_written += size ;
	ret = write ( chan->fd, dat, size ) ;

	return ret ;
}

/*
	log_getchar reads one character from the file channel.  This call is
	non-blocking and may return a negative value if an error occurs
*/
int
log_getchar ( AF_CHANNEL *chan )
{
	int i, result;

	i = 0 ;
	result = read ( chan->fd, &i, 1 ) ;

	if ( result <= 0 )
		return -1 ;

	return 0xff & i ;
}
