/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* ******************************************************************** */
/* ***** HeightFieldModel is based on code from Jonathan Belson's ***** */
/* *********************** landscape package. ************************* */
/* ******************************************************************** */


#include <stdio.h>
#include "HeightFieldModel.h"
#include "GlobalTrace.h"
#include "GlobalSanityCheck.h"
#include "GuiColormapLinear.h"
#include "GuiColormapBands.h"
#include "MathTrig.h"


/*
 *  constructor: initialize
 */
HeightFieldModel::HeightFieldModel (HeightField *HF)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightFieldModel\n");

	SanityCheck::bailout ((!HF), "HF==NULL", "HeightFieldModel::HeightFieldModel");
	//SanityCheck::bailout ((!HF->d_hf), "HF->d_hf==NULL", "HeightFieldModel::HeightFieldModel");

	p_HF = HF;
	p_cMap = NULL;
	d_xsize = d_ysize = d_size = 0;
	d_xpos = 0;	
	d_ypos = -((p_HF->getWidth()+p_HF->getHeight())/2)*0.5;
	d_zpos = p_HF->getWidth()*1.25;
	d_xrot = d_yrot = d_zrot = 0.0;
	d_wRes = 1;
	b_sea = FALSE;

	d_tModel2D.data = NULL;
	d_tModel2D.tri = NULL;
	d_tModel3D.data = NULL;
	d_tModel3D.tri = NULL;

}


/*
 *  destructor: nothing to clean up
 */
HeightFieldModel::~HeightFieldModel ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- HeightFieldModel\n");
	if (d_tModel2D.data)
		delete [] d_tModel2D.data;
	if (d_tModel2D.tri)
		delete [] d_tModel2D.tri;
	if (d_tModel3D.data)
		delete [] d_tModel3D.data;
	if (d_tModel3D.tri)
		delete [] d_tModel3D.tri;

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "--- HeightFieldModel\n");
}


/* 
 * refresh: refresh the data model after the height field has changed
 */
void HeightFieldModel::refresh (GuiColormap *cMap, int wireResolution, 
				float yScale, bool sea, int elvMode, 
				float scale=1)
{
	SanityCheck::bailout ((!cMap), "cMap==NULL", "HeightFieldModel::refresh");
	SanityCheck::bailout ((!wireResolution), "wireResolution==0", "HeightFieldModel::refresh");
	SanityCheck::bailout ((yScale<=0), "yScale<=0", "HeightFieldModel::refresh");
	SanityCheck::bailout ((yScale>1), "yScale>1", "HeightFieldModel::refresh");

	p_cMap = cMap;
	d_wRes = wireResolution;
	d_yScale = yScale;
	b_sea = sea;
	buildModel ();
	transformD32 ();
}


/*
 * get2DPoints: return the 3 triangle points for a given index
 */
int HeightFieldModel::get2DPoints (int i, PT2 *p1, PT2 *p2, PT2 *p3, int *cIndex)
{
	SanityCheck::bailout ((!p1), "p1==NULL", "HeightFieldModel::get2DPoints");
	SanityCheck::bailout ((!p2), "p2==NULL", "HeightFieldModel::get2DPoints");
	SanityCheck::bailout ((!p3), "p3==NULL", "HeightFieldModel::get2DPoints");

	if (SanityCheck::warning ((!(i<d_tModel2D.nTri)), "triangle array out of bounds", "HeightFieldModel::get2DPoints"))
		return -1;

	p1->x = d_tModel2D.data[d_tModel2D.tri[i].v1].x;
	p1->y = d_tModel2D.data[d_tModel2D.tri[i].v1].y;
	p2->x = d_tModel2D.data[d_tModel2D.tri[i].v2].x;
	p2->y = d_tModel2D.data[d_tModel2D.tri[i].v2].y;
	p3->x = d_tModel2D.data[d_tModel2D.tri[i].v3].x;
	p3->y = d_tModel2D.data[d_tModel2D.tri[i].v3].y;
	if (cIndex)
		*cIndex = d_tModel2D.tri[i].cIndex;

	return 0;
}


/*
 * buildModel: build a model of triangles from the original height field
 */
int HeightFieldModel::buildModel ()
{
	int	x, y, idx=0, src = 0;
	char	buf[80];


	// set model scale with with regard to wirescale resolution
	d_xsize = p_HF->getWidth()/d_wRes;
	d_ysize = p_HF->getHeight()/d_wRes;
	d_size = d_xsize*d_ysize;
	d_scale = 1;

	sprintf (buf, "Building Model (%d, %d)...\n", d_xsize, d_ysize);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	if (d_tModel3D.data)
		delete [] d_tModel3D.data;
	d_tModel3D.data = new PT3[d_size]; 
	d_tModel3D.xsize = d_xsize;
	d_tModel3D.ysize = d_ysize;

	if (d_tModel3D.tri)
		delete [] d_tModel3D.tri;
	d_tModel3D.tri = new TriVert[d_size*2];
	d_tModel3D.nTri = 0;

	transform (0.0, 0.0, 0.0);

	idx = 0;
	// now fill in triangle structures
	for (y=0; y<d_ysize-1; y++) 
	    {
	    for (x=0; x<d_xsize-1; x++) // Don't want to split triangle across landscape 
		{
		unsigned int	col=0; //r, g, b;
		float 		elv;
		TriVert		*tri = NULL; 

		//
		// First triangle...
		//
		tri = &d_tModel3D.tri[idx];
		tri->v1 = src;
		tri->v2 = src + 1;
		tri->v3 = src + d_xsize + 1;

		MathTrig::normalVector (d_tModel3D, tri);

		elv = d_tModel3D.data[tri->v1].y;
		if (elv < d_tModel3D.data[tri->v2].y) 
			elv = d_tModel3D.data[tri->v2].y;
		if (elv < d_tModel3D.data[tri->v3].y) 
			elv = d_tModel3D.data[tri->v3].y;

		col = getColorIndex (elv);
		tri->cIndex = col;
		d_tModel3D.nTri++;
		idx++;

		//
		// Second triangle...
		//
		tri = &d_tModel3D.tri[idx];
		tri->v1 = src;
		tri->v2 = src + d_xsize + 1;
		tri->v3 = src + d_xsize;

		MathTrig::normalVector(d_tModel3D, tri);

		elv = d_tModel3D.data[tri->v1].y;
		if (elv < d_tModel3D.data[tri->v2].y) 
			elv = d_tModel3D.data[tri->v2].y;
		if (elv < d_tModel3D.data[tri->v3].y) 
			elv = d_tModel3D.data[tri->v3].y;

		col = getColorIndex (elv);
		tri->cIndex = col;
		d_tModel3D.nTri++;
		idx++;
		src++;
		if (x == d_xsize-2)	// adjust for row we don't process
			src++;
		}
	    }

	return 0;
}



/*
 * getColorIndex: get the appropriate color index for elv and p_cMap
 */
int HeightFieldModel::getColorIndex (float elv)
{
	int		ncolors, nbands=-1, lboff=-1, lbands=-1, cbs=-1, 
			col=-1;
	float		loff;
	GuiColormapBands *CMB=NULL;

	
	ncolors = p_cMap->getColorCount();
	loff = 0.25;
	if (!p_cMap->d_linear)
		{
		CMB = static_cast<GuiColormapBands*>(p_cMap);
		nbands = CMB->getNumBands ();
		lboff = CMB->getLandBandOffset ();
		lbands = nbands - lboff;		// # of land bands
		cbs = CMB->getBandSize ();
		}

	if (p_cMap->d_linear)
		{
		if (b_sea)
			col = (int)( (ncolors*(1-loff)) *
				     ((elv-d_min)/d_range) +
				     ncolors*loff );
		else
			col = (int)(ncolors*((elv-d_min)/d_range));
		}
	else
		{
		if (b_sea)
			col = (int)( ((nbands-(lboff-1)) * 
				     ((elv-d_min)/d_range) +
				     (lboff-1)) * cbs );
			//col = (int)((ncolors-((lbands-1)*cbs)) *
			//	    ((elv-d_min)/d_range) +
			//	    ((lbands-1)*cbs) );
		else
			col = (int)(nbands*((elv-d_min)/d_range)*cbs);
		}

	return col;
}


/*
 * transformD32: transform 3-d array into 2-d
 */
int HeightFieldModel::transformD32 ()
{
	int 	maxx = d_xsize/d_scale,
		maxy = d_ysize/d_scale,
		i=0;

	SanityCheck::bailout ((!d_scale), "scale==0", "HeightFieldModel::buildModel");

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Transform 3D to 2D ...\n");

	if (d_tModel2D.data)
		delete [] d_tModel2D.data;
	d_tModel2D.data = new PTD2[maxx*maxy];
	d_tModel2D.nTri = 0;

	// Transform all the points into 2d...
	for (int y=0; y<maxy; y++) 
	    for (int x=0; x<maxx; x++) 
		{
		d_tModel2D.data[i].x = d_tModel3D.data[i].x /
				      (d_tModel3D.data[i].z/(d_xsize*d_wRes/d_scale));
		d_tModel2D.data[i].y = d_tModel3D.data[i].y /
				      (d_tModel3D.data[i].z/(d_ysize*d_wRes/d_scale));
		d_tModel2D.data[i].depth = d_tModel3D.data[i].z;
		i++;
		}

	// Copy triangle structures...
	if (d_tModel2D.tri)
		delete [] d_tModel2D.tri;
	d_tModel2D.tri = new TriVert[d_tModel3D.nTri];
	for (i=0; i<d_tModel3D.nTri; i++) 
		d_tModel2D.tri[i] = d_tModel3D.tri[i];	// Copy by assignment 

	d_tModel2D.nTri = d_tModel3D.nTri;
	return 0;
}


/*
 * transform: transform the terrain object from original data according to
 *  	      current rotation/translation values
 */
void HeightFieldModel::transform (float a, float b, float c)
{
	int	sx, sy;
	float	sf = ((d_xsize+d_ysize)/2)*d_yScale;
	char	buf[80];

	sprintf (buf, "Transform (%f, %f, %f) ...\n", a, b, c);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	// Update terrain's local rotation
	d_xrot += a;
	d_yrot += b;
	d_zrot += c;

	// Repopulate model structure with data from height data
	int x, y, tx, ty, idx=0;
	for (y=0, sy=0, ty=d_ysize/2; y<d_ysize; y++, sy+=d_wRes, ty--)
	    for (x=0, sx=0, tx=-d_xsize/2; x<d_xsize; x++, sx+=d_wRes, tx++)
		{
		d_tModel3D.data[idx].x = tx*d_wRes;
		if (b_sea)
			d_tModel3D.data[idx].y = p_HF->getElSea(sx,sy)*sf*d_wRes;	
		else
			d_tModel3D.data[idx].y = p_HF->getEl(sx,sy)*sf*d_wRes;	
		d_tModel3D.data[idx].z = ty*d_wRes;
		//printf ("%d, %d: %d, %d, %f\n", tx, ty, sx, sy, p_HF->getEl(sx,sy)*sf*d_wRes);
		idx++;
		}

	// Apply transformation
	rotateY (d_yrot);
	rotateZ (d_zrot);
	translate (d_xpos, d_ypos, d_zpos);

	// Update surface normals
	for (int i=0; i<d_tModel3D.nTri; i++) 
		MathTrig::normalVector(d_tModel3D, &d_tModel3D.tri[i]);
}


/*
 * rotateY: rotate terrain object bout Y axis.
 */
void HeightFieldModel::rotateY (float yrot)
{
	float 	oldx, oldz;
	char	buf[80];

	if (!yrot%360)
		return;

	sprintf (buf, "Rotating Y (%f) ...\n", yrot);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	// Rotate about y axis
	for (int i=0; i<d_size; i++) 
		{
		oldx = d_tModel3D.data[i].x;
		oldz = d_tModel3D.data[i].z;
		d_tModel3D.data[i].x=oldx*cos(yrot)+oldz*sin(yrot);
		d_tModel3D.data[i].z=-oldx*sin(yrot)+oldz*cos(yrot);
		}
}


/*
 * rotateZ: rotate terrain object bout Y axis.
 */
void HeightFieldModel::rotateZ (float zrot)
{
	float 	oldy, oldz;
	char	buf[80];

	if (!zrot%360)
		return;

	sprintf (buf, "Rotating Z (%f) ...\n", zrot);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	// Rotate about y axis
	for (int i=0; i<d_size; i++) 
		{
		oldy = d_tModel3D.data[i].y;
		oldz = d_tModel3D.data[i].z;
		d_tModel3D.data[i].y=oldy*cos(zrot)+oldz*sin(zrot);
		d_tModel3D.data[i].z=-oldy*sin(zrot)+oldz*cos(zrot);
		}
}


/*
 * translate: translate terrain object by (x, y, z), leveling off at sea level
 */
void HeightFieldModel::translate (float x, float y, float z)
{
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Translate ...\n");

	d_min = 1000000000;
	d_max = -1000000000;
	for (int i=0; i<d_size; i++) 
		{
		d_tModel3D.data[i].x += x;
		d_tModel3D.data[i].y += y;
		d_tModel3D.data[i].z += z;

		if (d_tModel3D.data[i].y < d_min)
			d_min = d_tModel3D.data[i].y;
		if (d_tModel3D.data[i].y > d_max)
			d_max = d_tModel3D.data[i].y;	
		}
	d_range = (d_max-d_min)*1.01;		// make sure we don't overflow
}

