/********************************************************************************
 * Copyright (c) Des Herriott 1993, 1994
 *               Erik Kunze   1995, 1996, 1997
 *
 * Permission to use, distribute, and sell this software and its documentation
 * for any purpose is hereby granted without fee, provided that the above
 * copyright notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that the name
 * of the copyright holder not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.  The
 * copyright holder makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or implied
 * warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
 *
 * THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Des Herriott
 *         Erik Kunze
 *
 * changed by EKU
 *******************************************************************************/
#ifndef lint
static char rcsid[] = "$Id: screen.c,v 3.30 1997/09/13 17:00:04 erik Rel $";
#endif

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#ifdef MITSHM
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
#if defined(nec_ews) && !defined(__STDC__)
void *shmat(int, void*, int);
#endif
#endif
#ifdef OFFIX_DND
#include <OffiX/DragAndDropTypes.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "z80.h"
#ifdef LINE_VIDEO
#include "emul.h"
#endif
#include "debug.h"
#include "resource.h"
#include "mem.h"
#include "io.h"
#include "util.h"
#include "dialog.h"
#include "main.h"
#include "menu.h"
#include "monitor.h"
#include "snapshot.h"
#include "tables.h"
#include "screen.h"
#include "xzx.icon"

#define shade(x,inkpap,paper) \
						(((x) & (inkpap)) ^ (paper))

#define XYToAttr(x,y)	(((x) >> 3) + ((y & 0xf8) << 2))

#define	EXPAND_BOX	{ \
						if (x < xMin) xMin = x;        \
						x += (GETCFG(scale) << 3) - 1; \
						if (x > xMax) xMax = x;        \
						if (y < yMin) yMin = y;        \
						y += GETCFG(scale) - 1;        \
						if (y > yMax) yMax = y;        \
						screenDirty++;                 \
					}

#define MAXIMIZE_BOX	{ \
						xMin = yMin = 0;  \
						xMax = hsize - 1; \
						yMax = vsize - 1; \
						screenDirty++;    \
					}

#define MINIMIZE_BOX	{ \
						xMax = yMax = 0;  \
						xMin = hsize - 1; \
						yMin = vsize - 1; \
						screenDirty = 0;  \
					}

#ifndef EXCLUDE_EXOTIC
static void initScaleupTable(void);
#endif
#ifdef LINE_VIDEO
static void scrWriteXim(uns16, uns8, uns8);
#else
static void scrWriteXim(uns16, uns8);
#endif
#ifndef EXCLUDE_PIXMAP
#ifdef LINE_VIDEO
static void scrWritePix(uns16, uns8, uns8);
#else
static void scrWritePix(uns16, uns8);
#endif
#endif
#ifdef MITSHM
static void scrRefreshXshm(void);
#endif
static void scrRefreshXim(void);
#ifndef EXCLUDE_PIXMAP
static void scrRefreshPix(void);
#endif
static void quitX(void);
static void processKeypress(XKeyEvent *);
static void processKeyrelease(XKeyEvent *);
static void notifyWm(int *, char **);
static void calcWinPosition(int *, int *);
#ifndef EXCLUDE_PIXMAP
static int initPixmap(void);
#endif
static int initImage(void);
static int getColors(unsigned long *);

Display *Dpy;
void (*ScreenRefresh)(void);


static Screen *sptr;
static int scrn;
static Window root, mainWin, borderWin;
static int white, black;
static Colormap cmap;
static GC theGc;
static XImage *xim;
#ifndef EXCLUDE_PIXMAP
static GC fgGc, bgGc;
static Pixmap thePix;
#endif
#ifdef MITSHM
static XShmSegmentInfo xshminfo;
#endif
static int colorDepth, bytesPerLine, bitsPerPixel;
static int hsize, vsize;
static void **ximAddrTable;
static char scrChanged[PIXEL_LENGTH];
#ifdef LINE_VIDEO

static char scrChangedByte[PIXEL_LENGTH];

static char scrChangedAttr[PIXEL_LENGTH];

static char scrChangedDelayed[PIXEL_LENGTH];
static char scrChangedDelayedByte[PIXEL_LENGTH];
static char scrChangedDelayedAttr[PIXEL_LENGTH];
static int screenDirtyDelayed;
static void (*screenWrite)(uns16, uns8, uns8);
#else
static void (*screenWrite)(uns16, uns8);
#endif
static Atom deleteAtom;
#ifdef OFFIX_DND
static Atom dndProtocol;
static Atom dndSelection;
#endif

static struct {
	int port;
	int bit;
} keyTable[128] = {
	{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},
	{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},
	{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},
	{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00},

	{7,0xfe},{3,0xfe},{5,0xfe},{3,0xfb},{3,0xf7},{3,0xef},{4,0xef},{4,0xf7},

	{4,0xfb},{4,0xfd},{7,0xef},{6,0xfb},{7,0xf7},{6,0xf7},{7,0xfb},{0,0xef},

	{4,0xfe},{3,0xfe},{3,0xfd},{3,0xfb},{3,0xf7},{3,0xef},{4,0xef},{4,0xf7},

	{4,0xfb},{4,0xfd},{0,0xfd},{5,0xfd},{2,0xf7},{6,0xfd},{2,0xef},{0,0xf7},

	{3,0xfd},{1,0xfe},{7,0xef},{0,0xf7},{1,0xfb},{2,0xfb},{1,0xf7},{1,0xef},

	{6,0xef},{5,0xfb},{6,0xf7},{6,0xfb},{6,0xfd},{7,0xfb},{7,0xf7},{5,0xfd},

	{5,0xfe},{2,0xfe},{2,0xf7},{1,0xfd},{2,0xef},{5,0xf7},{0,0xef},{2,0xfd},

	{0,0xfb},{5,0xef},{0,0xfd},{8,0x00},{8,0x00},{8,0x00},{6,0xef},{4,0xfe},

	{0,0xfb},{1,0xfe},{7,0xef},{0,0xf7},{1,0xfb},{2,0xfb},{1,0xf7},{1,0xef},

	{6,0xef},{5,0xfb},{6,0xf7},{6,0xfb},{6,0xfd},{7,0xfb},{7,0xf7},{5,0xfd},

	{5,0xfe},{2,0xfe},{2,0xf7},{1,0xfd},{2,0xef},{5,0xf7},{0,0xef},{2,0xfd},

	{0,0xfb},{5,0xef},{0,0xfd},{8,0x00},{8,0x00},{8,0x00},{8,0x00},{8,0x00}
};
static int unshiftCursor = 0;
static int emulatedShift = 0;

static unsigned short rgbVals[NCOLORS][3] = {
	{ 0x0000, 0x0000, 0x0000 },
	{ 0x0000, 0x0000, 0xbfff },
	{ 0xbfff, 0x0000, 0x0000 },
	{ 0xbfff, 0x0000, 0xbfff },
	{ 0x0000, 0xbfff, 0x0000 },
	{ 0x0000, 0xbfff, 0xbfff },
	{ 0xbfff, 0xbfff, 0x0000 },
	{ 0xbfff, 0xbfff, 0xbfff },
	{ 0x0000, 0x0000, 0x0000 },
	{ 0x0000, 0x0000, 0xffff },
	{ 0xffff, 0x0000, 0x0000 },
	{ 0xffff, 0x0000, 0xffff },
	{ 0x0000, 0xffff, 0x0000 },
	{ 0x0000, 0xffff, 0xffff },
	{ 0xffff, 0xffff, 0x0000 },
	{ 0xffff, 0xffff, 0xffff },
};
static unsigned long colors[NCOLORS];
static unsigned long ink[FLASH];
static unsigned long paper[FLASH];
static int xMin, xMax, yMin, yMax;
static int screenDirty;
static int flashPhase;
#ifndef EXCLUDE_EXOTIC

static long scaleup[256];
static uns8 halftone[NCOLORS][4] = {
	{ 0xff, 0xff, 0xff, 0xff },
	{ 0xee, 0xff, 0xbb, 0xff },
	{ 0x55, 0xff, 0xaa, 0xff },
	{ 0x55, 0xbb, 0x55, 0xff },
	{ 0xaa, 0x55, 0xaa, 0x55 },
	{ 0x51, 0xaa, 0x15, 0xaa },
	{ 0xaa, 0x44, 0xaa, 0x00 },
	{ 0x11, 0x00, 0x44, 0x00 },
	{ 0xff, 0xff, 0xff, 0xff },
	{ 0x77, 0xdf, 0x77, 0xfd },
	{ 0xaa, 0xdd, 0xaa, 0x77 },
	{ 0xae, 0x55, 0xea, 0x55 },
	{ 0x55, 0x22, 0x55, 0x88 },
	{ 0x88, 0x22, 0x88, 0x22 },
	{ 0x00, 0x40, 0x00, 0x04 },
	{ 0x00, 0x00, 0x00, 0x00 }
};
static int wantColor;
#endif

#ifndef EXCLUDE_EXOTIC
static void
initScaleupTable(void)
{
	int j, k, l, m;
	unsigned long bigval;
	for(l = 0; l < 256; l++)
	{
		for (m = l, bigval = j = 0; j < 8; j++)
		{
			for(k = GETCFG(scale); k > 0 ; k--)
			{
				if (BitmapBitOrder(Dpy) == LSBFirst)
				{
					bigval = (bigval << 1) | (m & 1);
				}
				else
				{
					bigval = (bigval >> 1) | ((m & 1) << 31);
				}
			}
			m >>= 1;
		}
		if (ImageByteOrder(Dpy) == MSBFirst)
		{
			if (BitmapBitOrder(Dpy) == MSBFirst)
			{
				bigval >>= 32 - (GETCFG(scale) << 3);
			}
			else
			{
				bigval <<= 32 - (GETCFG(scale) << 3);
			}
		}
		scaleup[l] = bigval;
	}

	if (white)
	{
		for (j = 0; j < NCOLORS; j++)
		{
			for(k = 0; k < 4; k++)
			{
				halftone[j][k] ^= 255;
			}
		}
	}
}
#endif

static void
#ifdef LINE_VIDEO
scrWriteXim(uns16 addr, uns8 val, uns8 attr)
#else
scrWriteXim(uns16 addr, uns8 val)
#endif
{
	int x, y;
#ifndef LINE_VIDEO
	uns8 attr;
#endif
	assert(addr < (PIXEL_LENGTH + ATTR_LENGTH));
	x = DtblX[addr];
	y = DtblY[addr];
#ifndef LINE_VIDEO
	attr = RealMemory[ScreenSelect][ATTR_OFFSET + XYToAttr(x,y)];
#endif
	attr = flashPhase && attr & FLASH ? Reverse[attr & ~FLASH] : attr & ~FLASH;
	x *= GETCFG(scale);
	y *= GETCFG(scale);
#ifndef EXCLUDE_EXOTIC
	if (bitsPerPixel >= 8)
#endif
	{
		unsigned long ipix, ppix;
		int j, k;
		ipix = ink[attr];
		ppix = paper[attr];
		switch(colorDepth)
		{
			case 8:
				{
					uns8 *ximptr = (uns8 *)ximAddrTable[addr];
					for (j = 0; j < 8; j++)
					{
						for (k = 0; k < GETCFG(scale); k++)
						{
							*ximptr++ = (uns8)(val & 0x80 ? ipix : ppix);
						}
						val <<= 1;
					}
				}
				break;
			case 16:
				{
					uns16 *ximptr = (uns16 *)ximAddrTable[addr];
					for (j = 0; j < 8; j++)
					{
						for (k = 0; k < GETCFG(scale); k++)
						{
							*ximptr++ = (uns16)(val & 0x80 ? ipix : ppix);
						}
						val <<= 1;
					}
				}
				break;
			case 32:
				{
					long *ximptr = (long *)ximAddrTable[addr];
					for (j = 0; j < 8; j++)
					{
						for (k = 0; k < GETCFG(scale); k++)
						{
							*ximptr++ = val & 0x80 ? ipix : ppix;
						}
						val <<= 1;
					}
				}
				break;
		}
		for (j = 1; j < GETCFG(scale); j++)
		{
			(void)memcpy((char *)ximAddrTable[addr] + j * bytesPerLine,
						 ximAddrTable[addr],
						 bitsPerPixel * GETCFG(scale));
		}
	}
#ifndef EXCLUDE_EXOTIC
	else
	{
		assert(bitsPerPixel == 1);
		if (GETCFG(scale) == 1)
		{
			val = (uns8)scaleup[val];
			*(uns8 *)ximAddrTable[addr] = (ink[attr] ? val : 0) |
										  (paper[attr] ? val ^ 0xff : 0);
		}
		else
		{
			long bigval;
			uns8 *ximptr;
			uns8 pmask, ipmask;
			int icol, pcol;
			int j, k;
			bigval = scaleup[val];
			ximptr = (uns8 *)ximAddrTable[addr];
			k = y & 0x03;
			if (wantColor)
			{
				icol = (attr & INK) | ((attr & BRIGHT) >> 3);
				pcol = ((attr & PAPER) >> 3) | ((attr & BRIGHT) >> 3);
			}
			else
			{
				pmask = paper[attr] * 0xff;
				ipmask = (ink[attr] * 0xff) ^ pmask;
			}
			for (j = 0; j < GETCFG(scale); j++)
			{
				if (wantColor)
				{
					pmask  = halftone[pcol][k];
					ipmask = halftone[icol][k] ^ pmask;
				}

				if (ImageByteOrder(Dpy) == BitmapBitOrder(Dpy))
				{
					ximptr[0] = shade(bigval, ipmask, pmask);
					ximptr[1] = shade(bigval >> 8, ipmask, pmask);
					if (GETCFG(scale) > 2)
					{
						ximptr[2] = shade(bigval >> 16, ipmask, pmask);
					}
				}
				else
				{
					ximptr[0] = shade(bigval >> 24, ipmask, pmask);
					ximptr[1] = shade(bigval >> 16, ipmask, pmask);
					if (GETCFG(scale) > 2)
					{
						ximptr[2] = shade(bigval >> 8, ipmask, pmask);
					}
				}
				ximptr += GETCFG(scale) << 5;
				k = ++k & 0x03;
			}
		}
	}
#endif
	EXPAND_BOX;
}

#ifndef EXCLUDE_PIXMAP
static void
#ifdef LINE_VIDEO
scrWritePix(uns16 addr, uns8 val, uns8 attr)
#else
scrWritePix(uns16 addr, uns8 val)
#endif
{
	int x, x2, y;
	unsigned long ipix, ppix;
#ifndef LINE_VIDEO
	uns8 attr;
#endif
	static unsigned long lastIpix, lastPpix;
	int nfg = 0, nbg = 0;
	int i;
	static XPoint fg_pts[8], bg_pts[8];
	static XRectangle fg_recs[8], bg_recs[8];
	assert(addr < (PIXEL_LENGTH + ATTR_LENGTH));
	x = DtblX[addr];
	y = DtblY[addr];
#ifndef LINE_VIDEO
	attr = RealMemory[ScreenSelect][ATTR_OFFSET + XYToAttr(x,y)];
#endif
	attr = flashPhase && attr & FLASH ? Reverse[attr & ~FLASH] : attr & ~FLASH;
	x *= GETCFG(scale);
	y *= GETCFG(scale);
	x2 = x;
	ipix = ink[attr];
	ppix = paper[attr];
	if (GETCFG(scale) == 1)
	{
		for (i = 0x80; i > 0; i >>= 1, x++)
		{
			if (val & i)
			{
				fg_pts[nfg].x = (short)x;
				fg_pts[nfg++].y = (short)y;
			}
			else
			{
				bg_pts[nbg].x = (short)x;
				bg_pts[nbg++].y = (short)y;
			}
		}
		if (nfg)
		{
			if (ipix != lastIpix)
			{
				XSetForeground(Dpy, fgGc, ipix);
				lastIpix = ipix;
			}
			XDrawPoints(Dpy, thePix, fgGc, fg_pts, nfg, CoordModeOrigin);
		}
		if (nbg)
		{
			if (ppix != lastPpix)
			{
				XSetForeground(Dpy, bgGc, ppix);
				lastPpix = ppix;
			}
			XDrawPoints(Dpy, thePix, bgGc, bg_pts, nbg, CoordModeOrigin);
		}
	}
	else
	{
		for (i = 0x80; i > 0; i >>= 1, x += GETCFG(scale))
		{
			if (val & i)
			{
				fg_recs[nfg].x = (short)x;
				fg_recs[nfg].y = (short)y;
				fg_recs[nfg].width = (unsigned short)GETCFG(scale);
				fg_recs[nfg++].height = (unsigned short)GETCFG(scale);
			}
			else
			{
				bg_recs[nbg].x = (short)x;
				bg_recs[nbg].y = (short)y;
				bg_recs[nbg].width = (unsigned short)GETCFG(scale);
				bg_recs[nbg++].height = (unsigned short)GETCFG(scale);
			}
		}
		if (nfg)
		{
			if (ipix != lastIpix)
			{
				XSetForeground(Dpy, fgGc, ipix);
				lastIpix = ipix;
			}
			XFillRectangles(Dpy, thePix, fgGc, fg_recs, nfg);
		}
		if (nbg)
		{
			if (ppix != lastPpix)
			{
				XSetForeground(Dpy, bgGc, ppix);
				lastPpix = ppix;
			}
			XFillRectangles(Dpy, thePix, bgGc, bg_recs, nbg);
		}
	}
	x = x2;
	EXPAND_BOX;
}
#endif

void
ScreenWrite(uns16 addr, uns8 val)
{
#ifdef LINE_VIDEO
	int x, y;
	uns8 attr;
	assert(addr < (PIXEL_LENGTH + ATTR_LENGTH));
	x = DtblX[addr];
	y = DtblY[addr];
	attr = RealMemory[ScreenSelect][ATTR_OFFSET + XYToAttr(x,y)];
	if (y + TOP_BORDER >= Vline)
	{
		scrChanged[addr] = '\001';
		scrChangedByte[addr] = val;
		scrChangedAttr[addr] = attr;
		screenDirty++;
	}
	else
	{
		scrChangedDelayed[addr] = '\001';
		scrChangedDelayedByte[addr] = val;
		scrChangedDelayedAttr[addr] = attr;
		screenDirtyDelayed++;
	}
#else
	assert(addr < (PIXEL_LENGTH + ATTR_LENGTH));
	scrChanged[addr] = '\001';
	screenDirty++;
#endif
}

#ifdef MITSHM
static void
scrRefreshXshm(void)
{
	if (screenDirty)
	{
		int i;
		for (i = 0; i < PIXEL_LENGTH; i++)
		{
			if (scrChanged[i])
			{
#ifdef LINE_VIDEO
				screenWrite(i, scrChangedByte[i], scrChangedAttr[i]);
#else
				screenWrite(i, RealMemory[ScreenSelect][i]);
#endif
				scrChanged[i] = '\0';
			}
		}
		(void)XShmPutImage(Dpy, mainWin, theGc, xim, xMin, yMin, xMin, yMin,
						   xMax - (xMin - 1), yMax - (yMin - 1), False);
		XFlush(Dpy);
		MINIMIZE_BOX;
	}
}
#endif

static void
scrRefreshXim(void)
{
	if (screenDirty)
	{
		int i;
		for (i = 0; i < PIXEL_LENGTH; i++)
		{
			if (scrChanged[i])
			{
#ifdef LINE_VIDEO
				screenWrite(i, scrChangedByte[i], scrChangedAttr[i]);
#else
				screenWrite(i, RealMemory[ScreenSelect][i]);
#endif
				scrChanged[i] = '\0';
			}
		}
		XPutImage(Dpy, mainWin, theGc, xim, xMin, yMin, xMin, yMin,
				  xMax - (xMin - 1), yMax - (yMin - 1));
		XFlush(Dpy);
		MINIMIZE_BOX;
	}
}

#ifndef EXCLUDE_PIXMAP
static void
scrRefreshPix(void)
{
	if (screenDirty)
	{
		int i;
		for (i = 0; i < PIXEL_LENGTH; i++)
		{
			if (scrChanged[i])
			{
#ifdef LINE_VIDEO
				screenWrite(i, scrChangedByte[i], scrChangedAttr[i]);
#else
				screenWrite(i, RealMemory[ScreenSelect][i]);
#endif
				scrChanged[i] = '\0';
			}
		}
		XCopyArea(Dpy, thePix, mainWin, theGc, xMin, yMin,
				  xMax - (xMin - 1), yMax - (yMin - 1), xMin, yMin);
		XFlush(Dpy);
		MINIMIZE_BOX
	}
}
#endif

#ifdef LINE_VIDEO
void
ScreenUpdateBuffers(void)
{
	int i;
	if (screenDirtyDelayed)
	{
		screenDirtyDelayed = 0;
		for (i = 0; i < PIXEL_LENGTH; i++)
		{
			if (scrChangedDelayed[i])
			{
				scrChanged[i] = '\001';
				scrChangedByte[i] = scrChangedDelayedByte[i];
				scrChangedAttr[i] = scrChangedDelayedAttr[i];
				scrChangedDelayed[i] = '\0';
				screenDirty++;
			}
		}
	}
}
#endif

void
ForceScreenRefresh(void)
{
	int i;
	uns8 *mem = &RealMemory[ScreenSelect][0];
	for (i = 0; i < PIXEL_LENGTH; i++, mem++)
	{
		ScreenWrite(i, *mem);
	}
}

void
AttrWrite(uns16 addr, uns8 val)
{
#ifdef LINE_VIDEO
	int i, line;
	uns8 *mem;
	addr = ((addr & 0x300) << 3) | (addr & 0xff);
	mem = &RealMemory[ScreenSelect][addr];
	for (i = 0, line = DtblY[addr] + TOP_BORDER; i < 8 && line < Vline;
	     i++, line++, addr += X_PIXELS, mem += X_PIXELS)
	{
		scrChangedDelayed[addr] = '\001';
		scrChangedDelayedByte[addr] = *mem;
		scrChangedDelayedAttr[addr] = val;
		screenDirtyDelayed++;
	}
	for (; i < 8; i++, addr += X_PIXELS, mem += X_PIXELS)
	{
		scrChanged[addr] = '\001';
		scrChangedByte[addr] = *mem;
		scrChangedAttr[addr] = val;
		screenDirty++;
	}
#else
	int i;
	uns8 *mem;
	addr = ((addr & 0x300) << 3) | (addr & 0xff);
	mem = &RealMemory[ScreenSelect][addr];
	for (i = 0; i < 8; i++, addr += X_PIXELS, mem += X_PIXELS)
	{
		ScreenWrite(addr, *mem);
	}
#endif
}

void
DoFlashing(void)
{
	uns8 *mem;
	uns16 addr;
	if (GETCFG(flashing))
	{
		flashPhase = !flashPhase;
		addr = ATTR_OFFSET;
		mem = &RealMemory[ScreenSelect][ATTR_OFFSET];
		for (; addr < ATTR_OFFSET + ATTR_LENGTH; addr++, mem++)
		{
			if (*mem & FLASH)
			{
				AttrWrite(addr, *mem);
			}
		}
	}
}

void
SetBorderColor(uns8 color)
{
	XSetWindowBackground(Dpy, borderWin, colors[color]);
	XClearWindow(Dpy, borderWin);
	XFlush(Dpy);
}

static void
quitX(void)
{
#ifndef EXCLUDE_PIXMAP
	if (!GETCFG(pixmaps))
#endif
	{
#ifdef MITSHM
		if (GETCFG(mitshm))
		{
			(void)XShmDetach(Dpy, &xshminfo);
			XDestroyImage(xim);
			(void)shmdt(xshminfo.shmaddr);
			(void)shmctl(xshminfo.shmid, IPC_RMID, 0);
		}
		else
#endif
		{
			XDestroyImage(xim);
		}
	}
	XAutoRepeatOn(Dpy);
	XCloseDisplay(Dpy);
}

static void
processKeypress(XKeyEvent *kev)
{
	KeySym ks, ks0;

	(void)XLookupString(kev, NULL, 0, &ks, NULL);

	ks0 = XLookupKeysym(kev, 0);

	if (ks0 == GETCFG(joyUp))
	{
		InPorts[P_KEMPSTON] |= B_UP;
	}
	else if (ks0 == GETCFG(joyDown))
	{
		InPorts[P_KEMPSTON] |= B_DOWN;
	}
	else if (ks0 == GETCFG(joyLeft))
	{
		InPorts[P_KEMPSTON] |= B_LEFT;
	}
	else if (ks0 == GETCFG(joyRight))
	{
		InPorts[P_KEMPSTON] |= B_RIGHT;
	}
	else if (ks0 == GETCFG(joyFire))
	{
		InPorts[P_KEMPSTON] |= B_FIRE;
	}
#ifdef SLOWDOWN

	else if (ks == GETCFG(slowdownKey))
	{
		++GETCFG(slowdown);
	}
	else if (ks == GETCFG(speedupKey))
	{
		if (GETCFG(slowdown))
		{
			--GETCFG(slowdown);
		}
	}
#endif
	switch (ks) {

		case XK_F1:
			MainMenu(OSD_HELP);
			break;
		case XK_F2:
			WriteSnapshotWithFS();
			break;
		case XK_F3:
			ReadSnapshotWithFS();
			break;
		case XK_F4:
			WriteScreenDumpWithFS();
			break;
		case XK_F5:
			MainMenu(OSD_RESET);
			break;
		case XK_F6:
			MainMenu(OSD_NMI);
 			break;
		case XK_F7:
			Debugger(DBG_FKEY);
			break;
		case XK_F8:
			break;
		case XK_F9:
			HardwareMenu();
			break;
		case XK_F10:
		case XK_F12:
			MainMenu(OSD_QUIT);
			break;
		case XK_Tab:

			unshiftCursor = !unshiftCursor;
			break;
		case XK_backslash:
		case XK_bar:

			(void)memset(KeyPorts, 0xff, sizeof(KeyPorts));
			unshiftCursor = 0;
			emulatedShift = 0;
			break;

		case XK_Return:
			KeyPorts[6] &= 0xfe;
			break;
		case XK_Control_L:
		case XK_Control_R:
			KeyPorts[0] &= 0xfe;
			KeyPorts[7] &= 0xfd;
			emulatedShift++;
			break;
		case XK_Shift_L:
			KeyPorts[0] &= 0xfe;
			emulatedShift++;
			break;
		case XK_Alt_L:
		case XK_Alt_R:
		case XK_Meta_L:
		case XK_Meta_R:
			KeyPorts[7] &= 0xfd;
			emulatedShift++;
			break;
		case XK_BackSpace:
		case XK_Delete:
			KeyPorts[0] &= 0xfe;
			KeyPorts[4] &= 0xfe;
			break;
		case XK_Escape:
			KeyPorts[0] &= 0xfe;
			KeyPorts[3] &= 0xfe;
			break;
		case XK_Up:
			if (!unshiftCursor)
			{
				KeyPorts[0] &= 0xfe;
			}
			KeyPorts[4] &= 0xf7;
			break;
		case XK_Down:
			if (!unshiftCursor)
			{
				KeyPorts[0] &= 0xfe;
			}
			KeyPorts[4] &= 0xef;
			break;
		case XK_Left:
			if (!unshiftCursor)
			{
				KeyPorts[0] &= 0xfe;
			}
			KeyPorts[3] &= 0xef;
			break;
		case XK_Right:
			if (!unshiftCursor)
			{
				KeyPorts[0] &= 0xfe;
			}
			KeyPorts[4] &= 0xfb;
			break;
		case XK_sterling:
			if (!emulatedShift)
			{
				KeyPorts[7] &= 0xfd;
				KeyPorts[0] &= 0xfb;
				break;
			}

		default:

			if (emulatedShift)
			{
				if (ks0 < 128)
				{
					KeyPorts[keyTable[ks0].port] &= keyTable[ks0].bit;
				}
			}
			else if (ks < 128)
			{
				if	(ks >= 'A' && ks <= 'Z')
				{

					KeyPorts[0] &= 0xfe;
				}
				else if (!((ks >= 'a' && ks <= 'z')
						   || (ks >= '0' && ks <= '9')
						   || ks == ' '))
				{

					KeyPorts[7] &= 0xfd;
				}
				KeyPorts[keyTable[ks].port] &= keyTable[ks].bit;
			}
			break;
	}
}

static void
processKeyrelease(XKeyEvent *kev)
{
	KeySym ks, ks0;
	(void)XLookupString(kev, NULL, 0, &ks, NULL);
	ks0 = XLookupKeysym(kev, 0);
	if (ks0 == GETCFG(joyUp))
	{
		InPorts[P_KEMPSTON] &= ~B_UP;
	}
	else if (ks0 == GETCFG(joyDown))
	{
		InPorts[P_KEMPSTON] &= ~B_DOWN;
	}
	else if (ks0 == GETCFG(joyLeft))
	{
		InPorts[P_KEMPSTON] &= ~B_LEFT;
	}
	else if (ks0 == GETCFG(joyRight))
	{
		InPorts[P_KEMPSTON] &= ~B_RIGHT;
	}
	else if (ks0 == GETCFG(joyFire))
	{
		InPorts[P_KEMPSTON] &= ~B_FIRE;
	}
	switch (ks)
	{

		case XK_Return:
			KeyPorts[6] |= 0x01;
			break;
		case XK_Control_L:
		case XK_Control_R:
			KeyPorts[0] |= 0x01;
			KeyPorts[7] |= 0x02;
			if (emulatedShift)
			{
				emulatedShift--;
			}
			break;
		case XK_Shift_L:
			KeyPorts[0] |= 0x01;
			if (emulatedShift)
			{
				emulatedShift--;
			}
			break;
		case XK_Alt_L:
		case XK_Alt_R:
		case XK_Meta_L:
		case XK_Meta_R:
			KeyPorts[7] |= 0x02;
			if (emulatedShift)
			{
				emulatedShift--;
			}
			break;
		case XK_BackSpace:
		case XK_Delete:
			KeyPorts[0] |= 0x01;
			KeyPorts[4] |= 0x01;
			break;
		case XK_Escape:
			KeyPorts[0] |= 0x01;
			KeyPorts[3] |= 0x01;
			break;
		case XK_Up:
			if (!unshiftCursor)
			{
				KeyPorts[0] |= 0x01;
			}
			KeyPorts[4] |= 0x08;
			break;
		case XK_Down:
			if (!unshiftCursor)
			{
				KeyPorts[0] |= 0x01;
			}
			KeyPorts[4] |= 0x10;
			break;
		case XK_Left:
			if (!unshiftCursor)
			{
				KeyPorts[0] |= 0x01;
			}
			KeyPorts[3] |= 0x10;
			break;
		case XK_Right:
			if (!unshiftCursor)
			{
				KeyPorts[0] |= 0x01;
			}
			KeyPorts[4] |= 0x04;
			break;
		case XK_sterling:
			if (!emulatedShift)
			{
				KeyPorts[7] |= 0x02;
				KeyPorts[0] |= 0x04;
				break;
			}

		default:

			if (emulatedShift)
			{
				if (ks0 < 128)
				{
					KeyPorts[keyTable[ks0].port] |= ~keyTable[ks0].bit;
				}
			}
			else if (ks < 128)
			{
				if	(ks >= 'A' && ks <= 'Z')
				{

					KeyPorts[0] |= 0x01;
				}
				else if (!((ks >= 'a' && ks <= 'z')
						   || (ks >= '0' && ks <= '9')
						   || ks == ' '))
				{

					KeyPorts[7] |= 0x02;
				}
				KeyPorts[keyTable[ks].port] |= ~keyTable[ks].bit;
			}
			break;
	}
}

static void
notifyWm(int *argcnt, char **argvec)
{
	XTextProperty appName, iconName;
	Pixmap iconPix;
	XSizeHints xsh;
	XWMHints xwmh;
	XClassHint xch;
	if (!XStringListToTextProperty(&Version, 1, &appName)
		|| !XStringListToTextProperty(&Version, 1, &iconName))
	{
		Msg(M_PERR, "XStringListToTextProperty failed");
		return;
	}
	iconPix = XCreatePixmapFromBitmapData(Dpy, root, xzx_bits, xzx_width,
										  xzx_height, black, white,
										  DefaultDepth(Dpy, scrn));
	xsh.flags = PSize | PMinSize | PMaxSize |
				(strlen(GETCFG(geometry)) > 0 ? USPosition : PPosition);
	xsh.min_width  = hsize;
	xsh.min_height = vsize;
	xsh.max_width  = hsize + BORDER_WIDTH * 2;
	xsh.max_height = vsize + BORDER_WIDTH * 2;
	xwmh.initial_state = NormalState;
	xwmh.input = True;
	xwmh.icon_pixmap = iconPix;
	xwmh.flags = StateHint | IconPixmapHint | InputHint;
	xch.res_name = "xzx";
	xch.res_class = "Xzx";
	XSetWMProperties(Dpy, borderWin, &appName, &iconName, argvec, *argcnt,
					 &xsh, &xwmh, &xch);
	XFree(appName.value);
	XFree(iconName.value);

	deleteAtom = XInternAtom(Dpy, "WM_DELETE_WINDOW", False);
	(void)XSetWMProtocols(Dpy, borderWin, &deleteAtom, 1);
#ifdef OFFIX_DND

	dndProtocol  = XInternAtom(Dpy, "_DND_PROTOCOL", False);
	dndSelection = XInternAtom(Dpy, "_DND_SELECTION", False);
#endif
}

static void
calcWinPosition(int *x, int *y)
{
	int tmpX, tmpY;
	unsigned int w, h;
	int mask;
	*x = 0;
	*y = 0;
	if (!GETCFG(geometry)[0])
	{
		return;
	}
	mask = XParseGeometry(GETCFG(geometry), &tmpX, &tmpY, &w, &h);
	if (!mask)
	{
		Msg(M_WARN, "invalid geometry specification <%s>", GETCFG(geometry));
		GETCFG(geometry)[0] = '\0';
		return;
	}
	if (mask & XValue)
	{
		*x = (mask & XNegative) ?
			 (DisplayWidth(Dpy, scrn) + tmpX - (hsize + BORDER_WIDTH * 2)) :
			 tmpX;
	}
	if (mask & YValue)
	{
		*y = (mask & YNegative) ?
			 (DisplayHeight(Dpy, scrn) + tmpY - (vsize + BORDER_WIDTH * 2)) :
			 tmpY;
	}
}

#ifndef EXCLUDE_PIXMAP
static int
initPixmap(void)
{
	thePix = XCreatePixmap(Dpy, root, hsize, vsize, DefaultDepth(Dpy, scrn));
	ScreenRefresh = scrRefreshPix;
	screenWrite = scrWritePix;
	return 0;
}
#endif

static int
initImage(void)
{
	int i;
#ifdef MITSHM
	if (GETCFG(mitshm))
	{
		int x, y, z;

		if (XQueryExtension(Dpy, "MIT-SHM", &x, &y, &z))
		{
			(void)XShmQueryVersion(Dpy, &x, &y, (Bool *)&GETCFG(mitshm));
		}
		else
		{
			SETCFG(mitshm, False);
		}
	}
	if (GETCFG(mitshm))
	{
		xim = XShmCreateImage(Dpy, DefaultVisual(Dpy, scrn),
							  DefaultDepth(Dpy, scrn), ZPixmap, NULL,
							  &xshminfo, hsize, vsize);
		if (!xim)
		{
			Msg(M_PERR, "XShmCreateImage failed");
			return -1;
		}
		xshminfo.shmid = shmget(IPC_PRIVATE, xim->bytes_per_line * xim->height,
								IPC_CREAT | 0777);
		if (xshminfo.shmid == -1)
		{
			Msg(M_PERR, "shmget <xshminfo.shmid> failed");
			return -1;
		}
		xshminfo.shmaddr = xim->data = shmat(xshminfo.shmid, 0, 0);
		if (!xshminfo.shmaddr)
		{
			Msg(M_PERR, "shmat <xshminfo.shmaddr> failed");
			return -1;
		}
		xshminfo.readOnly = False;
		if (!XShmAttach(Dpy, &xshminfo))
		{
			Msg(M_PERR, "XShmAttach failed");
			return -1;
		}
		ScreenRefresh = scrRefreshXshm;
	}
	else
#endif
	{
		xim = XCreateImage(Dpy, DefaultVisual(Dpy, scrn),
						   DefaultDepth(Dpy, scrn), ZPixmap, 0, NULL,
						   hsize, vsize, 8, 0);
		if (!xim)
		{
			Msg(M_PERR, "XCreateImage failed");
			return -1;
		}
		xim->data = (char *)Malloc(xim->bytes_per_line * xim->height,
								   "initImage");
		ScreenRefresh = scrRefreshXim;
	}
	screenWrite = scrWriteXim;
#ifdef DEBUG
	if (GETCFG(debug) & D_X11)
	{
		Msg(M_DEBUG, "bytesPerLine  : %d", xim->bytes_per_line);
		Msg(M_DEBUG, "bitsPerPixel  : %d", xim->bits_per_pixel);
		Msg(M_DEBUG, "bitmapBitOrder: %s first",
			BitmapBitOrder(Dpy) == LSBFirst ? "LSB" : "MSB");
		Msg(M_DEBUG, "imageByteOrder: %s first",
			ImageByteOrder(Dpy) == LSBFirst ? "LSB" : "MSB");
	}
#endif
	bytesPerLine = xim->bytes_per_line;
	bitsPerPixel = xim->bits_per_pixel;
#ifndef EXCLUDE_EXOTIC
	if ((bitsPerPixel > 1 && bitsPerPixel < 8)
#else
	if (bitsPerPixel < 8
#endif
		|| bytesPerLine != COLS * colorDepth * GETCFG(scale))
	{
		Msg(M_FATAL, "XZX does not support your X-server configuration");
	}

	ximAddrTable = Malloc(sizeof(void *) * PIXEL_LENGTH, "initImage");
	for (i = 0; i < PIXEL_LENGTH; i++)
	{
		ximAddrTable[i] = xim->data + GETCFG(scale) *
						(DtblX[i] * bitsPerPixel / 8  + DtblY[i] * bytesPerLine);
	}
	return 0;
}

static int
getColors(unsigned long *col)
{
	XColor xc;
	int i, j, k;
#ifdef DEBUG
	if (GETCFG(debug) & D_X11)
	{
		Msg(M_DEBUG, "colorDepth    : %d", DefaultDepthOfScreen(sptr));
	}
#endif
#ifndef EXCLUDE_PIXMAP
	if (!GETCFG(pixmaps))
#endif
	{
		switch(DefaultDepthOfScreen(sptr))
		{
#ifndef EXCLUDE_EXOTIC
			case 1:
				wantColor = !GETCFG(mono);
				SETCFG(mono, True);
				colorDepth = 1;
				break;
#endif
			case 8:
				colorDepth = 8;
				break;
			case 16:
				colorDepth = 16;
				break;
			case 24:
			case 32:
				colorDepth = 32;
				break;
			default:
				Msg(M_FATAL, "color depth %dbpp not supported",
					DefaultDepthOfScreen(sptr));
				break;
		}
	}

	if (GETCFG(mono))
	{
		int fg, bg;
		for (i = 0; i < 8; i++)
		{
			col[i] = col[i + 8] = i < 4 ? black : white;
		}

		for (i = 0; i < 2; i ++)
		{
			for (j = 0; j < 8; j++)
			{
				for (k = 0; k < 8; k++)
				{
					if (j > k)
					{
						fg = black;
						bg = white;
					}
					else if (j < k)
					{
						fg = white;
						bg = black;
					}
					else
					{
						if (j < GREEN || (j > WHITE && j < BRIGHT_GREEN))
						{
							fg = bg = black;
						}
						else
						{
							fg = bg = white;
						}
					}
					ink  [(i << 6) + (j << 3) + k] = fg;
					paper[(i << 6) + (j << 3) + k] = bg;
				}
			}
		}
	}
	else
	{
		xc.flags = DoRed | DoGreen | DoBlue;
		for (i = 0; i < NCOLORS; i++)
		{
			xc.red   = rgbVals[i][0];
			xc.green = rgbVals[i][1];
			xc.blue  = rgbVals[i][2];
			if (!XAllocColor(Dpy, cmap, &xc))
			{
				Msg(M_WARN, "XAllocColor failed on color %d", i);
				if (i)
				{

					XFreeColors(Dpy, cmap, col, i - 1, 0);
				}
				return -1;
			}
			col[i] = xc.pixel;
		}

		for (i = 0; i < 2; i ++)
		{
			for (j = 0; j < 8; j++)
			{
				for (k = 0; k < 8; k++)
				{
					ink  [(i << 6) + (j << 3) + k] = col[k + (i << 3)];
					paper[(i << 6) + (j << 3) + k] = col[j + (i << 3)];
				}
			}
		}
	}
	return 0;
}

void
InitX(int *argcp, char **argvp)
{
	int xpos, ypos;

	if (!(Dpy = GetResources(argcp, argvp)))
	{
		Msg(M_FATAL, "couldn't connect to X display");
	}

	scrn = DefaultScreen(Dpy);
	sptr = DefaultScreenOfDisplay(Dpy);
	root = DefaultRootWindow(Dpy);
	white = WhitePixel(Dpy, scrn);
	black = BlackPixel(Dpy, scrn);
	hsize = HSize;
	vsize = VSize;

	theGc = XCreateGC(Dpy, root, 0, NULL);
	XCopyGC(Dpy, DefaultGC(Dpy, scrn), ~0, theGc);
	XSetGraphicsExposures(Dpy, theGc, False);
#ifndef EXCLUDE_PIXMAP

	fgGc = XCreateGC(Dpy, root, 0, NULL);
	XCopyGC(Dpy, DefaultGC(Dpy, scrn), ~0, fgGc);
	XSetGraphicsExposures(Dpy, fgGc, False);
	bgGc = XCreateGC(Dpy, root, 0, NULL);
	XCopyGC(Dpy, DefaultGC(Dpy, scrn), ~0, bgGc);
	XSetGraphicsExposures(Dpy, bgGc, False);
#endif

	if (!GETCFG(private))
	{
		cmap  = DefaultColormap(Dpy, scrn);
		if (getColors(colors) == -1)
		{
			Msg(M_WARN, "color allocation failed - using private colormap");
			SETCFG(private, True);
		}
	}
	if (GETCFG(private))
	{
		cmap = XCreateColormap(Dpy, root, DefaultVisual(Dpy, scrn), AllocNone);
		if (getColors(colors) == -1)
		{
			Msg(M_FATAL, "couldn't initialise private colormap");
		}
	}
	flashPhase = !GETCFG(flashing);

#ifndef EXCLUDE_PIXMAP
	if (GETCFG(pixmaps))
	{
		if (initPixmap())
		{
			Msg(M_FATAL, "couldn't initialise pixmap");
		}
	}
	else
#endif
	{
	  tryagain:
		if (initImage())
		{
#ifdef MITSHM
			if (GETCFG(mitshm))
			{
				Msg(M_WARN,
					"MIT-SHM initialisation failed, trying without MIT-SHM");
				SETCFG(mitshm, False);
				goto tryagain;
			}
#endif
			Msg(M_FATAL, "couldn't initialise image");
		}
	}
	MINIMIZE_BOX;
	(void)memset(scrChanged, 0, sizeof(scrChanged));
#ifdef LINE_VIDEO
	(void)memset(scrChangedDelayed, 0, sizeof(scrChangedDelayed));
	screenDirtyDelayed = 0;
#endif
#ifndef EXCLUDE_EXOTIC
#ifndef EXCLUDE_PIXMAP
	if (!GETCFG(pixmaps) && bitsPerPixel < 8)
#else
	if (bitsPerPixel < 8)
#endif
	{
		initScaleupTable();
	}
#endif
	calcWinPosition(&xpos, &ypos);

	borderWin = XCreateSimpleWindow(Dpy, root, xpos, ypos,
									hsize + BORDER_WIDTH * 2,
									vsize + BORDER_WIDTH * 2,
									0, colors[WHITE], colors[WHITE]);
	mainWin = XCreateSimpleWindow(Dpy, borderWin, BORDER_WIDTH, BORDER_WIDTH,
								  hsize, vsize, 0, colors[WHITE],
								  colors[WHITE]);

	notifyWm(argcp, argvp);

	if (GETCFG(private))
	{
		XSetWindowColormap(Dpy, borderWin, cmap);
	}
	XSelectInput(Dpy, mainWin, ExposureMask);
	XSelectInput(Dpy, borderWin, KeyPressMask | KeyReleaseMask | ExposureMask |
				 EnterWindowMask | LeaveWindowMask | StructureNotifyMask);

	XMapRaised(Dpy, borderWin);
	XMapRaised(Dpy, mainWin);
	XFlush(Dpy);
	OnQuit(quitX);
}

void
CheckEvents(void)
{
	static XEvent xev;
#ifdef X_TERMINAL
	static long keyMask = KeyPressMask;
	static Time time = 0;
	while (XCheckMaskEvent(Dpy, keyMask | ExposureMask | EnterWindowMask |
						   LeaveWindowMask | StructureNotifyMask, &xev))
	{
#else
	while (XEventsQueued(Dpy, QueuedAfterReading))
	{
		XNextEvent(Dpy, &xev);
#endif
#ifdef DEBUG
		if (GETCFG(debug) & D_X11)
		{
			static char *eventNames[] = {
				"",
				"",
				"KeyPress",
				"KeyRelease",
				"ButtonPress",
				"ButtonRelease",
				"MotionNotify",
				"EnterNotify",
				"LeaveNotify",
				"FocusIn",
				"FocusOut",
				"KeymapNotify",
				"Expose",
				"GraphicsExpose",
				"NoExpose",
				"VisibilityNotify",
				"CreateNotify",
				"DestroyNotify",
				"UnmapNotify",
				"MapNotify",
				"MapRequest",
				"ReparentNotify",
				"ConfigureNotify",
				"ConfigureRequest",
				"GravityNotify",
				"ResizeRequest",
				"CirculateNotify",
				"CirculateRequest",
				"PropertyNotify",
				"SelectionClear",
				"SelectionRequest",
				"SelectionNotify",
				"ColormapNotify",
				"ClientMessage",
				"MappingNotify"
			};
			assert(xev.type < sizeof(eventNames)/sizeof(eventNames[0]));
			Msg(M_DEBUG, "got X event %s", eventNames[xev.type]);
		}
#endif
		switch(xev.type)
		{
			case Expose:

				if (!xev.xexpose.count)
				{

					MAXIMIZE_BOX;

					if (InDialog)
					{
						ScreenRefresh();
					}
				}
				break;
			case ConfigureNotify:
				XMoveWindow(Dpy, mainWin,
							(xev.xconfigure.width - hsize) / 2,
							(xev.xconfigure.height - vsize) / 2);
				break;
			case EnterNotify:
				if (xev.xcrossing.detail != NotifyInferior && !InDialog)
				{
					XAutoRepeatOff(Dpy);
				}
				break;
			case LeaveNotify:

				if (xev.xcrossing.detail != NotifyInferior && !InDialog)
				{
					XAutoRepeatOn(Dpy);

					(void)memset(KeyPorts, 0xff, sizeof(KeyPorts));
					unshiftCursor = 0;
					emulatedShift = 0;
				}
				break;
			case KeyPress:
				if (InDialog)
				{
					(void)XLookupString((XKeyEvent *)&xev, NULL, 0, &KeyDialog,
										NULL);
				}
				else
				{
					processKeypress((XKeyEvent *)&xev);
#ifdef X_TERMINAL
					time = xev.xkey.time;
#endif
				}
				break;
			case KeyRelease:

				if (!InDialog)
				{
#ifdef X_TERMINAL
					if (xev.xkey.time > time)
					{

#ifdef DEBUG
						if (GETCFG(debug) & D_X11)
						{
							Msg(M_DEBUG, "delaying KeyRelease event");
						}
#endif
						XPutBackEvent(Dpy, &xev);
						time = xev.xkey.time;
						goto end;
					}
#endif
					processKeyrelease((XKeyEvent *)&xev);
				}
				break;
			case ClientMessage:

				if (xev.xclient.format == 32
					&& xev.xclient.data.l[0] == (long)deleteAtom)
				{
					Quit(0);
				}
#ifdef OFFIX_DND

				else if (xev.xclient.message_type == dndProtocol &&
						 xev.xclient.data.l[0] == DndFile)
				{
					Atom dummy_a;
					int dummy_f;
					unsigned long dummy_s, dummy_r;
					unsigned char *data;
					(void)XGetWindowProperty(Dpy, root, dndSelection,
											 0L, 100000L,
											 False, AnyPropertyType,
											 &dummy_a, &dummy_f,
											 &dummy_s, &dummy_r,
											 &data);
					(void)ReadSnapshot(data);
					XFree(data);
				}
#endif
				break;
		}
	}
#ifdef X_TERMINAL

  end:
	keyMask = (keyMask == KeyPressMask) ? KeyReleaseMask : KeyPressMask;
#endif
}

