<?php
/**
 * The IMP_tree class provides a tree view of the folders in an IMAP/POP3
 * repository.  It provides access functions to iterate through this tree and
 * query information about individual mailboxes.
 *
 * $Horde: imp/lib/IMAP/Tree.php,v 1.25.2.49 2008/05/07 13:45:16 jan Exp $
 *
 * Copyright 2000-2008 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @author  Jon Parise <jon@horde.org>
 * @author  Anil Madhavapeddy <avsm@horde.org>
 * @author  Michael Slusarz <slusarz@horde.org>
 * @package IMP
 */

/* Constants for mailboxElt attributes.
 * All versions of c-client (c-client/mail.h) define these constants:
 *   LATT_NOINFERIORS (long) 0x1 = 1
 *   LATT_NOSELECT (long) 0x2 = 2
 *   LATT_MARKED (long) 0x4 = 4
 *   LATT_UNMARKED (long) 0x8 = 8
 *
 * Newer versions of c-client (imap-2002 and greater) define these constants:
 *   LATT_REFERRAL (long) 0x10 = 16
 *   LATT_HASCHILDREN (long) 0x20 = 32
 *   LATT_HASNOCHILDREN (long) 0x40 = 64
 * ...but these constant names do not appear in PHP until PHP 4.3.5 and 5.0.
 *
 * Also, no need to define LATT_REFERRAL at the present time since we don't use
 * it anywhere.
 */

if (!defined('LATT_HASCHILDREN')) {
    @define('LATT_HASCHILDREN', 32);
    @define('LATT_HASNOCHILDREN', 64);
}

/* Start at 128 for our local bitmasks to allow for the c-client LATT_*
   constants. */
define('IMPTREE_ELT_NAMESPACE', 128);
define('IMPTREE_ELT_IS_OPEN', 256);
define('IMPTREE_ELT_IS_SUBSCRIBED', 512);
define('IMPTREE_ELT_IS_DISCOVERED', 1024);
define('IMPTREE_ELT_IS_POLLED', 2048);
define('IMPTREE_ELT_NEED_SORT', 4096);
define('IMPTREE_ELT_VFOLDER', 8192);
define('IMPTREE_ELT_NONIMAP', 16384);
define('IMPTREE_ELT_NOSHOW', 32768);

/* The isOpen() expanded mode constants. */
define('IMPTREE_OPEN_NONE', 0);
define('IMPTREE_OPEN_ALL', 1);
define('IMPTREE_OPEN_USER', 2);

/* The initalization mode to use when creating the tree. */
define('IMPTREE_INIT_SUB', 1);
define('IMPTREE_INIT_UNSUB', 2);
define('IMPTREE_INIT_FETCHALL', 4);

/* The manner to which to traverse the tree when calling next(). */
define('IMPTREE_NEXT_SHOWCLOSED', 1);
define('IMPTREE_NEXT_SHOWSUB', 2);

/* The string used to indicate the base of the tree. */
define('IMPTREE_BASE_ELT', '%');

/** Defines used with the output from the build() function. */
define('IMPTREE_SPECIAL_INBOX', 1);
define('IMPTREE_SPECIAL_TRASH', 2);
define('IMPTREE_SPECIAL_DRAFT', 3);
define('IMPTREE_SPECIAL_SPAM', 4);
define('IMPTREE_SPECIAL_SENT', 5);

/** Defines used with folderList(). */
define('IMPTREE_FLIST_CONTAINER', 1);
define('IMPTREE_FLIST_UNSUB', 2);
define('IMPTREE_FLIST_OB', 4);

/* Add a percent to folder key since it allows us to sort by name but never
 * conflict with an IMAP mailbox of the same name (since '%' is an invalid
 * character in an IMAP mailbox string). */
/** Defines used with virtual folders. */
define('IMPTREE_VFOLDER_LABEL', _("Virtual Folders"));
define('IMPTREE_VFOLDER_KEY', IMPTREE_VFOLDER_LABEL . '%');

/** Defines used with namespace display. */
define('IMPTREE_SHARED_LABEL', _("Shared Folders"));
define('IMPTREE_SHARED_KEY', IMPTREE_SHARED_LABEL . '%');
define('IMPTREE_OTHER_LABEL', _("Other Users' Folders"));
define('IMPTREE_OTHER_KEY', IMPTREE_OTHER_LABEL . '%');

class IMP_Tree {

    /**
     * Array containing the mailbox tree.
     *
     * @var array
     */
    var $_tree;

    /**
     * Location of current element in the tree.
     *
     * @var string
     */
    var $_currparent = null;

    /**
     * Location of current element in the tree.
     *
     * @var integer
     */
    var $_currkey = null;

    /**
     * Location of current element in the tree.
     *
     * @var array
     */
    var $_currstack = array();

    /**
     * Show unsubscribed mailboxes?
     *
     * @var boolean
     */
    var $_showunsub = false;

    /**
     * Parent list.
     *
     * @var array
     */
    var $_parent = array();

    /**
     * The cached list of mailboxes to poll.
     *
     * @var array
     */
    var $_poll = null;

    /**
     * The cached list of expanded folders.
     *
     * @var array
     */
    var $_expanded = null;

    /**
     * Cached list of subscribed mailboxes.
     *
     * @var array
     */
    var $_subscribed = null;

    /**
     * Cached list of unsubscribed mailboxes.
     *
     * @var array
     */
    var $_unsubscribed = null;

    /**
     * Init mode flag.
     *
     * @var integer
     */
    var $_initmode = 0;

    /**
     * Tree changed flag.  Set when something in the tree has been altered.
     *
     * @var boolean
     */
    var $_changed = false;

    /**
     * Have we shown unsubscribed folders previously?
     *
     * @var boolean
     */
    var $_unsubview = false;

    /**
     * The IMAP_Sort object.
     *
     * @var IMAP_Sort
     */
    var $_imap_sort = null;

    /**
     * The server string for the current server.
     *
     * @var string
     */
    var $_server = '';

    /**
     * The server string used for the delimiter.
     *
     * @var string
     */
    var $_delimiter = '/';

    /**
     * The list of namespaces to add to the tree.
     *
     * @var array
     */
    var $_namespaces = array();

    /**
     * Does the IMAP server support the children extension?
     *
     * @var boolean
     */
    var $_childrensupport = null;

    /**
     * Used to determine the list of element changes.
     *
     * @var array
     */
    var $_eltdiff = null;

    /**
     * Array used to cache data used by element().
     *
     * @var array
     */
    var $_elt = array();

    /**
     * See $open parameter in build().
     *
     * @var boolean
     */
    var $_forceopen = false;

    /**
     * Attempts to return a reference to a concrete IMP_Tree instance.
     *
     * If an IMP_Tree object is currently stored in the local session,
     * recreate that object.  Else, create a new instance.  Ensures that only
     * one IMP_Tree instance is available at any time.
     *
     * This method must be invoked as:<pre>
     *   $imp_tree = &IMP_Tree::singleton();
     * </pre>
     *
     * @return IMP_Tree  The IMP_Tree object or null.
     */
    function &singleton()
    {
        static $instance;

        if (!isset($instance)) {
            if (!empty($_SESSION['imp']['cache']['imp_tree'])) {
                require_once 'Horde/SessionObjects.php';
                $cacheSess = &Horde_SessionObjects::singleton();
                $instance = $cacheSess->query($_SESSION['imp']['cache']['imp_tree']);
            }
            if (empty($instance) || is_a($instance, 'PEAR_Error')) {
                $instance = new IMP_Tree();
            }
            register_shutdown_function(array(&$instance, '_store'));
        }

        return $instance;
    }

    /**
     * Constructor.
     */
    function IMP_Tree()
    {
        $this->_server = IMP::serverString();

        if ($_SESSION['imp']['base_protocol'] != 'pop3') {
            $ptr = reset($_SESSION['imp']['namespace']);
            $this->_delimiter = $ptr['delimiter'];
            $this->_childrensupport = (bool)$_SESSION['imp']['imap_server']['children'];
            $this->_namespaces = (empty($GLOBALS['conf']['user']['allow_folders'])) ? array() : $_SESSION['imp']['namespace'];
        }

        $this->init();
    }

    /**
     * Store a serialized version of ourself in the current session.
     *
     * @access private
     */
    function _store()
    {
        /* We only need to restore the object if the tree has changed. */
        if (!empty($this->_changed)) {
            /* Don't store $_expanded and $_poll - these values are handled
             * by the subclasses.
             * Don't store $_imap_sort or $_eltdiff - this needs to be
             * regenerated for each request.
             * Don't store $_currkey, $_currparent, and $_currStack since the
             * user MUST call reset() before cycling through the tree.
             * Reset the $_changed flag. */
            $this->_currkey = $this->_currparent = $this->_eltdiff = $this->_expanded = $this->_imap_sort = $this->_poll = null;
            $this->_currStack = array();
            $this->_changed = false;

            require_once 'Horde/SessionObjects.php';
            $cacheSess = &Horde_SessionObjects::singleton();

            /* Reuse old session ID if possible. */
            if (isset($_SESSION['imp']['cache']['imp_tree'])) {
                $cacheSess->overwrite($_SESSION['imp']['cache']['imp_tree'], $this, false);
            } else {
                $_SESSION['imp']['cache']['imp_tree'] = $cacheSess->storeOid($this, false);
            }
        }
    }

    /**
     * Returns a list of folders/mailboxes matching a certain path.
     *
     * @access private
     *
     * @param string $path  The mailbox path.
     *
     * @return array  A list of mailbox_objects whose name matched $path.
     *                The server string has been removed from the name.
     */
    function _getList($path)
    {
        $unique = array();

        if (!$this->_showunsub) {
            $this->_initSubscribed();
        }
        $imp_imap = &IMP_IMAP::singleton();
        $newboxes = @imap_getmailboxes($imp_imap->stream(), $this->_server, $path);
        if (is_array($newboxes)) {
            foreach ($newboxes as $box) {
                if ($this->_showunsub ||
                    !isset($this->_subscribed[$box->name])) {
                    /* Strip off server string. */
                    $box = $this->_removeServerString($box);
                    if ($box->name && !isset($unique[$box->name])) {
                        $unique[$box->name] = $box;
                    }
                }
            }
        }

        return $unique;
    }

    /**
     * Make a single mailbox tree element.
     * An element consists of the following items (we use single letters here
     * to save in session storage space):
     *   'a'  --  Attributes
     *   'c'  --  Level count
     *   'l'  --  Label
     *   'p'  --  Parent node
     *   'v'  --  Value
     *
     * @access private
     *
     * @param object stdClass $ob  The object returned by imap_getmailboxes().
     *
     * @return array  See format above.
     */
    function _makeElt($ob)
    {
        $elt = array(
            'a' => $ob->attributes,
            'c' => 0,
            'p' => IMPTREE_BASE_ELT,
            'v' => $ob->name
        );

        /* Set subscribed values. Make sure INBOX is always subscribed to so
         * there is always at least 1 viewable element. */
        if ($elt['v'] == 'INBOX') {
            $this->_setSubscribed($elt, true);
        } elseif (!$this->isSubscribed($elt)) {
            $this->_initSubscribed();
            $this->_setSubscribed($elt, isset($this->_subscribed[$elt['v']]));
        }

        /* Check for polled status. */
        if (!$this->isPolled($elt)) {
            $this->getPollList();
            $this->_setPolled($elt, isset($this->_poll[$elt['v']]));
        }

        /* Check for open status. */
        $this->_setOpen($elt, $this->isOpen($elt));

        /* Computed values. */
        $ns_info = $this->_getNamespace($elt['v']);
        $tmp = explode(is_null($ns_info) ? $this->_delimiter : $ns_info['delimiter'], $elt['v']);
        $elt['c'] = count($tmp) - 1;
        $elt['l'] = String::convertCharset($tmp[$elt['c']], 'UTF7-IMAP');

        if ($_SESSION['imp']['base_protocol'] != 'pop3') {
            if ($elt['c'] != 0) {
                $elt['p'] = implode(is_null($ns_info) ? $this->_delimiter : $ns_info['delimiter'], array_slice($tmp, 0, $elt['c']));
            }

            if (!is_null($ns_info)) {
                switch ($ns_info['type']) {
                case 'personal':
                    /* Strip personal namespace. */
                    if (!empty($ns_info['name']) && ($elt['c'] != 0)) {
                        --$elt['c'];
                        if (strpos($elt['p'], $ns_info['delimiter']) === false) {
                            $elt['p'] = IMPTREE_BASE_ELT;
                        } elseif (strpos($elt['v'], $ns_info['name'] . 'INBOX' . $ns_info['delimiter']) === 0) {
                            $elt['p'] = 'INBOX';
                        }
                    }
                    break;

                case 'other':
                case 'shared':
                    if (substr($ns_info['name'], 0, -1 * strlen($ns_info['delimiter'])) == $elt['v']) {
                        $elt['a'] = LATT_NOSELECT | IMPTREE_ELT_NAMESPACE | IMPTREE_ELT_IS_SUBSCRIBED;
                    }

                    if ($GLOBALS['prefs']->getValue('tree_view')) {
                        $name = ($ns_info['type'] == 'other') ? IMPTREE_OTHER_KEY : IMPTREE_SHARED_KEY;
                        if ($elt['c'] == 0) {
                            $elt['p'] = $name;
                            ++$elt['c'];
                        } elseif ($this->_tree[$name] && IMPTREE_ELT_NOSHOW) {
                            if ($elt['c'] == 1) {
                                $elt['p'] = $name;
                            }
                        } else {
                            ++$elt['c'];
                        }
                    }
                    break;
                }
            }
        }

        return $elt;
    }

    /**
     * Initalize the list at the top level of the hierarchy.
     */
    function init()
    {
        static $already_init = null;

        $initmask = (($_SESSION['imp']['base_protocol'] == 'pop3') ||
                     !$GLOBALS['prefs']->getValue('subscribe') ||
                     $_SESSION['imp']['showunsub'])
            ? IMPTREE_INIT_UNSUB : IMPTREE_INIT_SUB;
        if ($GLOBALS['prefs']->getValue('show_sidebar')) {
            $initmask |= IMPTREE_INIT_FETCHALL;
        }

        if ($already_init == $initmask) {
            return;
        } else {
            $already_init = $initmask;
        }

        /* Reset all class variables to the defaults. */
        $this->_changed = true;
        $this->_currkey = $this->_currparent = $this->_subscribed = $this->_unsubscribed = null;
        $this->_currstack = $this->_tree = $this->_parent = array();
        $this->_showunsub = $this->_unsubview = ($initmask & IMPTREE_INIT_UNSUB);

        /* Set initialization mode. */
        $this->_initmode = $initmask;

        /* Get the initial list of mailboxes from the subclass. */
        $boxes = array();
        $tv = $GLOBALS['prefs']->getValue('tree_view');

        /* Create a placeholder element to the base of the tree list so we can
         * keep track of whether the base level needs to be sorted. */
        $this->_tree[IMPTREE_BASE_ELT] = array('a' => IMPTREE_ELT_IS_DISCOVERED | IMPTREE_ELT_NEED_SORT, 'v' => IMPTREE_BASE_ELT);

        if (empty($GLOBALS['conf']['user']['allow_folders'])) {
            $ob = $this->_getList('INBOX');
            $boxes['INBOX'] = reset($ob);
            $this->_addLevel($boxes);
            return;
        }

        foreach ($this->_namespaces as $key => $val) {
            /* We only need to provide the list of folders in the base
             * personal namespace.  Else, just use the base namespace entry. */
            if (($val['type'] == 'personal') || empty($val['name'])) {
                $query = $val['name'] . '%';
            } else {
                $query = (empty($val['delimiter'])) ? $val['name'] : rtrim($val['name'], $val['delimiter']);
                if (isset($this->_tree[$query])) {
                    continue;
                }
            }
            $tmp = $this->_getList($query);
            if (!empty($tmp)) {
                if ($val['type'] == 'personal') {
                    /* IMAP servers put the INBOX in the personal namespace -
                     * simply rename to 'INBOX' since that is where we
                     * always access the mailbox. */
                    $inbox_str = $val['name'] . 'INBOX';
                    if (!empty($val['name']) && isset($tmp[$inbox_str])) {
                        $tmp = array('INBOX' => $tmp[$inbox_str]) + $tmp;
                        $tmp['INBOX']->name = 'INBOX';
                        unset($tmp[$inbox_str]);
                    }
                } elseif ($tv) {
                    $name = ($val['type'] == 'other') ? IMPTREE_OTHER_KEY : IMPTREE_SHARED_KEY;
                    if (!isset($this->_tree[$name])) {
                        $ob = new stdClass;
                        $ob->attributes = LATT_NOSELECT | IMPTREE_ELT_NAMESPACE | IMPTREE_ELT_IS_DISCOVERED | IMPTREE_ELT_IS_SUBSCRIBED | IMPTREE_ELT_NONIMAP | IMPTREE_ELT_NOSHOW;
                        $ob->fullServerPath = $ob->name = $name;
                        $elt = $this->_makeElt($ob);
                        $elt['l'] = ($val['type'] == 'other') ? IMPTREE_OTHER_LABEL : IMPTREE_SHARED_LABEL;
                        $this->_insertElt($elt);
                    }

                    foreach ($this->_namespaces as $val2) {
                        if (($val2['type'] == $val['type']) &&
                            ($val2['name'] != $val['name'])) {
                            $this->_tree[$name] &= ~IMPTREE_ELT_NOSHOW;
                            break;
                        }
                    }
                }
                $boxes = array_merge($boxes, $tmp);
            }
        }

        if (!isset($boxes['INBOX'])) {
            $ob = $this->_getList('INBOX');
            $boxes['INBOX'] = reset($ob);
        }

        /* Create the list (INBOX and any other hierarchies). */
        $this->_addLevel($boxes);

        /* End initialization mode. */
        $this->_initmode = 0;

        /* Convert 'INBOX' to localized name. */
        $this->_tree['INBOX']['l'] = _("Inbox");

        /* Add virtual folders to the tree. */
        $this->insertVFolders($GLOBALS['imp_search']->listQueries(true));
    }

    /**
     * Expand a mail folder.
     *
     * @param string $folder      The folder name to expand.
     * @param boolean $expandall  Expand all folders under this one?
     */
    function expand($folder, $expandall = false)
    {
        $folder = $this->_convertName($folder);

        if (!isset($this->_tree[$folder])) {
            return;
        }
        $elt = &$this->_tree[$folder];

        /* Merge in next level of information. */
        if ($this->hasChildren($elt)) {
            $this->_changed = true;
            if (!$this->isDiscovered($elt)) {
                $info = $this->_childrenInfo($folder);
                if (!empty($info)) {
                    if ($this->_initmode) {
                        if (($this->_initmode & IMPTREE_INIT_FETCHALL) ||
                            $this->isOpen($elt)) {
                            $this->_addLevel($info);
                        }
                    } else {
                        $this->_addLevel($info);
                        $this->_setOpen($elt, true);
                    }
                }
            } else {
                if (!($this->_initmode & IMPTREE_INIT_FETCHALL)) {
                    $this->_setOpen($elt, true);
                }

                /* Expand all children beneath this one. */
                if ($expandall && !empty($this->_parent[$folder])) {
                    foreach ($this->_parent[$folder] as $val) {
                        $this->expand($this->_tree[$val]['v'], true);
                    }
                }
            }
        }
    }

    /**
     * Collapse a mail folder.
     *
     * @param string $folder  The folder name to collapse.
     */
    function collapse($folder)
    {
        $folder = $this->_convertName($folder);

        if (!isset($this->_tree[$folder])) {
            return;
        }

        $this->_changed = true;

        /* Set the folder attributes to not expanded. */
        $this->_setOpen($this->_tree[$folder], false);
    }

    /**
     * Sets the internal array pointer to the next element, and returns the
     * next object.
     *
     * @param integer $mask  A mask with the following elements:
     * <pre>
     * IMPTREE_NEXT_SHOWCLOSED - Don't ignore closed elements.
     * IMPTREE_NEXT_SHOWSUB - Only show subscribed elements.
     * </pre>
     *
     * @return mixed  Returns the next element or false if the element doesn't
     *                exist.
     */
    function next($mask = 0)
    {
        if (is_null($this->_currkey) && is_null($this->_currparent)) {
            return false;
        }

        $curr = $this->current();

        $old_showunsub = $this->_showunsub;
        if ($mask & IMPTREE_NEXT_SHOWSUB) {
            $this->_showunsub = false;
        }

        if ($this->_activeElt($curr) &&
            (($mask & IMPTREE_NEXT_SHOWCLOSED) || $this->isOpen($curr)) &&
            ($this->_currparent != $curr['v'])) {
            /* If the current element is open, and children exist, move into
             * it. */
            $this->_currstack[] = array('k' => $this->_currkey, 'p' => $this->_currparent);
            $this->_currkey = 0;
            $this->_currparent = $curr['v'];
            $this->_sortLevel($curr['v']);

            $curr = $this->current();
            if ($GLOBALS['prefs']->getValue('tree_view') &&
                $this->isNamespace($curr) &&
                !$this->_isNonIMAPElt($curr) &&
                ($this->_tree[$curr['p']] && IMPTREE_ELT_NOSHOW)) {
                $this->next($mask);
            }
        } else {
            /* Else, increment within the current subfolder. */
            $this->_currkey++;
        }

        /* If the pointer doesn't point to an element, try to move back to
           the previous subfolder.  If there is no previous subfolder,
           return false. */
        $curr = $this->current();
        if (!$curr &&
            ($mask & IMPTREE_NEXT_SHOWCLOSED) &&
            !$this->isDiscovered($this->_tree[$this->_currparent]) &&
            $this->hasChildren($this->_tree[$this->_currparent])) {
            $this->_addLevel($this->_childrenInfo($this->_tree[$this->_currparent]['v']));
            $curr = $this->current();
        }

        if (!$curr) {
            if (empty($this->_currstack)) {
                $this->_currkey = null;
                $this->_currparent = null;
                $this->_showunsub = $old_showunsub;
                return false;
            } else {
                do {
                    $old = array_pop($this->_currstack);
                    $this->_currkey = $old['k'] + 1;
                    $this->_currparent = $old['p'];
                } while ((($curr = $this->current()) == false) &&
                         count($this->_currstack));
            }
        }

        $res = $this->_activeElt($curr);
        $this->_showunsub = $old_showunsub;
        return ($res) ? $curr : $this->next($mask);
    }

    /**
     * Set internal pointer to the head of the tree.
     * This MUST be called before you can traverse the tree with next().
     *
     * @return mixed  Returns the element at the head of the tree or false
     *                if the element doesn't exist.
     */
    function reset()
    {
        $this->_currkey = 0;
        $this->_currparent = IMPTREE_BASE_ELT;
        $this->_currstack = array();
        $this->_sortLevel($this->_currparent);
        return $this->current();
    }

    /**
     * Return the current tree element.
     *
     * @return array  The current tree element or false if there is no
     *                element.
     */
    function current()
    {
        if (!isset($this->_parent[$this->_currparent][$this->_currkey])) {
            return false;
        }

        return $this->_tree[$this->_parent[$this->_currparent][$this->_currkey]];
    }

    /**
     * Determines if there are more elements in the current tree level.
     *
     * @return boolean  True if there are more elements, false if this is the
     *                  last element.
     */
    function peek()
    {
        for ($i = ($this->_currkey + 1); ; $i++) {
            if (!isset($this->_parent[$this->_currparent][$i])) {
                return false;
            }
            if ($this->_activeElt($this->_tree[$this->_parent[$this->_currparent][$i]])) {
                return true;
            }
        }
    }

    /**
     * Returns the requested element.
     *
     * @param string $name  The name of the tree element.
     *
     * @return array  Returns the requested element or false if not found.
     */
    function get($name)
    {
        $name = $this->_convertName($name);
        return (isset($this->_tree[$name])) ? $this->_tree[$name] : false;
    }

    /**
     * Insert a folder/mailbox into the tree.
     *
     * @param mixed $id  The name of the folder (or a list of folder names)
     *                   to add (must be present on the mail server).
     */
    function insert($id)
    {
        if (is_array($id)) {
            /* We want to add from the BASE of the tree up for efficiency
             * sake. */
            $this->_sortList($id);
        } else {
            $id = array($id);
        }

        $adds = array();

        foreach ($id as $val) {
            $ns_info = $this->_getNamespace($val);
            if (is_null($ns_info)) {
                if (strpos($val, IMPTREE_VFOLDER_KEY . $this->_delimiter) === 0) {
                    $adds[] = IMPTREE_VFOLDER_KEY;
                }
                $adds[] = $val;
            } else {
                /* Break apart the name via the delimiter and go step by
                 * step through the name to make sure all subfolders exist
                 * in the tree. */
                $parts = explode($ns_info['delimiter'], $val);
                $parts[0] = $this->_convertName($parts[0]);
                $parts_count = count($parts);
                for ($i = 0; $i < $parts_count; ++$i) {
                    $adds[] = implode($ns_info['delimiter'], array_slice($parts, 0, $i + 1));
                }
            }
        }

        $imp_imap = &IMP_IMAP::singleton();

        foreach (array_unique($adds) as $id) {
            if (isset($this->_tree[$id])) {
                continue;
            }

            if (strpos($id, IMPTREE_VFOLDER_KEY) === 0) {
                $ob = new stdClass;
                $ob->attributes = IMPTREE_ELT_IS_DISCOVERED | IMPTREE_ELT_IS_SUBSCRIBED | IMPTREE_ELT_VFOLDER;
                $ob->fullServerPath = $ob->name = $id;
                if ($id == IMPTREE_VFOLDER_KEY) {
                    $ob->attributes |= LATT_NOSELECT | LATT_HASCHILDREN | IMPTREE_ELT_NONIMAP;
                    $elt = $this->_makeElt($ob);
                    $elt['l'] = IMPTREE_VFOLDER_LABEL;
                } else {
                    $ob->attributes |= LATT_HASNOCHILDREN;
                    $elt = $this->_makeElt($ob);
                    $elt['l'] = $elt['v'] = String::substr($id, String::length(IMPTREE_VFOLDER_KEY) + String::length($this->_delimiter));
                }
            } else {
                $ob = $this->_getList($id);
                if (empty($ob)) {
                    continue;
                }
                $elt = $this->_makeElt(reset($ob));
                if (!$this->isSubscribed($elt)) {
                    $tmp = @imap_lsub($imp_imap->stream(), $this->_server, $elt['v']);
                    if (!empty($tmp)) {
                        $this->_setSubscribed($elt, true);
                    }
                }
            }

            if ($this->_insertElt($elt)) {
                /* We know that the parent folder has children. */
                if (isset($this->_tree[$elt['p']])) {
                    $this->_setChildren($this->_tree[$elt['p']], true);
                }

                /* Make sure we are sorted correctly. */
                if (count($this->_parent[$elt['p']]) > 1) {
                    $this->_setNeedSort($this->_tree[$elt['p']], true);
                }

                /* Add to subscribed/unsubscribed list. */
                if ($this->isSubscribed($elt) &&
                    !is_null($this->_subscribed)) {
                    $this->_subscribed[$elt['v']] = 1;
                } elseif (!$this->isSubscribed($elt) &&
                          !is_null($this->_unsubscribed)) {
                    $this->_unsubscribed[$elt['v']] = 1;
                }
            }
        }
    }

    /**
     * Insert an element into the tree.
     *
     * @access private
     *
     * @param array $elt  The element to insert. The key in the tree is the
     *                    'v' (value) element of the element.
     *
     * @return boolean  True if added to the tree.
     */
    function _insertElt($elt)
    {
        if (strlen($elt['l']) && !isset($this->_tree[$elt['v']])) {
            // UW fix - it may return both 'foo' and 'foo/' as folder names.
            // Only add one of these (without the namespace character) to
            // the tree.  See Ticket #5764.
            $ns_info = $this->_getNamespace($elt['v']);
            if (isset($this->_tree[rtrim($elt['v'], is_null($ns_info) ? $this->_delimiter : $ns_info['delimiter'])])) {
                return false;
            }

            $this->_changed = true;
            /* Set the parent array to the value in $elt['p']. */
            if (empty($this->_parent[$elt['p']])) {
                $this->_parent[$elt['p']] = array();
            }
            $this->_parent[$elt['p']][] = $elt['v'];
            $this->_tree[$elt['v']] = $elt;
            return true;
        }

        return false;
    }

    /**
     * Delete an element from the tree.
     *
     * @param mixed $id  The element name or an array of element names.
     *
     * @return boolean  Return true on success, false on error.
     */
    function delete($id)
    {
        if (is_array($id)) {
            /* We want to delete from the TOP of the tree down to ensure that
             * parents have an accurate view of what children are left. */
            $this->_sortList($id);
            $id = array_reverse($id);

            $success = true;
            foreach ($id as $val) {
                $currsuccess = $this->delete($val);
                if (!$currsuccess) {
                    $success = false;
                }
            }
            return $success;
        } else {
            $id = $this->_convertName($id, true);
        }

        $vfolder_base = ($id == IMPTREE_VFOLDER_LABEL);
        $search_id = $GLOBALS['imp_search']->createSearchID($id);

        if ($vfolder_base ||
            (isset($this->_tree[$search_id]) &&
             $this->isVFolder($this->_tree[$search_id]))) {
            if (!$vfolder_base) {
                $id = $search_id;

            }
            $parent = $this->_tree[$id]['p'];
            unset($this->_tree[$id]);

            /* Delete the entry from the parent tree. */
            $key = array_search($id, $this->_parent[$parent]);
            unset($this->_parent[$parent][$key]);

            /* Rebuild the parent tree. */
            if (!$vfolder_base && empty($this->_parent[$parent])) {
                $this->delete($parent);
            } else {
                $this->_parent[$parent] = array_values($this->_parent[$parent]);
            }
            $this->_changed = true;

            return true;
        }

        $ns_info = $this->_getNamespace($id);

        if (($id == 'INBOX') ||
            !isset($this->_tree[$id]) ||
            ($id == $ns_info['name'])) {
            return false;
        }

        $this->_changed = true;

        $elt = &$this->_tree[$id];

        /* Do not delete from tree if there are child elements - instead,
         * convert to a container element. */
        if ($this->hasChildren($elt)) {
            $this->_setContainer($elt, true);
            return true;
        }

        $parent = $elt['p'];

        /* Delete the tree entries. */
        unset($this->_tree[$id]);
        unset($this->_subscribed[$id]);
        unset($this->_unsubscribed[$id]);

        /* Delete the entry from the parent tree. */
        $key = array_search($id, $this->_parent[$parent]);
        unset($this->_parent[$parent][$key]);

        if (empty($this->_parent[$parent])) {
            /* This folder is now completely empty (no children).  If the
             * folder is a container only, we should delete the folder from
             * the tree. */
            unset($this->_parent[$parent]);
            if (isset($this->_tree[$parent])) {
                $this->_setChildren($this->_tree[$parent], null);
                if ($this->isContainer($this->_tree[$parent]) &&
                    !$this->isNamespace($this->_tree[$parent])) {
                    $this->delete($parent);
                } elseif (!$this->hasChildren($this->_tree[$parent])) {
                    $this->_removeExpandedList($parent);
                    $this->_setChildren($this->_tree[$parent], false);
                    $this->_setOpen($this->_tree[$parent], false);
                }
            }
        } else {
            /* Rebuild the parent tree. */
            $this->_parent[$parent] = array_values($this->_parent[$parent]);
        }

        /* Remove the mailbox from the expanded folders list. */
        $this->_removeExpandedList($id);

        /* Remove the mailbox from the nav_poll list. */
        $this->removePollList($id);

        return true;
    }

    /**
     * Subscribe an element to the tree.
     *
     * @param mixed $id  The element name or an array of element names.
     */
    function subscribe($id)
    {
        if (!is_array($id)) {
            $id = array($id);
        }

        foreach ($id as $val) {
            $val = $this->_convertName($val);
            if (isset($this->_tree[$val])) {
                $this->_changed = true;
                $this->_setSubscribed($this->_tree[$val], true);
                $this->_setContainer($this->_tree[$val], false);
                if (!is_null($this->_subscribed)) {
                    $this->_subscribed[$val] = 1;
                }
                unset($this->_unsubscribed[$val]);
            } else {
                $this->insert($val);
            }
        }
    }

    /**
     * Unsubscribe an element from the tree.
     *
     * @param mixed $id  The element name or an array of element names.
     */
    function unsubscribe($id)
    {
        if (!is_array($id)) {
            $id = array($id);
        } else {
            /* We want to delete from the TOP of the tree down to ensure that
             * parents have an accurate view of what children are left. */
            $this->_sortList($id);
            $id = array_reverse($id);
        }

        foreach ($id as $val) {
            $val = $this->_convertName($val);

            /* INBOX can never be unsubscribed to (if in mail mode). */
            if (isset($this->_tree[$val]) && ($val != 'INBOX')) {
                $this->_changed = true;
                $this->_unsubview = true;

                $elt = &$this->_tree[$val];

                /* Do not delete from tree if there are child elements -
                 * instead, convert to a container element. */
                if (!$this->_showunsub && $this->hasChildren($elt)) {
                    $this->_setContainer($elt, true);
                }

                /* Set as unsubscribed, add to unsubscribed list, and remove
                 * from subscribed list. */
                $this->_setSubscribed($elt, false);
                if (!is_null($this->_unsubscribed)) {
                    $this->_unsubscribed[$val] = 1;
                }
                unset($this->_subscribed[$val]);
            }
        }
    }

    /**
     * Add another level of hierarchy to the tree.
     *
     * @access private
     *
     * @param array $list  A list of stdClass objects in the format returned
     *                     from imap_getmailboxes().
     */
    function _addLevel($list)
    {
        $expandall = ($this->_initmode & IMPTREE_INIT_FETCHALL);
        $parent = null;

        foreach ($list as $val) {
            if (!is_object($val)) {
                continue;
            }

            if (isset($this->_tree[$val->name])) {
                $elt = $this->_tree[$val->name];
            } else {
                $elt = $this->_makeElt($val);
                $this->_insertElt($elt);
            }
            $parent = $elt['p'];
            if ($expandall || $this->isOpen($elt)) {
                $this->expand($elt['v']);
            }
        }

        /* Sort the list. */
        if (!is_null($parent) && !empty($this->_parent[$parent])) {
            $this->_setDiscovered($this->_tree[$parent], true);
            if (count($this->_parent[$parent]) > 1) {
                $this->_setNeedSort($this->_tree[$parent], true);
            }
        }
    }

    /**
     * Set an attribute for an element.
     *
     * @access private
     *
     * @param array &$elt     The tree element.
     * @param integer $const  The constant to set/remove from the bitmask.
     * @param boolean $bool   Should the attribute be set?
     */
    function _setAttribute(&$elt, $const, $bool)
    {
        if ($bool) {
            $elt['a'] |= $const;
        } else {
            $elt['a'] &= ~$const;
        }
    }

    /**
     * Does the element have any active children?
     *
     * @param array $elt  A tree element.
     *
     * @return boolean  True if the element has active children.
     */
    function hasChildren($elt)
    {
        static $hasChildrenCache = array();

        if ($this->_isNonIMAPElt($elt)) {
            if ($this->isVFolder($elt) || $this->_showunsub) {
                return true;
            }

            if (isset($this->_parent[$elt['v']])) {
                foreach ($this->_parent[$elt['v']] as $val) {
                    if (!$this->isNamespace($this->_tree[$val]) ||
                        $this->hasChildren($this->_tree[$val])) {
                        return true;
                    }
                }
            }
            return false;
        }

        /* Don't do the following if we are dealing with a namespace
         * container. */
        $is_ns = $this->isNamespace($elt);
        if (!$is_ns) {
            /* Not all IMAP servers support the HASCHILDREN flag (like UW!) so
             * we need to skip this check if the IMAP server doesn't set
             * either HASCHILDREN or HASNOCHILDREN. */
            if (!empty($this->_childrensupport) ||
                (is_null($this->_childrensupport) &&
                 ($elt['a'] & LATT_HASCHILDREN) ||
                 ($elt['a'] & LATT_HASNOCHILDREN))) {
                $ret = ($elt['a'] & LATT_HASCHILDREN);

                /* CHECK: If we are viewing all folders, and there is a folder
                 * listed as expanded but it does not contain any children,
                 * then we should remove it from the expanded list since it
                 * doesn't exist anymore. */
                if ($this->_showunsub && !$ret) {
                    $this->_initExpandedList();
                    if (!empty($this->_expanded[$elt['v']])) {
                        $this->_removeExpandedList($elt['v']);
                        $this->_setOpen($this->_tree[$elt['v']], false);
                    }
                }

                if (!$ret) {
                    return false;
                }

                /* If we are viewing all elements (subscribed and unsubscribed)
                 * and we reach this point we know that there must be viewable
                 * children so return true. */
                if ($this->_showunsub) {
                    return true;
                }
            }
        }

        /* Cache results from below since, most likely if we get this far,
         * this code will be accessed several times in the current request. */
        if (isset($hasChildrenCache[$elt['v']])) {
            return $hasChildrenCache[$elt['v']];
        }

        /* If we reach this point, then we are either in subscribe-only mode
         * or we are dealing with a namespace container. Check for the
         * existence of any subscribed mailboxes below the current node. */
        $this->_initSubscribed();
        $folder_list = $this->_subscribed;
        if ($this->_showunsub) {
            $this->_initUnsubscribed();
            $folder_list += $this->_unsubscribed;
        }
        $ns_info = $this->_getNamespace($elt['v']);
        if (!is_null($ns_info)) {
            $search_str = $elt['v'] . $ns_info['delimiter'];
            if (($elt['v'] == 'INBOX') &&
                ($ns_info['name'] == ('INBOX' . $ns_info['delimiter']))) {
                $search_str .= $ns_info['name'];
            }
            foreach (array_keys($folder_list) as $val) {
                if (strpos($val, $search_str) === 0) {
                    $this->_hasChildrenCache[$elt['v']] = true;
                    return true;
                }
            }
        }

        /* Do one final check if this is a namespace container - if we get
         * this far, and are viewing all folders, then we know we have no
         * children so make sure the element is not set to expanded/open. */
        if ($is_ns && $this->_showunsub) {
            $this->_initExpandedList();
            if (!empty($this->_expanded[$elt['v']])) {
                $this->_removeExpandedList($elt['v']);
                $this->_setOpen($this->_tree[$elt['v']], false);
            }
        }

        $hasChildrenCache[$elt['v']] = false;
        return false;
    }

    /**
     * Set the children attribute for an element.
     *
     * @access private
     *
     * @param array &$elt  A tree element.
     * @param mixed $bool  The setting. If null, clears the flag.
     */
    function _setChildren(&$elt, $bool)
    {
        if (is_null($bool)) {
            $this->_setAttribute($elt, LATT_HASCHILDREN, false);
            $this->_setAttribute($elt, LATT_HASNOCHILDREN, false);
        } else {
            $this->_setAttribute($elt, LATT_HASCHILDREN, $bool);
            $this->_setAttribute($elt, LATT_HASNOCHILDREN, !$bool);
        }
    }

    /**
     * Has the tree element been discovered?
     *
     * @param array $elt  A tree element.
     *
     * @return integer  Non-zero if the element has been discovered.
     */
    function isDiscovered($elt)
    {
        return $elt['a'] & IMPTREE_ELT_IS_DISCOVERED;
    }

    /**
     * Set the discovered attribute for an element.
     *
     * @access private
     *
     * @param array &$elt    A tree element.
     * @param boolean $bool  The setting.
     */
    function _setDiscovered(&$elt, $bool)
    {
        $this->_setAttribute($elt, IMPTREE_ELT_IS_DISCOVERED, $bool);
    }

    /**
     * Is the tree element open?
     *
     * @param array $elt  A tree element.
     *
     * @return integer  True if the element is open.
     */
    function isOpen($elt)
    {
        if (!$this->_initmode) {
            return (($elt['a'] & IMPTREE_ELT_IS_OPEN) && $this->hasChildren($elt));
        } else {
            switch ($GLOBALS['prefs']->getValue('nav_expanded')) {
            case IMPTREE_OPEN_NONE:
                return false;
                break;

            case IMPTREE_OPEN_ALL:
                return true;
                break;

            case IMPTREE_OPEN_USER:
                $this->_initExpandedList();
                return !empty($this->_expanded[$elt['v']]);
                break;
            }
        }
    }

    /**
     * Set the open attribute for an element.
     *
     * @access private
     *
     * @param array &$elt    A tree element.
     * @param boolean $bool  The setting.
     */
    function _setOpen(&$elt, $bool)
    {
        $this->_setAttribute($elt, IMPTREE_ELT_IS_OPEN, $bool);
        if (!$this->_initmode) {
            $this->_initExpandedList();
            if ($bool) {
                $this->_addExpandedList($elt['v']);
            } else {
                $this->_removeExpandedList($elt['v']);
            }
        }
    }

    /**
     * Is this element a container only, not a mailbox (meaning you can
     * not open it)?
     *
     * @param array $elt  A tree element.
     *
     * @return integer  True if the element is a container.
     */
    function isContainer($elt)
    {
        return (($elt['a'] & LATT_NOSELECT) ||
                (!$this->_showunsub &&
                 !$this->isSubscribed($elt) &&
                 $this->hasChildren($elt)));
    }

    /**
     * Set the element as a container?
     *
     * @access private
     *
     * @param array &$elt    A tree element.
     * @param boolean $bool  Is the element a container?
     */
    function _setContainer(&$elt, $bool)
    {
        $this->_setAttribute($elt, LATT_NOSELECT, $bool);
    }

    /**
     * Is the user subscribed to this element?
     *
     * @param array $elt  A tree element.
     *
     * @return integer  True if the user is subscribed to the element.
     */
    function isSubscribed($elt)
    {
        return $elt['a'] & IMPTREE_ELT_IS_SUBSCRIBED;
    }

    /**
     * Set the subscription status for an element.
     *
     * @access private
     *
     * @param array &$elt    A tree element.
     * @param boolean $bool  Is the element subscribed to?
     */
    function _setSubscribed(&$elt, $bool)
    {
        $this->_setAttribute($elt, IMPTREE_ELT_IS_SUBSCRIBED, $bool);
    }

    /**
     * Is the element a namespace container?
     *
     * @param array $elt  A tree element.
     *
     * @return integer  True if the element is a namespace container.
     */
    function isNamespace($elt)
    {
        return $elt['a'] & IMPTREE_ELT_NAMESPACE;
    }

    /**
     * Is the element a non-IMAP element?
     *
     * @param array $elt  A tree element.
     *
     * @return integer  True if the element is a non-IMAP element.
     */
    function _isNonIMAPElt($elt)
    {
        return $elt['a'] & IMPTREE_ELT_NONIMAP;
    }

    /**
     * Remove the server string from the 'name' parameter.
     *
     * @access private
     *
     * @param object stdClass $ob  An object returned from
     *                             imap_getmailboxes().
     *
     * @return stdClass  The object returned with the server string stripped
     *                   from the 'name' parameter.
     */
    function _removeServerString($ob)
    {
        $ob->fullServerPath = $ob->name;
        $ob->name = $this->_convertName(substr($ob->name, strpos($ob->name, '}') + 1));
        return $ob;
    }

    /**
     * Initialize the expanded folder list.
     *
     * @access private
     */
    function _initExpandedList()
    {
        if (is_null($this->_expanded)) {
            $serialized = $GLOBALS['prefs']->getValue('expanded_folders');
            $this->_expanded = ($serialized) ? unserialize($serialized) : array();
        }
    }

    /**
     * Add an element to the expanded list.
     *
     * @access private
     *
     * @param string $id  The element name to remove.
     */
    function _addExpandedList($id)
    {
        $this->_initExpandedList();
        $this->_expanded[$id] = true;
        $GLOBALS['prefs']->setValue('expanded_folders', serialize($this->_expanded));
    }

    /**
     * Remove an element from the expanded list.
     *
     * @access private
     *
     * @param string $id  The element name to remove.
     */
    function _removeExpandedList($id)
    {
        $this->_initExpandedList();
        unset($this->_expanded[$id]);
        $GLOBALS['prefs']->setValue('expanded_folders', serialize($this->_expanded));
    }

    /**
     * Initializes and returns the list of mailboxes to poll.
     *
     * @param boolean $prune  Prune non-existant folders from list?
     * @param boolean $sort   Sort the directory list?
     *
     * @return array  The list of mailboxes to poll.
     */
    function getPollList($prune = false, $sort = false)
    {
        if (is_null($this->_poll)) {
            /* We ALWAYS poll the INBOX. */
            $this->_poll = array('INBOX' => 1);

            /* Add the list of polled mailboxes from the prefs. */
            if ($GLOBALS['prefs']->getValue('nav_poll_all')) {
                $navPollList = array_flip($this->folderList());
            } else {
                $navPollList = @unserialize($GLOBALS['prefs']->getValue('nav_poll'));
            }
            if ($navPollList) {
                $this->_poll += $navPollList;
            }
        }

        $plist = ($prune) ? array_values(array_intersect(array_keys($this->_poll), $this->folderList())) : $this->_poll;
        if ($sort) {
            require_once IMP_BASE . '/lib/IMAP/Sort.php';
            $ns_new = $this->_getNamespace(null);
            $imap_sort = new IMP_IMAP_Sort($ns_new['delimiter']);
            $imap_sort->sortMailboxes($plist);
        }

        return $plist;
    }

    /**
     * Add element to the poll list.
     *
     * @param mixed $id  The element name or a list of element names to add.
     */
    function addPollList($id)
    {
        if (!is_array($id)) {
            $id = array($id);
        }

        if (!empty($id) && !$GLOBALS['prefs']->isLocked('nav_poll')) {
            require_once IMP_BASE . '/lib/Folder.php';
            $imp_folder = &IMP_Folder::singleton();
            $this->getPollList();
            foreach ($id as $val) {
                if (!$this->isSubscribed($this->_tree[$val])) {
                    $imp_folder->subscribe(array($val));
                }
                $this->_poll[$val] = true;
                $this->_setPolled($this->_tree[$val], true);
            }
            $GLOBALS['prefs']->setValue('nav_poll', serialize($this->_poll));
            $this->_changed = true;
        }
    }

    /**
     * Remove element from the poll list.
     *
     * @param string $id  The folder/mailbox or a list of folders/mailboxes
     *                    to remove.
     */
    function removePollList($id)
    {
        if (!is_array($id)) {
            $id = array($id);
        }

        $removed = false;

        if (!$GLOBALS['prefs']->isLocked('nav_poll')) {
            $this->getPollList();
            foreach ($id as $val) {
                if ($val != 'INBOX') {
                    unset($this->_poll[$val]);
                    if (isset($this->_tree[$val])) {
                        $this->_setPolled($this->_tree[$val], false);
                    }
                    $removed = true;
                }
            }
            if ($removed) {
                $GLOBALS['prefs']->setValue('nav_poll', serialize($this->_poll));
                $this->_changed = true;
            }
        }
    }

    /**
     * Does the user want to poll this mailbox for new/unseen messages?
     *
     * @param array $elt  A tree element.
     *
     * @return integer  True if the user wants to poll the element.
     */
    function isPolled($elt)
    {
        return ($GLOBALS['prefs']->getValue('nav_poll_all')) ? true : ($elt['a'] & IMPTREE_ELT_IS_POLLED);
    }

    /**
     * Set the polled attribute for an element.
     *
     * @access private
     *
     * @param array &$elt    A tree element.
     * @param boolean $bool  The setting.
     */
    function _setPolled(&$elt, $bool)
    {
        $this->_setAttribute($elt, IMPTREE_ELT_IS_POLLED, $bool);
    }

    /**
     * Flag the element as needing its children to be sorted.
     *
     * @access private
     *
     * @param array &$elt    A tree element.
     * @param boolean $bool  The setting.
     */
    function _setNeedSort(&$elt, $bool)
    {
        $this->_setAttribute($elt, IMPTREE_ELT_NEED_SORT, $bool);
    }

    /**
     * Does this element's children need sorting?
     *
     * @param array $elt  A tree element.
     *
     * @return integer  True if the children need to be sorted.
     */
    function _needSort($elt)
    {
        return (($elt['a'] & IMPTREE_ELT_NEED_SORT) && (count($this->_parent[$elt['v']]) > 1));
    }

    /**
     * Initialize the list of subscribed mailboxes.
     *
     * @access private
     */
    function _initSubscribed()
    {
        if (is_null($this->_subscribed)) {
            $this->_changed = true;
            $this->_subscribed = array();
            $sublist = array();

            // If subscription mode is off, we know that all mailboxes are
            // subscribed mailboxes.
            $imap_cmd = ($GLOBALS['prefs']->getValue('subscribe')) ? 'imap_lsub' : 'imap_list';

            /* INBOX is always subscribed to. */
            $this->_subscribed['INBOX'] = 1;

            $imp_imap = &IMP_IMAP::singleton();

            foreach ($this->_namespaces as $val) {
                $tmp = call_user_func($imap_cmd, $imp_imap->stream(), $this->_server, $val['name'] . '*');
                if (!empty($tmp)) {
                    $sublist = array_merge($sublist, $tmp);
                }
            }

            if (!empty($sublist)) {
                foreach ($sublist as $val) {
                    $this->_subscribed[substr($val, strpos($val, '}') + 1)] = 1;
                }
            }
        }
    }

    /**
     * Initialize the list of unsubscribed mailboxes.
     *
     * @access private
     */
    function _initUnsubscribed()
    {
        if (is_null($this->_unsubscribed)) {
            $this->_changed = true;
            $this->_unsubscribed = array();

            // If subscription mode is off, we know there are no unsubscribed
            // mailboxes.
            if (!$GLOBALS['prefs']->getValue('subscribe')) {
                return;
            }

            $this->_initSubscribed();
            $all_list = array();

            $imp_imap = &IMP_IMAP::singleton();

            /* Get list of all mailboxes. */
            foreach ($this->_namespaces as $val) {
                $tmp = @imap_list($imp_imap->stream(), $this->_server, $val['name'] . '*');
                if (!empty($tmp)) {
                    $all_list = array_merge($all_list, $tmp);
                }
            }

            if (!empty($all_list)) {
                /* Find all mailboxes that aren't in the subscribed list. */
                foreach ($all_list as $val) {
                    $val = substr($val, strpos($val, '}') + 1);
                    if (!isset($this->_subscribed[$val])) {
                        $this->_unsubscribed[$val] = 1;
                    }
                }
            }
        }
    }

    /**
     * Should we expand all elements?
     */
    function expandAll()
    {
        foreach ($this->_parent[IMPTREE_BASE_ELT] as $val) {
            $this->expand($val, true);
        }
    }

    /**
     * Should we collapse all elements?
     */
    function collapseAll()
    {
        foreach ($this->_tree as $key => $val) {
            if ($key !== IMPTREE_BASE_ELT) {
                $this->collapse($val['v']);
            }
        }

        /* Clear all entries from the expanded list. */
        $this->_initExpandedList();
        foreach ($this->_expanded as $key => $val) {
            $this->_removeExpandedList($key);
        }

    }

    /**
     * Return the list of mailboxes in the next level.
     *
     * @access private
     *
     * @param string $id  The current mailbox.
     *
     * @return array  A list of mailbox objects or the empty list.
     *                See _getList() for format.
     */
    function _childrenInfo($id)
    {
        $info = array();
        $ns_info = $this->_getNamespace($id);

        if (is_null($ns_info)) {
            return $info;
        }

        if (($id == 'INBOX') &&
            ($ns_info['name'] == ('INBOX' . $ns_info['delimiter']))) {
            $search = $id . $ns_info['delimiter'] . $ns_info['name'];
        } else {
            $search = $id . $ns_info['delimiter'];
        }

        $info = $this->_getList($search . '%');

        if (isset($this->_tree[$id])) {
            $this->_setChildren($this->_tree[$id], !empty($info));
        }

        return $info;
    }

    /**
     * Switch subscribed/unsubscribed viewing.
     *
     * @param boolean $unsub  Show unsubscribed elements?
     */
    function showUnsubscribed($unsub)
    {
        if ($unsub === $this->_showunsub) {
            return;
        }

        $this->_showunsub = $unsub;
        $this->_changed = true;

        /* If we are switching from unsubscribed to subscribed, no need
         * to do anything (we just ignore unsubscribed stuff). */
        if ($unsub === false) {
            return;
        }

        /* If we are switching from subscribed to unsubscribed, we need
         * to add all unsubscribed elements that live in currently
         * discovered items. */
        $this->_unsubview = true;
        $this->_initUnsubscribed();
        if (empty($this->_unsubscribed)) {
            return;
        }

        $this->_initmode = IMPTREE_INIT_UNSUB;
        $this->insert(array_keys($this->_unsubscribed));
        $this->_initmode = 0;
    }

    /**
     * Get information about new/unseen/total messages for the given element.
     *
     * @param string $name  The element name.
     *
     * @return array  Array with the following fields:
     * <pre>
     * 'messages'  --  Number of total messages.
     * 'newmsg'    --  Number of new messages.
     * 'unseen'    --  Number of unseen messages.
     * </pre>
     */
    function getElementInfo($name)
    {
        $status = array();

        require_once IMP_BASE . '/lib/IMAP/Cache.php';
        $imap_cache = &IMP_IMAP_Cache::singleton();
        $sts = $imap_cache->getStatus(null, $name);
        if (!empty($sts)) {
            $status = array(
                'messages' => $sts->messages,
                'unseen' => (isset($sts->unseen) ? $sts->unseen : 0),
                'newmsg' => (isset($sts->recent) ? $sts->recent : 0)
            );
        }

        return $status;
    }

    /**
     * Sorts a list of mailboxes.
     *
     * @access private
     *
     * @param array &$mbox   The list of mailboxes to sort.
     * @param boolean $base  Are we sorting a list of mailboxes in the base
     *                       of the tree.
     */
    function _sortList(&$mbox, $base = false)
    {
        if (is_null($this->_imap_sort)) {
            require_once 'Horde/IMAP/Sort.php';
            $this->_imap_sort = &new IMAP_Sort($this->_delimiter);
        }

        if ($base) {
            $basesort = array();
            foreach ($mbox as $val) {
                $basesort[$val] = ($val == 'INBOX') ? 'INBOX' : $this->_tree[$val]['l'];
            }
            $this->_imap_sort->sortMailboxes($basesort, true, true, true);
            $mbox = array_keys($basesort);
        } else {
            $this->_imap_sort->sortMailboxes($mbox, true);
        }

        if ($base) {
            for ($i = 0, $count = count($mbox); $i < $count; ++$i) {
                if ($this->_isNonIMAPElt($this->_tree[$mbox[$i]])) {
                    /* Already sorted by name - simply move to the end of
                     * the array. */
                    $mbox[] = $mbox[$i];
                    unset($mbox[$i]);
                }
            }
            $mbox = array_values($mbox);
        }
    }

    /**
     * Is the given element an "active" element (i.e. an element that should
     * be worked with given the current viewing parameters).
     *
     * @access private
     *
     * @param array $elt  A tree element.
     *
     * @return boolean  True if it is an active element.
     */
    function _activeElt($elt)
    {
        return ($this->_showunsub ||
                ($this->isSubscribed($elt) && !$this->isContainer($elt)) ||
                $this->hasChildren($elt));
    }

    /**
     * Convert a mailbox name to the correct, internal name (i.e. make sure
     * INBOX is always capitalized for IMAP servers).
     *
     * @access private
     *
     * @param string $name  The mailbox name.
     *
     * @return string  The converted name.
     */
    function _convertName($name)
    {
        return (strcasecmp($name, 'INBOX') == 0) ? 'INBOX' : $name;
    }

    /**
     * Get namespace info for a full folder path.
     *
     * @access private
     *
     * @param string $mailbox  The folder path.
     *
     * @return mixed  The namespace info for the folder path or null if the
     *                path doesn't exist.
     */
    function _getNamespace($mailbox)
    {
        if (!in_array($mailbox, array(IMPTREE_OTHER_KEY, IMPTREE_SHARED_KEY, IMPTREE_VFOLDER_KEY)) &&
            (strpos($mailbox, IMPTREE_VFOLDER_KEY . $this->_delimiter) !== 0)) {
            return IMP::getNamespace($mailbox);
        }
        return null;
    }

    /**
     * Set the start point for determining element differences via eltDiff().
     *
     * @since Horde 3.1
     */
    function eltDiffStart()
    {
        $this->_eltdiff = Util::cloneObject($this->_tree);
    }

    /**
     * Return the list of elements that have changed since nodeDiffStart()
     * was last called.
     *
     * @since Horde 3.1
     *
     * @return array  An array with the following keys:
     * <pre>
     * 'a' => A list of elements that have been added.
     * 'c' => A list of elements that have been changed.
     * 'd' => A list of elements that have been deleted.
     * </pre>
     *                Returns false if no changes have occurred.
     */
    function eltDiff()
    {
        if (!$this->_changed || !$this->_eltdiff) {
            return false;
        }

        $added = $changed = $deleted = array();

        /* Determine the deleted items. */
        $deleted = array_values(array_diff(array_keys($this->_eltdiff), array_keys($this->_tree)));

        foreach ($this->_tree as $key => $val) {
            if (!isset($this->_eltdiff[$key])) {
                $added[] = $key;
            } elseif ($val != $this->_eltdiff[$key]) {
                $changed[] = $key;
            }
        }

        if (empty($added) && empty($changed) && empty($deleted)) {
            return false;
        } else {
            return array('a' => $added, 'c' => $changed, 'd' => $deleted);
        }
    }

    /**
     * Inserts virtual folders into the tree.
     *
     * @param array $id_list  An array with the folder IDs to add as the key
     *                        and the labels as the value.
     */
    function insertVFolders($id_list)
    {
        if (empty($id_list) ||
            empty($GLOBALS['conf']['user']['allow_folders'])) {
            return;
        }

        $adds = $id = array();

        foreach ($id_list as $key => $val) {
            $id[$GLOBALS['imp_search']->createSearchID($key)] = $val;
        }

        foreach (array_keys($id) as $key) {
            $adds[] = IMPTREE_VFOLDER_KEY . $this->_delimiter . $key;
        }

        $this->_initmode = IMPTREE_INIT_SUB;
        $this->insert($adds);
        $this->_initmode = 0;

        foreach ($id as $key => $val) {
            $this->_tree[$key]['l'] = $val;
        }

        /* Sort the Virtual Folder list in the object, if necessary. */
        if ($this->_needSort($this->_tree[IMPTREE_VFOLDER_KEY])) {
            $vsort = array();
            foreach ($this->_parent[IMPTREE_VFOLDER_KEY] as $val) {
                $vsort[$val] = $this->_tree[$val]['l'];
            }
            natcasesort($vsort);
            $this->_parent[IMPTREE_VFOLDER_KEY] = array_keys($vsort);
            $this->_setNeedSort($this->_tree[IMPTREE_VFOLDER_KEY], false);
            $this->_changed = true;
        }
    }

    /**
     * Builds a list of folders, suitable to render a folder tree.
     *
     * @param integer $mask  The mask to pass to next().
     * @param boolean $open  If using the base folder icons, display a
     *                       different icon whether the folder is opened or
     *                       closed.
     *
     * @return array  An array with three elements: the folder list, the total
     *                number of new messages, and a list with folder names
     *                suitable for user interaction.
     *                The folder list array contains the following added
     *                entries on top of the entries provided by element():
     * <pre>
     * 'display' - The mailbox name run through IMP::displayFolder().
     * 'peek' - See peek().
     * </pre>
     */
    function build($mask = 0, $open = true)
    {
        $displayNames = $newmsgs = $rows = array();
        $this->_forceopen = $open;

        /* Start iterating through the list of mailboxes, displaying them. */
        $mailbox = $this->reset();
        do {
            $row = $this->element($mailbox['v']);

            $row['display'] = ($this->_isNonIMAPElt($mailbox)) ? $mailbox['l'] : IMP::displayFolder($mailbox['v']);
            $row['peek'] = $this->peek();

            if (!empty($row['newmsg'])) {
                $newmsgs[$row['value']] = $row['newmsg'];
            }

            /* Hide folder prefixes from the user. */
            if ($row['level'] >= 0) {
                $rows[] = $row;
                $displayNames[] = addslashes($row['display']);
            }
        } while (($mailbox = $this->next($mask)));

        $this->_forceopen = false;

        return array($rows, $newmsgs, $displayNames);
    }

    /**
     * Get any custom icon configured for the given element.
     *
     * @params array $elt  A tree element.
     *
     * @return array  An array with the 'icon', 'icondir', and 'alt'
     *                information for the element, or false if no icon
     *                available.
     */
    function getCustomIcon($elt)
    {
        static $mbox_icons;

        if (isset($mbox_icons) && !$mbox_icons) {
            return false;
        }

        /* Call the mailbox icon hook, if requested. */
        if (empty($GLOBALS['conf']['hooks']['mbox_icon'])) {
            $mbox_icons = false;
            return false;
        }

        if (!isset($mbox_icons)) {
            $mbox_icons = Horde::callHook('_imp_hook_mbox_icons', array(),
                                          'imp', false);
            if (!$mbox_icons) {
                return false;
            }
        }

        if (isset($mbox_icons[$elt['v']])) {
            return array('icon' => $mbox_icons[$elt['v']]['icon'], 'icondir' => $mbox_icons[$elt['v']]['icondir'], 'alt' => $mbox_icons[$elt['v']]['alt']);
        }

        return false;
    }

    /**
     * Returns whether this element is a virtual folder.
     *
     * @param array $elt  A tree element.
     *
     * @return integer  True if the element is a virtual folder.
     */
    function isVFolder($elt)
    {
        return $elt['a'] & IMPTREE_ELT_VFOLDER;
    }

    /**
     * Rename a current folder.
     *
     * @since IMP 4.1
     *
     * @param array $old  The old folder names.
     * @param array $new  The new folder names.
     */
    function rename($old, $new)
    {
        foreach ($old as $key => $val) {
            $polled = (isset($this->_tree[$val])) ? $this->isPolled($this->_tree[$val]) : false;
            if ($this->delete($val)) {
                $this->insert($new[$key]);
                if ($polled) {
                    $this->addPollList($new[$key]);
                }
            }
        }
    }

    /**
     * Returns a list of all IMAP folders in the tree
     *
     * @since IMP 4.1
     *
     * @param integer $mask  A mask with the following elements:
     * <pre>
     * IMPTREE_FLIST_CONTAINER - Show container elements.
     * IMPTREE_FLIST_UNSUB - Show unsubscribed elements.
     * IMPTREE_FLIST_OB - Return full tree object.
     * </pre>
     *
     * @return array  An array of IMAP mailbox names.
     */
    function folderList($mask = 0)
    {
        if ($_SESSION['imp']['base_protocol'] == 'pop3') {
            return array('INBOX');
        }

        $ret_array = array();

        $diff_unsub = (($mask & IMPTREE_FLIST_UNSUB) != $this->_showunsub) ? $this->_showunsub : null;
        $this->showUnsubscribed($mask & IMPTREE_FLIST_UNSUB);

        $mailbox = $this->reset();
        do {
            if ((($mask & IMPTREE_FLIST_CONTAINER) ||
                 !$this->isContainer($mailbox)) &&
                !$this->isVFolder($mailbox)) {
                $ret_array[] = ($mask & IMPTREE_FLIST_OB) ? $mailbox : $mailbox['v'];
            }
        } while (($mailbox = $this->next(IMPTREE_NEXT_SHOWCLOSED)));

        if (!is_null($diff_unsub)) {
            $this->showUnsubscribed($diff_unsub);
        }

        return $ret_array;
    }

    /**
     * Is the mailbox open in the sidebar?
     *
     * @since IMP 4.1.1
     *
     * @param array $mbox  A mailbox name.
     *
     * @return integer  True if the mailbox is open in the sidebar.
     */
    function isOpenSidebar($mbox)
    {
        switch ($GLOBALS['prefs']->getValue('nav_expanded_sidebar')) {
        case IMPTREE_OPEN_USER:
            $this->_initExpandedList();
            return !empty($this->_expanded[$mbox]);
            break;

        case IMPTREE_OPEN_ALL:
            return true;
            break;

        case IMPTREE_OPEN_NONE:
        default:
            return false;
            break;
        }
    }

    /**
     * Init frequently used element() data.
     *
     * @access private
     */
    function _initElement()
    {
        if (!empty($this->_elt)) {
            return;
        }

        global $prefs, $registry;

        /* Initialize the user's identities. */
        require_once 'Horde/Identity.php';
        $identity = &Identity::singleton(array('imp', 'imp'));

        $this->_elt = array(
            'trash' => IMP::folderPref($prefs->getValue('trash_folder'), true),
            'draft' => IMP::folderPref($prefs->getValue('drafts_folder'), true),
            'spam' => IMP::folderPref($prefs->getValue('spam_folder'), true),
            'sent' => $identity->getAllSentmailFolders(),
            'image_dir' => $registry->getImageDir(),
        );
    }

    /**
     * Return extended information on an element.
     *
     * @since IMP 4.2
     *
     * @param string $name  The name of the tree element.
     *
     * @return array  Returns the element with extended information, or false
     *                if not found.  The information returned is as follows:
     * <pre>
     * 'alt' - The alt text for the icon.
     * 'base_elt' - The return from get().
     * 'children' - Does the element have children?
     * 'container' - Is this a container element?
     * 'editvfolder' - Can this virtual folder be edited?
     * 'icon' - The name of the icon graphic to use.
     * 'icondir' - The path of the icon directory.
     * 'level' - The deepness level of this element.
     * 'mbox_val' - A html-ized version of 'value'.
     * 'msgs' - The number of total messages in the element (if polled).
     * 'name' - A html-ized version of 'label'.
     * 'newmsg' - The number of new messages in the element (if polled).
     * 'parent' - The parent element value.
     * 'polled' - Show polled information?
     * 'special' - An integer mask indicating if this is a "special" element.
     * 'specialvfolder' - Is this a "special" virtual folder?
     * 'unseen' - The number of unseen messages in the element (if polled).
     * 'user_icon' - Use a user defined icon?
     * 'value' - The value of this element (i.e. element id).
     * 'vfolder' - Is this a virtual folder?
     * </pre>
     */
    function element($mailbox)
    {
        $mailbox = $this->get($mailbox);
        if (!$mailbox) {
            return false;
        }

        $this->_initElement();

        $row = array(
            'base_elt' => $mailbox,
            'children' => $this->hasChildren($mailbox, true),
            'container' => false,
            'editvfolder' => false,
            'icondir' => $this->_elt['image_dir'],
            'iconopen' => null,
            'level' => $mailbox['c'],
            'mbox_val' => htmlspecialchars($mailbox['v']),
            'name' => htmlspecialchars($mailbox['l']),
            'newmsg' => 0,
            'parent' => $mailbox['p'],
            'polled' => false,
            'special' => 0,
            'specialvfolder' => false,
            'user_icon' => false,
            'value' => $mailbox['v'],
            'vfolder' => false,
        );

        $icon = $this->getCustomIcon($mailbox);

        if (!$this->isContainer($mailbox)) {
            /* We are dealing with mailboxes here.
             * Determine if we need to poll this mailbox for new messages. */
            if ($this->isPolled($mailbox)) {
                /* If we need message information for this folder, update
                 * it now. */
                $msgs_info = $this->getElementInfo($mailbox['v']);
                if (!empty($msgs_info)) {
                    $row['polled'] = true;
                    if (!empty($msgs_info['newmsg'])) {
                        $row['newmsg'] = $msgs_info['newmsg'];
                    }
                    $row['msgs'] = $msgs_info['messages'];
                    $row['unseen'] = $msgs_info['unseen'];
                }
            }


            switch ($mailbox['v']) {
            case 'INBOX':
                $row['icon'] = 'folders/inbox.png';
                $row['alt'] = _("Inbox");
                $row['special'] = IMPTREE_SPECIAL_INBOX;
                break;

            case $this->_elt['trash']:
                if ($GLOBALS['prefs']->getValue('use_vtrash')) {
                    $row['icon'] = ($this->isOpen($mailbox)) ? 'folders/folder_open.png' : 'folders/folder.png';
                    $row['alt'] = _("Mailbox");
                } else {
                    $row['icon'] = 'folders/trash.png';
                    $row['alt'] = _("Trash folder");
                    $row['special'] = IMPTREE_SPECIAL_TRASH;
                }
                break;

            case $this->_elt['draft']:
                $row['icon'] = 'folders/drafts.png';
                $row['alt'] = _("Draft folder");
                $row['special'] = IMPTREE_SPECIAL_DRAFT;
                break;

            case $this->_elt['spam']:
                $row['icon'] = 'folders/spam.png';
                $row['alt'] = _("Spam folder");
                $row['special'] = IMPTREE_SPECIAL_SPAM;
                break;

            default:
                if (in_array($mailbox['v'], $this->_elt['sent'])) {
                    $row['icon'] = 'folders/sent.png';
                    $row['alt'] = _("Sent mail folder");
                    $row['special'] = IMPTREE_SPECIAL_SENT;
                } else {
                    $row['icon'] = ($this->isOpen($mailbox)) ? 'folders/folder_open.png' : 'folders/folder.png';
                    $row['alt'] = _("Mailbox");
                }
                break;
            }

            /* Virtual folders. */
            if ($this->isVFolder($mailbox)) {
                $row['vfolder'] = true;
                $row['editvfolder'] = $GLOBALS['imp_search']->isEditableVFolder($mailbox['v']);
                if ($GLOBALS['imp_search']->isVTrashFolder($mailbox['v'])) {
                    $row['specialvfolder'] = true;
                    $row['icon'] = 'folders/trash.png';
                    $row['alt'] = _("Virtual Trash Folder");
                } elseif ($GLOBALS['imp_search']->isVINBOXFolder($mailbox['v'])) {
                    $row['specialvfolder'] = true;
                    $row['icon'] = 'folders/inbox.png';
                    $row['alt'] = _("Virtual INBOX Folder");
                }
            }
        } else {
            /* We are dealing with folders here. */
            $row['container'] = true;
            if ($this->_forceopen && $this->isOpen($mailbox)) {
                $row['icon'] = 'folders/folder_open.png';
                $row['alt'] = _("Opened Folder");
            } else {
                $row['icon'] = 'folders/folder.png';
                $row['iconopen'] = 'folders/folder_open.png';
                $row['alt'] = ($this->_forceopen) ? _("Closed Folder") : _("Folder");
            }

            if ($this->isNamespace($mailbox) &&
                $this->_isNonIMAPElt($mailbox)) {
                $row['children'] = true;
            }
        }

        /* Overwrite the icon information now. */
        if (!empty($icon)) {
            $row['icon'] = $icon['icon'];
            $row['icondir'] = $icon['icondir'];
            if (!empty($icon['alt'])) {
                $row['alt'] = $icon['alt'];
            }
            $row['iconopen'] = isset($icon['iconopen']) ? $icon['iconopen'] : null;
            $row['user_icon'] = true;
        }

        return $row;
    }

    /**
     * Sort a level in the tree.
     *
     * @access private
     *
     * @param string $id  The parent folder whose children need to be sorted.
     */
    function _sortLevel($id)
    {
        if ($this->_needSort($this->_tree[$id])) {
            $this->_sortList($this->_parent[$id], ($id === IMPTREE_BASE_ELT));
            $this->_setNeedSort($this->_tree[$id], false);
            $this->_changed = true;
        }
    }

    /**
     * Determines the mailbox name to create given a parent and the new name.
     *
     * @param string $parent  The parent name.
     * @param string $parent  The new mailbox name.
     *
     * @return string  The full path to the new mailbox, or PEAR_Error.
     */
    function createMailboxName($parent, $new)
    {
        $ns_info = (empty($parent)) ? IMP::defaultNamespace() : $this->_getNamespace($parent);
        if (is_null($ns_info)) {
            if ($this->isNamespace($this->_tree[$parent])) {
                $ns_info = $this->_getNamespace($new);
                if (in_array($ns_info['type'], array('other', 'shared'))) {
                    return $new;
                }
            }
            return PEAR::raiseError(_("Can not directly create mailbox in this folder."), 'horde.error');
        }

        $mbox = $ns_info['name'];
        if (!empty($parent)) {
            $mbox .= rtrim(substr_replace($parent, '', 0, strlen($ns_info['name'])), $ns_info['delimiter']) . $ns_info['delimiter'];
        }
        return $mbox . $new;
    }

}
