/* DataSet -- an expandable set of data of different types (eg: string, bool, int, double)
 */

#ifndef DATASET_H
#define DATASET_H

#include "common.h"

#include <iostream>
#include <vector>
#include <string>
#include <cstdlib>
#include <cstdio>
#include <cassert>
using namespace std;

class BadDataType {};
class InvalidIndex {};

class Data
{
public:
        Data(const Data& other) : dtype(DATA_TYPE_INT) {
                *this = other;
        };
        Data& operator=(const Data& other);
        Data(const string& dt) : dtype(DATA_TYPE_STRING) {stringData = new string(dt);};
        Data(const int dt) : dtype(DATA_TYPE_INT), intData(dt) {};
        Data(const double dt) : dtype(DATA_TYPE_DOUBLE), doubleData(dt) {};
        ~Data() {
                if (dtype == DATA_TYPE_STRING)
                        delete stringData;
        };

        string getString() const;
        int getInt() const throw(BadDataType);
        bool getBool() const throw(BadDataType);
        double getDouble() const throw(BadDataType);
        
        enum DATA_TYPE {DATA_TYPE_STRING, DATA_TYPE_INT, DATA_TYPE_BOOL, DATA_TYPE_DOUBLE};
        DATA_TYPE getType() const {return dtype;};
private:
        DATA_TYPE dtype;
        union {
                string* stringData;
                int intData;
                bool boolData;
                double doubleData;
        };
};

class DataSet
{
public:
        // there is no bool constructor here because char* get implicitly cast to bool
        // before string
        explicit DataSet(const string& dt);
        explicit DataSet(int dt) {storage.push_back(Data(dt));}
        explicit DataSet(double dt) {storage.push_back(Data(dt));}
        DataSet() {};

        DataSet& operator=(const DataSet& other) {
                storage = other.storage;
                return *this;
        }

        void addString(const string& dt) {storage.push_back(Data(dt));}
        void addInt(int dt) {storage.push_back(Data(dt));}
        void addDouble(double dt) {storage.push_back(Data(dt));}
        void addBool(bool dt) {storage.push_back(Data(dt));}
        void add(Data dt) {storage.push_back(dt);}
        void addDataSet(const DataSet& other);

        // remove any empty strings
        void noBlanks();

        void removeEntry(int n) {
                assert(validIndex(n));
                vector<Data>::iterator t = storage.begin();
                t += n;
                storage.erase(t);
        }                

        // suitable for displaying to the user
        string toString() const;
        // escaped string - suitable as a serialised form of this class
        string asString() const;
        
        void display(ostream& os) const {os << toString().c_str();};
        int count() const {return storage.size();};
        bool validIndex(int index) const {return index < int(storage.size());};

        bool getBool(int index = 0) const throw(BadDataType, InvalidIndex) {
                if (!validIndex(index))
                        throw InvalidIndex();
                return storage[index].getBool();
        }
        int getInt(int index = 0) const throw(BadDataType, InvalidIndex) {
                if (!validIndex(index))
                        throw InvalidIndex();
                return storage[index].getInt();
        }
        double getDouble(int index = 0) const throw(BadDataType, InvalidIndex) {
                if (!validIndex(index))
                        throw InvalidIndex();
                return storage[index].getDouble();
        }
        string getString(int index = 0) const throw(InvalidIndex) {
                if (!validIndex(index))
                        throw InvalidIndex();
                return storage[index].getString();
        }
        void clear() {
                storage.clear();
        }                

        const Data& operator[](int index) const {assert(validIndex(index)); return storage[index];};
private:
        vector<Data> storage;
};

inline ostream& operator<<(ostream& os, const DataSet& dtset)
{
        dtset.display(os);
        return os;
}

#endif
