/*
    Copyright (C) 2000 Steve Brown
    Copyright (C) 2000,2001,2002 Guillaume Morin, Alcve

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    $Id: group.c,v 1.131 2003/05/12 14:59:18 gmorin Exp $
    $Source: /cvsroot/nss-mysql/nss-mysql/src/group.c,v $
    $Date: 2003/05/12 14:59:18 $
    $Author: gmorin $
*/


#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <grp.h>

#include "nss-mysql.h"

#include "lib.h"
#include "parser.h"
#include "passwd.h"
#include "group.h"
#include "ent.h"

#if USE_GROUP

/* This structure is used to share information between
 * all ***grent functions. This is a cache of all GIDs */

NSS_STATUS _nss_mysql_initgroups (const char *user, gid_t group, long *start,
              long int *size, gid_t * groups, long int limit,int *errnop);
NSS_STATUS _nss_mysql_setgrent (void);
NSS_STATUS _nss_mysql_endgrent (void);
NSS_STATUS _nss_mysql_getgrent_r (struct group *gr,
                char * buffer, size_t buflen,int * errnop);
NSS_STATUS _nss_mysql_getgrnam_r (const char * name, struct group *gr,
                char * buffer, size_t buflen,int *errnop);
NSS_STATUS _nss_mysql_getgrgid_r (const gid_t gid, struct group *gr,
                char * buffer, size_t buflen,int *errnop);
static void prepare_forkhandler(void);
static void parent_forkhandler(void);
static void child_forkhandler(void);
static void atexit_handler(void);
static NSS_STATUS handle_query(struct query * q,struct mysql_auth * mysql_auth,
                struct group * gr, char * buffer, size_t buflen, int * errnop);
static char * get_sql_query(struct query * q, struct parse_result * pr,
                struct mysql_auth * auth);

/*EOP*/

static int forkhandler_isset = 0;
static pthread_mutex_t forkhandler_mutex = NSS_MYSQL_PTHREAD_MUTEX_INITIALIZER;

/* get_sql_query:
 * 
 * Returns a complete SQL query string corresponding to the query
 * 
 * q: request:
 * pr: value gathered by the conf files parsing
 * auth: connection information
 */

char * get_sql_query(struct query * q, struct parse_result * pr,
                struct mysql_auth * auth) {
        struct groupoptions * goptions = pr->goptions_p;
        struct passwdoptions * options = pr->options_p;
        char * sql = NULL;
        char * e_name=NULL;
        char buffer[64];
        int call_free;

        if (DEBUG) {
                if (q->name) {
                        /* looking by name */
                        _nss_mysql_log(LOG_ERR,"group.c: get_query "
                                        "called for user %s",q->name);
                } else {
                        /* looking by gid */
                        _nss_mysql_log(LOG_ERR,"group.c: get_query "
                                        "called for gid %d",q->gid);
                }
        }
        
        /* escaping name */
        if (q->name)	{
                e_name = _nss_mysql_escape_string(q->name,auth,
                                buffer, sizeof buffer, &call_free);
                if (! e_name) {
                        return NULL;
                }
        }
        
        if (q->name) {
                /* we look by name */
                sql = _nss_mysql_sqlprintf("select %s,%s,%s,%s from %s "
                                "LEFT JOIN %s on %s=%s LEFT JOIN %s on %s=%s "
                                "and %s where %s = '%s' and %s order by "
                                "%s,%s desc",
                        goptions->groupnamecolumn,
                        goptions->gidcolumn,
                        goptions->passwordcolumn,
                        options->usercolumn,
                        /* from */
                        goptions->groupinfotable,
                        /* left join */
                        goptions->memberstable,
                        /* on */
                        goptions->groupidcolumn,
                        goptions->mgroupidcolumn,
                        /* left join */
                        options->table,
                        /* on */
                        goptions->museridcolumn,
                        options->useridcolumn,
                        /* and */
                        options->where[0] ? options->where : "1=1",
                        /* where */
                        goptions->groupnamecolumn,
                        e_name,
                        goptions->where[0] ? goptions->where : "1=1",
                        /* order by */
                        goptions->groupnamecolumn,
                        options->usercolumn);
                if (call_free)
                        free(e_name);
        } else {
                /* we look by GID */
                sql = _nss_mysql_sqlprintf("select %s,%s,%s,%s from %s "
                                "LEFT JOIN %s on %s=%s LEFT JOIN %s on %s=%s "
                                "and %s where %s = %d and %s order by %s"
                                ",%s desc",
                        goptions->groupnamecolumn,
                        goptions->gidcolumn,
                        goptions->passwordcolumn,
                        options->usercolumn,
                        /* from */
                        goptions->groupinfotable,
                        /* left join */
                        goptions->memberstable,
                        /* on */
                        goptions->groupidcolumn,
                        goptions->mgroupidcolumn,
                        /* left join */
                        options->table,
                        /* on */
                        goptions->museridcolumn,
                        options->useridcolumn,
                        /* and */
                        options->where[0] ? options->where : "1=1",
                        /* where */
                        /* for bulk retrieval, we retrieve every entry */
                        q->bulk ? "1" : goptions->gidcolumn,
                        q->bulk ? 1 : q->gid,
                        goptions->where[0] ? goptions->where : "1=1",
                        /* order by */
                        goptions->groupnamecolumn,
                        options->usercolumn);
        }

        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"group.c: get_query: SQL statement is "
                                "%s",sql);
        return sql;
}

/* handle_query:
 *
 * generate a sql query, send it, parse the result and store it in gr according
 * to the nss-mysql query "q"
 *
 * q: query to handle
 * mysql_auth: connection information
 * gr: to be filled
 * buffer: buffer to save dynamic data
 * buflen: sizeof buffer
 * errnop: pointer to errno
 */

NSS_STATUS handle_query(struct query * q,struct mysql_auth * mysql_auth,
                        struct group * gr, char * buffer, size_t buflen,
                        int * errnop) {
        struct passwdoptions options;
        struct groupoptions goptions;
        struct parse_result pr = {&options,&goptions,NULL};
        NSS_STATUS status;
        MYSQL_RES * result = NULL;
        char * sql_query = NULL;
        /*int call_mte = 0;*/

        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"group.c: handle_query called for "
                                "name %s, gid %d, bulk %d",q->name,q->gid,
                                q->bulk);
        
        memset(&options,0,sizeof(struct passwdoptions));
        memset(&goptions,0,sizeof(struct groupoptions));

        if (! _nss_mysql_read_conf_file("groups",&pr)) {
                _nss_mysql_log(LOG_ERR,"group.c: handle_query: conf file "
                                "parsing failed");
                status = NSS_STATUS_UNAVAIL;
                *errnop = ENOENT;
                goto out;
        }

        _nss_mysql_set_fork_handler(&forkhandler_isset,&forkhandler_mutex,
                        prepare_forkhandler,
                        parent_forkhandler,
                        child_forkhandler,
                        atexit_handler);

        if (! _nss_mysql_check_connection(mysql_auth,&options.con)) {
                status = NSS_STATUS_UNAVAIL;
                *errnop = ENOENT;
                goto out;
        }
       
        /*call_mte = 1;*/
        sql_query = get_sql_query(q,&pr,mysql_auth);
        if (! sql_query) {
                *errnop = EAGAIN;
                status = NSS_STATUS_TRYAGAIN;
                /* we need to unlock the mutex because 
                 * _nss_mysql_check_connection locks it
                 */
                pthread_mutex_unlock(mysql_auth->mutex);
                goto out;
        }
        
        status = _nss_mysql_send_query(mysql_auth,sql_query,&result,errnop);
        if (status != NSS_STATUS_SUCCESS)
                goto out;

        status = _nss_mysql_group_result_to_struct(gr,result,errnop,buffer,
                        buflen,0);
out:
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"group.c: handle_query(): reached "
                                "\"out\" label, result %p",result);
        s_free(sql_query);
        if (result)
                mysql_free_result(result);
        _nss_mysql_free_users(&options);
        _nss_mysql_free_groups(&goptions);
        /*
        if (call_mte)
                my_thread_end();
        */
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"group.c: leaving handle_query with "
                                "status %d (errno %s)",status,
                                strerror(*errnop));
        return status;
}

/* _nss_mysql_group_result_to_struct
 * Translate a set of rows (available in the result arg)
 * into a proper group structure (in the gr arg) using the
 * current MySQL connection (in the mysql_auth arg).  
 * The errnop argument will be set if an error occurs.  
 *
 */

NSS_STATUS _nss_mysql_group_result_to_struct(struct group * gr, 
                MYSQL_RES *result, 
                int * errnop, 
                char * buffer,size_t buflen, unsigned long max) {
        MYSQL_ROW sql_row;
        unsigned long num_rows;
        int error,first;
        char ** addrptr=NULL;
        char * end_of_buf;
        MYSQL_ROW_OFFSET last, initial_offset = mysql_row_tell(result);

        *errnop = ENOENT;


        num_rows = mysql_num_rows(result);

        if (num_rows == 0) {
                if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to"
                                "_struct: not found");
                return NSS_STATUS_NOTFOUND;
        }
        
        if (max && num_rows > max) {
                _nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to_struct: "
                                "Warning query returned %lu rows, was "
                                "expecting only %lu. Results will be "
                                "incorrect.",num_rows,max);
        }
        
        sql_row = mysql_fetch_row(result);
        if (sql_row == NULL) {
                if (DEBUG) 
                        _nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to_"
                                        "struct: mysql_fetch_row failed. "
                                        "End of bulk retrieval");
                return NSS_STATUS_NOTFOUND;
        }

        
        gr->gr_gid = _nss_mysql_strtol(sql_row[1],-1,&error);
        if (error) {
                _nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to_struct: gid "
                                "field(%s) cannot be converted to an integer. "
                                "Fix your database.", sql_row[1]);
                return NSS_STATUS_UNAVAIL;
        }
                
        if (_nss_mysql_isempty(sql_row[0])) {
                _nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to_struct: "
                                "name is empty for GID (%d). Fix your "
                                "database.",gr->gr_gid);
                return NSS_STATUS_UNAVAIL;
        }
        gr->gr_name = _nss_mysql_copy_to_buffer(&buffer,&buflen,sql_row[0]);
        if (! gr->gr_name)
                goto out_nomem;

        if (_nss_mysql_isempty(sql_row[2])) {
                _nss_mysql_log(LOG_ERR,"g_fill_strict: %s has an empty or null "
                                "password. Fix your database.",gr->gr_name);
                gr->gr_passwd = _nss_mysql_copy_to_buffer(&buffer,&buflen,"x");
        } else 
                gr->gr_passwd = _nss_mysql_copy_to_buffer(&buffer,&buflen,
                                sql_row[2]);
        if (! gr->gr_passwd)
                goto out_nomem;

        if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to_struct: "
                        "%s's gid is %d",gr->gr_name,gr->gr_gid);

        addrptr = (char **) buffer;
        gr->gr_mem = addrptr;
        
        end_of_buf = buffer+buflen-1;
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,
                                "nss_mysql_group_to_struct: "
                                "addr %p, data %p",addrptr,
                                buffer);

        /* we fill the members array */
        first = 1;
        last = mysql_row_tell(result);
        while(first || (sql_row=mysql_fetch_row(result))) {
                char * p,* tmp;
               
                if (strcmp(sql_row[0],gr->gr_name) != 0) {
                        /* this is another group */
                        mysql_row_seek(result,last);
                        break;
                }

                /* user name MUST not be empty */
                if (_nss_mysql_isempty(sql_row[3])) {
                        if (first) {
                                /* 
                                 * this is the first row we process. If 
                                 * user_name is NULL, this is not an error
                                 * It only means that the group has no
                                 * members
                                 */
                                if (DEBUG) 
                                        _nss_mysql_log(LOG_ERR,"_nss_mysql_gro"
                                                        "up_to_result: no mem"
                                                        "bers in %s.",
                                                        gr->gr_name);
                                break;
                        } else {
                                /* empty user name in the DB 
                                _nss_mysql_log(LOG_ERR,"Empty or NULL member "
                                        "for group %s(%d). Fix your "
                                        "database", gr->gr_name,
                                        gr->gr_gid);
                                return NSS_STATUS_UNAVAIL;
                                */
                                /* should be a disabled user */
                                goto end_of_loop;
                        }
                }

                first = 0;

                end_of_buf -= strlen(sql_row[3])+1;
                if ((void *) addrptr >= (void *) end_of_buf)
                        goto out_nomem;

                /* we do not want end_of_buf to change, so we use tmp */
                tmp = end_of_buf;
                p = _nss_mysql_copy_to_buffer(&tmp,NULL,sql_row[3]);

                /* duplicate check */
                if (! p)
                        goto out_nomem;

                /* we save the string address in the array */
                *addrptr = p;
                ++addrptr;
                if (DEBUG) _nss_mysql_log(LOG_ERR,"%s is a member of %s"
                                "(%d)",sql_row[3],gr->gr_name,
                                gr->gr_gid);
end_of_loop:
                last = mysql_row_tell(result);
        }
        if ((void *) addrptr >= (void *) end_of_buf)
                goto out_nomem;
        /* end */
        *addrptr = NULL;

        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to_struct: "
                                "success, exiting");

        *errnop = 0;
        return NSS_STATUS_SUCCESS;

out_nomem:
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"_nss_mysql_group_result_to_struct: "
                                "libc buffer is too small");
        /* we need more space in the buffer */
        *errnop = ERANGE;
        mysql_row_seek(result,initial_offset);
        return NSS_STATUS_TRYAGAIN;

}

/* Initgroups
 * Returns all groups which an user is in
 * Arguments :
 * user : user name
 * group: this gid must be excluded from groups
 * start: first element in groups
 * size: size of groups
 * groups: array of gid the user is in
 * limit: size limit of groups
 * errnop: pointer to the application errno
 */

#undef lockm
#undef unlockm
#define lockm() \
pthread_mutex_lock(&initgroups_mutex);

#define unlockm() \
pthread_mutex_unlock(&initgroups_mutex);

static pthread_mutex_t initgroups_mutex = NSS_MYSQL_PTHREAD_MUTEX_INITIALIZER;
static struct mysql_auth initgroups_auth = {NULL,0,&initgroups_mutex};

NSS_STATUS _nss_mysql_initgroups (const char *user, gid_t group, 
                long int *start, long int *size, gid_t * groups, 
                long int limit,int *errnop) {
        char * sql = NULL;
        MYSQL_RES * result = NULL;
        MYSQL_ROW sql_row;
        unsigned int i;
        struct passwdoptions options;
        struct groupoptions goptions;
        struct parse_result pr = {&options,&goptions,NULL};
        char * secure_user;

        memset(&options,0,sizeof(struct passwdoptions));
        memset(&goptions,0,sizeof(struct groupoptions));

        if (DEBUG) _nss_mysql_log(LOG_ERR,"initgroups called with user=%s,"
                        "group=%d,*start=%d,*size=%d,limit=%d",user,*start,
                        *size,limit);
        
        if (!user) {
                *errnop = ENOENT;
                return NSS_STATUS_NOTFOUND;
        }

        /* we parse conf files */
        if (! _nss_mysql_read_conf_file("groups",&pr)) {
                _nss_mysql_log(LOG_ERR,"_nss_mysql_initgroups: conf file "
                                "parsing failed");
                _nss_mysql_free_users(&options);
                _nss_mysql_free_groups(&goptions);
                *errnop = ENOENT;
                return NSS_STATUS_UNAVAIL;
        }

        _nss_mysql_set_fork_handler(&forkhandler_isset,&forkhandler_mutex,
                        prepare_forkhandler,
                        parent_forkhandler,
                        child_forkhandler,
                        atexit_handler);
        
        if (! _nss_mysql_check_connection(&initgroups_auth,&options.con)) {
                _nss_mysql_free_users(&options);
                _nss_mysql_free_groups(&goptions);
                *errnop = ENOENT;
                return NSS_STATUS_UNAVAIL;
        }

        /* we escape the user string */
        secure_user = malloc(strlen(user) * 2 + 1);
        if (secure_user == NULL) {
                unlockm();
                _nss_mysql_log(LOG_ERR,"initgroups: not enough memory to escape"
                                " the user string");
                *errnop = EAGAIN;
                my_thread_end();
                return NSS_STATUS_TRYAGAIN;
        }

#ifndef HAVE_MYSQL_REAL_ESCAPE_STRING
        mysql_escape_string(secure_user,user,strlen(user));
#else
        mysql_real_escape_string(initgroups_auth.mysql,secure_user,user,
                        strlen(user));
#endif

        /* we look for the group which the user is a member of
         * no aliasing for groupinfotable and optionstable because of the where
         * clauses
         */
        sql = _nss_mysql_sqlprintf("select %s from %s, %s, %s where %s=%s"
                        " and %s=%s and %s='%s' and %s <> %d",
                        goptions.gidcolumn,
                        /* from */
                        options.table,
                        goptions.memberstable,
                        goptions.groupinfotable,
                        /* where */
                        options.useridcolumn,
                        goptions.museridcolumn,
                        /* and */
                        goptions.mgroupidcolumn,
                        goptions.groupidcolumn,
                        /* and */
                        options.usercolumn,
                        secure_user,
                        /* and */
                        goptions.gidcolumn,
                        group);

        free(secure_user);

        if (! sql) {
                unlockm();
                goto out_nomem;
        }

        /* conf files where clauses */
        if (options.where[0]) {
                char * tmp;
                /* we try to alloc the space we need */
                tmp = realloc(sql,strlen(sql) + sizeof(" and ") + strlen(options.where));
                if (tmp == NULL) {
                        /* we could not, so we free the old one and leave */
                        _nss_mysql_log(LOG_ERR,"initgroups: can't allocate "
                                        "memory for where clauses");
                        free(sql);
                }
                sql = tmp;
                if (! sql) {
                        unlockm();
                        goto out_nomem;
                }

                strcat(sql," and ");
                strcat(sql,options.where);
        }

        if (goptions.where[0]) {
                /* same as above */
                char * tmp;
                tmp = realloc(sql,strlen(sql) + sizeof(" and ") + 
                                strlen(goptions.where));
                if (tmp == NULL) {
                        _nss_mysql_log(LOG_ERR,"initgroups: can't allocate "
                                        "memory for where clauses");
                        free(sql);
                }
                sql = tmp;
                if (! sql) {
                        unlockm();
                        goto out_nomem;
                }
                strcat(sql," and ");
                strcat(sql,goptions.where);
        }
        if (DEBUG) _nss_mysql_log(LOG_ERR,"initgroups: SQL statement: %s",sql);

        if (mysql_query(initgroups_auth.mysql,sql)) {
                _nss_mysql_log(LOG_ERR,"initgroups: mysql_query has failed: %s",
                                mysql_error(initgroups_auth.mysql));
                unlockm();
                goto out_unavail;
        }
        
        result = mysql_store_result(initgroups_auth.mysql);
        if (! result) {
                _nss_mysql_log(LOG_ERR,"initgroups: mysql_store_result has "
                                "failed: %s",
                                mysql_error(initgroups_auth.mysql));
                unlockm();
                goto out_unavail;
        }
        unlockm();
        s_free(sql); 
        i = mysql_num_rows(result);
        if (i == 0)	{
                if (DEBUG) _nss_mysql_log(LOG_ERR,"initgroups: no results "
                                "found. Exiting");
                *errnop = 0;
                _nss_mysql_free_groups(&goptions);
                _nss_mysql_free_users(&options);
                mysql_free_result(result);
                my_thread_end();
                return NSS_STATUS_SUCCESS;
        }

        while ((sql_row = mysql_fetch_row(result)))	{
                if (!sql_row) {
                        _nss_mysql_log(LOG_ERR,"initgroups: mysql_fetch_row "
                                        "failed: %s.",
                                        mysql_error(initgroups_auth.mysql));
                        goto out_unavail;
                }
                
                /* do we have enough memory space ? */
                if (*start == *size && limit <= 0) {
                        /* we have to write below the array limit, we 
                         * reallocate twice more space 
                         */
                        gid_t * tmp;
                        tmp = realloc (groups, 2 * *size * sizeof (*groups));
                        if (tmp == NULL) {
                                _nss_mysql_log(LOG_ERR,"initgroups: We could "
                                                "not have allocated the needed "
                                                "space. Stopping");
                                break;
                        }
                        groups = tmp;
                        *size *= 2;
                }
                
                groups[*start] = _nss_mysql_strtol(sql_row[0],-1,&i);
                if (i) {
                        _nss_mysql_log(LOG_ERR,"initgroups: cannot convert "
                                        "group_id (%s) for %s. "
                                        "Fix your database.",sql_row[0],user);
                }
                if (DEBUG) _nss_mysql_log(LOG_ERR,"initgroups: %s is a member "
                                "of %d",user,groups[*start]);
                
                ++*start; /* next element */
                if (*start == limit) {
                        /* we cannot go on */
                        if (DEBUG) _nss_mysql_log(LOG_ERR,"initgroups: reached "
                                        "limit at %d",*start);
                        break;
                }
        }
        if (DEBUG) _nss_mysql_log(LOG_ERR,"initgroups exited");
        mysql_free_result(result);
        _nss_mysql_free_groups(&goptions);
        _nss_mysql_free_users(&options);
        *errnop = 0;
        my_thread_end();
        return NSS_STATUS_SUCCESS;

out_nomem:
        s_free(sql);
        if (result)
                mysql_free_result(result);
        _nss_mysql_free_groups(&goptions);
        _nss_mysql_free_users(&options);
        *errnop = EAGAIN;
        my_thread_end();
        return NSS_STATUS_TRYAGAIN;

out_unavail:
        s_free(sql);
        if (result)
                mysql_free_result(result);
        _nss_mysql_free_groups(&goptions);
        _nss_mysql_free_users(&options);
        *errnop = ENOENT;
        my_thread_end();
        return NSS_STATUS_UNAVAIL;
}

/* setgrent
 * Initializes data for ...grent functions
 * NOTE this function does _NOT_ use errno
 */
NSS_STATUS _nss_mysql_setgrent (void) {
        return _nss_mysql_setent(GROUP_ENT_TYPE,get_sql_query);
}

/* endgrent
 * Kills all data for ...grent functions
 * NOTE this function does _NOT_ use errno
 */
NSS_STATUS _nss_mysql_endgrent (void) {
        return _nss_mysql_endent(GROUP_ENT_TYPE);
}

/* getgrent
 * Gets info for every group
 * Arguments:
 * gr: array to fill
 * buffer: buffer, unused
 * buflen: size of buffer
 * errnop: ptr to the application errno
 */

NSS_STATUS _nss_mysql_getgrent_r (struct group *gr,
                char * buffer, size_t buflen,int * errnop) {
        return _nss_mysql_getent_r(GROUP_ENT_TYPE,gr,buffer,buflen,errnop);
}

/* getgrnam
 * Looks for a group by its name
 * Arguments:
 * name: name of the group
 * gr: pointer to the gr struct we'll filll
 * buffer: buffer, unused
 * buflen: size of buffer
 * errnop: ptr to the application errno
 */

static pthread_mutex_t nam_mutex = NSS_MYSQL_PTHREAD_MUTEX_INITIALIZER;
static struct mysql_auth nam_auth = {NULL,0,&nam_mutex};

NSS_STATUS _nss_mysql_getgrnam_r (const char * name, struct group *gr,
                char * buffer, size_t buflen,int *errnop) {
        struct query q = {name,0,0,0};
        return handle_query(&q,&nam_auth,gr,buffer,buflen,errnop);
}       


/* getgrgid
 * Looks for a group by its GID
 * Arguments:
 * gid: gid of the group
 * gr: pointer to the gr struct we'll filll
 * buffer: buffer, unused
 * buflen: size of buffer
 * errnop: ptr to the application errno
 */


static pthread_mutex_t gid_mutex = NSS_MYSQL_PTHREAD_MUTEX_INITIALIZER;
static struct mysql_auth gid_auth = {NULL,0,&gid_mutex};

NSS_STATUS _nss_mysql_getgrgid_r (const gid_t gid, struct group *gr,
                char * buffer, size_t buflen,int *errnop) {
        struct query q = {0,0,gid,0};
        return handle_query(&q,&gid_auth,gr,buffer,buflen,errnop);
}

void prepare_forkhandler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"group.c: calling prepare_forkhandler");
        pthread_mutex_lock(nam_auth.mutex);
        pthread_mutex_lock(gid_auth.mutex);
        pthread_mutex_lock(initgroups_auth.mutex);
        pthread_mutex_lock(&forkhandler_mutex);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"group.c: end of prepare_forkhandler");
}

void parent_forkhandler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"group.c: calling parent_forkhandler");
        pthread_mutex_unlock(nam_auth.mutex);
        pthread_mutex_unlock(gid_auth.mutex);
        pthread_mutex_unlock(initgroups_auth.mutex);
        pthread_mutex_unlock(&forkhandler_mutex);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"group.c: end of parent_forkhandler");
}

void child_forkhandler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"group.c: calling child_forkhandler");
        gid_auth.pid = 0;
        nam_auth.pid = 0;
        initgroups_auth.pid = 0;
        pthread_mutex_unlock(nam_auth.mutex);
        pthread_mutex_unlock(gid_auth.mutex);
        pthread_mutex_unlock(initgroups_auth.mutex);
        pthread_mutex_unlock(&forkhandler_mutex);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"group.c: end of child_forkhandler");
}


void atexit_handler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"group.c: calling atexit_handler");
        pthread_mutex_lock(nam_auth.mutex);
        _nss_mysql_db_close(&nam_auth.mysql);
        pthread_mutex_lock(gid_auth.mutex);
        _nss_mysql_db_close(&gid_auth.mysql);
        pthread_mutex_lock(initgroups_auth.mutex);
        _nss_mysql_db_close(&initgroups_auth.mysql);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"group.c: end of atexit_handler");
}

#ifdef TEST

int main(void) {
        struct group gr;
        NSS_STATUS s;
        _nss_mysql_setgrent();
        while ((s=_nss_mysql_getgrent_r(&gr,NULL,0,&errno))
                        !=NSS_STATUS_NOTFOUND);
        _nss_mysql_endgrent();
        return 0;
}

#endif

#endif /* if USE_GROUP */
