/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
//
// htp-files.c
//
// htp specific file access
//
// Copyright (c) 1995-96 Jim Nelson.  Permission to distribute
// granted by the author.  No warranties are made on the fitness of this
// source code.
//
*/

#include "defs.h"
#include "option.h"
#include "streams.h"
#include "bool-proc.h"

/*
// returns the full, qualified pathname of the default htp include file
//
// Returns FALSE if unable to find the file.
*/
BOOL HtpDefaultFilename(char *filename, uint size)
{
    char *defFile;

    /* get the name of the default file from the HTPDEF environement */
    /* variable */
    if((defFile = getenv("HTPDEF")) == NULL)
    {
        return FALSE;
    }

    /* verify that the file exists */
    if(FileExists(defFile) == FALSE)
    {
        return FALSE;
    }

    /* copy the filename into the buffer and skeedaddle */
    StringCopy(filename, defFile, size);

    return TRUE;
}

/*
// compare files modified time/date stamp, as a dependency check ... returns
// TRUE if the dependency does not require an update, FALSE otherwise (which
// could either be a timestamp discrepency, or simply that the resulting file
// does not exist) ... if dependency checking is turned off, this function
// will always return FALSE.
//
// Returns ERROR if dependency file does not exist.
*/
BOOL IsTargetUpdated(const char *dependency, const char *target)
{
    struct stat dependStat;
    struct stat targetStat;
    char *dependName;
    char *targetName;

    assert(dependency != NULL);
    assert(target != NULL);

    /* always update targets? */
    if(DEPEND == FALSE)
    {
        return FALSE;
    }

    /* convert the dependency and target filenames for this filesystem */
    if((dependName = ConvertDirDelimiter(dependency)) == NULL)
    {
        return ERROR;
    }

    if((targetName = ConvertDirDelimiter(target)) == NULL)
    {
        FreeMemory(dependName);
        return ERROR;
    }

    /* get information on the dependency file */
    if(stat(dependName, &dependStat) != 0)
    {
        /* dependency file needs to exist */
        FreeMemory(dependName);
        FreeMemory(targetName);

        return ERROR;
    }

    /* get information on the target file */
    if(stat(targetName, &targetStat) != 0)
    {
        /* target file does not exist, dependency needs to be updated */
        FreeMemory(dependName);
        FreeMemory(targetName);

        return FALSE;
    }

    FreeMemory(dependName);
    FreeMemory(targetName);

    /* compare modification times to determine if up-to-date */
    return (dependStat.st_mtime <= targetStat.st_mtime) ? TRUE : FALSE;
}

/*
// searches for the specified file in the search path ... this function is
// very stupid, it simply gets the first directory in the search string,
// appends the file directly to the end, and tests for existance.  Repeat.
*/
BOOL SearchForFile(const char *filename, char *fullPathname, uint size)
{
    char *searchPathCopy;
    char *ptr;
    char *convertedName;
    FIND_TOKEN findToken;

    /* quick check for search path even being defined */
    if(searchPath[0] == NUL)
    {
        return FALSE;
    }

    /* need to make a copy of the search path for String...Token() to butcher up */
    if((searchPathCopy = DuplicateString(searchPath)) == NULL)
    {
        printf("%s: unable to allocate temporary buffer for include path (out of memory?)\n",
            PROGRAM_NAME);
        return FALSE;
    }

    /* look for ';' delimiter */
    ptr = StringFirstToken(&findToken, searchPathCopy, ";");
    while(ptr != NULL)
    {
        StringCopy(fullPathname, ptr, size);

        /* if the last character is not a directory delimiter, add it */
        if(strchr(ALL_FILESYSTEM_DELIMITERS, fullPathname[strlen(fullPathname) - 1]) == NULL)
        {
            strncat(fullPathname, DIR_DELIMITER_STRING, size);
        }

        /* append the file name */
        strncat(fullPathname, filename, size);

        /* need to do a complete conversion of delimiters in the filename, but */
        /* ConvertDirDelimiter() returns a AllocMemory()'d copy of the string ... */
        convertedName = ConvertDirDelimiter(fullPathname);

        /* check for existance */
        if(FileExists(convertedName) == TRUE)
        {
            /* clean up and get outta here */
            StringCopy(fullPathname, convertedName, size);

            FreeMemory(searchPathCopy);
            FreeMemory(convertedName);

            return TRUE;
        }

        FreeMemory(convertedName);
        convertedName = NULL;

        ptr = StringNextToken(&findToken);
    }

    /* clean up */
    FreeMemory(searchPathCopy);

    return FALSE;
}

/*
// TRUE = plaintext is filled with new plain text markup, FALSE if end of file,
// ERROR if a problem
// !! Don't like using ERROR in any BOOL return values
*/
BOOL ReadHtmlFile(STREAM *infile, STREAM *outfile, char **plaintext,
    uint *markupType)
{
    char ch;
    uint ctr;
    char *buffer;
    char *newbuffer;
    uint size;
    BOOL inQuotes;
    uint startLine;
    uint numread;
    bitmap open_markup;
    bitmap tag_specials;

    memset(open_markup, 0, sizeof(open_markup));
    BITMAP_SET(open_markup, HTML_OPEN_MARKUP);
    BITMAP_SET(open_markup, htpOpenMarkup);

    memset(tag_specials, 0, sizeof(tag_specials));
    BITMAP_SET(tag_specials, HTML_CLOSE_MARKUP);
    BITMAP_SET(tag_specials, htpCloseMarkup);
    BITMAP_SET(tag_specials, '"');

    assert(infile != NULL);
    assert(plaintext != NULL);

    /* if outfile is NULL, then the input stream is just being walked and */
    /* not parsed for output ... i.e., don't assert outfile != NULL */

    /* allocate some space for markup plaintext ... this will dynamically */
    /* expand if necessary, and has to be freed by the caller */
    if((buffer = AllocMemory(MIN_PLAINTEXT_SIZE)) == NULL)
    {
        HtpMsg(MSG_ERROR, NULL, "unable to allocate memory to read HTML file");
        return ERROR;
    }

    /* track the buffer size */
    size = MIN_PLAINTEXT_SIZE;

    for (;;)
    {
        numread = GetStreamBlock(infile, buffer, size, open_markup);
        if (numread == 0)
        {
            /*EOF*/
            /* no markup found, end of file */
            FreeMemory(buffer);
            
            return FALSE;
        }

        if (IS_OPEN_MARKUP(buffer[numread-1]))
        {
            /* get the type of markup for caller */
            *markupType = MarkupType(buffer[numread-1]);
            if (numread > 1 && outfile != NULL) {
                /* there is normal text, just copy it to the output file */
                buffer[numread-1] = NUL;
                StreamPrintF(outfile, "%s", buffer);
            }
            break;
        }

        /* no open markup found in 2KB.  Print the full block and
         * continue.  
         */
        if (outfile != NULL)
            StreamPrintF(outfile, "%s", buffer);
    }


    /* copy the markup into the buffer */
    ctr = 0;
    inQuotes = FALSE;
    startLine = infile->lineNumber;
    for(;;)
    {
        numread = GetStreamBlock(infile, buffer + ctr, size - ctr, 
                                  tag_specials);
        if (numread == 0)
        {
            /* EOF ... this is not acceptable before the markup is */
            /* terminated */
            FreeMemory(buffer);
            HtpMsg(MSG_ERROR, infile,
                   "EOF encountered inside markup tag (started on line %u)",
                   startLine);
            
            return ERROR;
        }
        
        ctr += numread;
        ch = buffer[ctr - 1];
        if ((IS_CLOSE_MARKUP(ch))
            && (inQuotes == FALSE)
            && (MarkupType(ch) == *markupType))
        {
            /* end of markup, terminate string and exit */
            buffer[ctr - 1] = NUL;
            break;
        }
        
        /* track quotation marks ... can only close markup when */
        /* all quotes have been closed */
        if(ch == '\"')
        {
            inQuotes = (inQuotes == TRUE) ? FALSE : TRUE;
        }
        
        /* check for overflow ... resize buffer if necessary */
        if(ctr >= size - 1)
        {
            newbuffer = ResizeMemory(buffer, size + PLAINTEXT_GROW_SIZE);
            if(newbuffer == NULL)
            {
                /* unable to enlarge buffer area */
                HtpMsg(MSG_ERROR, NULL, "unable to reallocate memory for reading HTML file");
                FreeMemory(buffer);
                return ERROR;
            }
                
            buffer = newbuffer;
            size += PLAINTEXT_GROW_SIZE;
        }
    }
        
    /* give the buffer pointer to the caller */
    *plaintext = buffer;
    
    return TRUE;
}   


BOOL FullyCheckDependencies(const char *in, const char *out)
{
    BOOL result;
    STREAM infile;
    char *plaintext;
    char title[128];
    HTML_MARKUP markup;
    const char *includeFile;
    const char *imageFile;
    BOOL readResult;
    BOOL checkResult;
    uint markupType;

    assert(in != NULL);
    assert(out != NULL);

    if(DEPEND == FALSE)
    {
        /* outta here */
        return FALSE;
    }

    /* check if target file is completely up to date compared to input file */
    result = IsTargetUpdated(in, out);
    if(result == ERROR)
    {
        printf("%s: unable to get file information for file \"%s\"\n",
            PROGRAM_NAME, in);
        return ERROR;
    }
    else if(result == FALSE)
    {
        /* target is not updated */
        return FALSE;
    }

    /* because target is up to date, need to search dependency file for */
    /* FILE INCLUDE tags and check those files likewise */

    /* open file */
    sprintf(title, "Dependency check for %s", in);
    if (CreateFileReader(&infile, in) == FALSE)
    {
        printf("%s: unable to open file \"%s\" for reading while checking dependencies\n",
            PROGRAM_NAME, in);
        return ERROR;
    }
    infile.name = title;

    /* assume everything is hunky-dory unless otherwise discovered */
    checkResult = TRUE;

    /* get the next markup tag from the input file */
    while((readResult = ReadHtmlFile(&infile, NULL, &plaintext, &markupType)) != FALSE)
    {
        if(readResult == ERROR)
        {
            /* error occurred processing the HTML file */
            checkResult = ERROR;
            break;
        }

        /* check markup type ... only interested in htp markups currently */
        if((markupType & MARKUP_TYPE_HTP) == 0)
        {
            continue;
        }

        /* received a markup ... check if its an INCLUDE markup */
        PlaintextToMarkup(plaintext, &markup);

        /* do not need plaintext any further */
        FreeMemory(plaintext);
        plaintext = NULL;

        /* if FILE INCLUDE markup, get the filename specified */
        includeFile = NULL;
        imageFile = NULL;
        if(IsMarkupTag(&markup, "FILE"))
        {
            if(IsAttributeInMarkup(&markup, "INCLUDE"))
            {
                includeFile = MarkupAttributeValue(&markup, "INCLUDE");
            }
            else if(IsAttributeInMarkup(&markup, "TEMPLATE"))
            {
                includeFile = MarkupAttributeValue(&markup, "TEMPLATE");
            }
        }
        else if(IsMarkupTag(&markup, "IMG"))
        {
            if(IsAttributeInMarkup(&markup, "SRC"))
            {
                imageFile = MarkupAttributeValue(&markup, "SRC");
            }
        }
        else if(IsMarkupTag(&markup, "OPT"))
        {
            if(IsAttributeInMarkup(&markup, "NODEPEND"))
            {
                /* !! dependency checking disabled in source file ... since this */
                /* can swing back and forth throughout the files, and its just */
                /* a pain to track what is technically the last one set, */
                /* if one is found, dependency checking is disabled and the */
                /* targets are not considered updated */
                /* this could be fixed with some work */
                checkResult = FALSE;
                DestroyMarkupStruct(&markup);
                break;
            }
        }

        /* by default assume everything is up to date unless more information */
        /* is available through other files */
        result = TRUE;

        /* check include or image file timestamps */
        if(includeFile != NULL)
        {
            /* !! the accursed recursion strikes again */
            /* check the dependencies based on this new file */
            result = FullyCheckDependencies(includeFile, out);
        }
        else if(imageFile != NULL)
        {
            /* check the image files timestamp as part of dependency checking */
            if(FileExists(imageFile))
            {
                result = IsTargetUpdated(imageFile, out);
            }
        }

        /* unneeded now */
        DestroyMarkupStruct(&markup);

        if(result != TRUE)
        {
            /* if FALSE, not up to date, no need to go further */
            /* if ERROR, need to stop and report to caller */
            checkResult = result;
            break;
        }

        /* otherwise, TRUE indicates that everything is okay, */
        /* so keep searching */
    }

    /* EOF encountered in the HTML input file ... target is updated */
    CloseStream(&infile);

    return checkResult;
}
