/* 
 * Prospect: a developer's system profiler.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Author: Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: bfile.c,v 1.12 2004/01/09 20:29:27 type2 Exp $ */

/*
 *******************************************************************************
 *
 *                            PROSPECT PROJECT
 *                   Binary trace output/input routines
 *
 *******************************************************************************
 */
#ifndef __LINT__
static const char gRCSid[] = "@(#) $Id: bfile.c,v 1.12 2004/01/09 20:29:27 type2 Exp $";
#endif

/*
 * System header files
 */
#include <unistd.h>
#include <stdio.h>
#include <string.h>

/*
 * Prospect header files
 */
#include "bfile.h"
#include "../dtree/dtree.h"
#include "prospect.h"
#include "linux_module.h"
#include "rec_proc.h"

/*
 * Global data bits
 */
bfile_version_t bfile_version=43;   /* this gets incremented when abi changes */

char *bbuf_name[] = {
    "Invalid",
    "End_of_file",
    "Samples",
    "Notes",
    "File_path",
    "gOp_struct",
    "gConf_struct",
    "Proc_struct",
    "Kernel_symbols",
    "Child_struct",
    "Max_name_limit"
};

/* 
 * void bfile_init_out(void)
 *
 * Initialize output trace file.  At this point, the file has been
 * opened and the only thing we write out is the initial 32-bit 
 * version number.
 */
void 
bfile_init_out(void)
{
    ssize_t result;

    mINFORM("In bfile_init_out()");
    
    if (!mTRACEOUT) return;

    result = write(gConf.fh_trace_out, &bfile_version, sizeof(bfile_version_t));
    mINFORM(" wrote %zu bytes (initial version number of %d)",
            result, bfile_version);
    if (result != sizeof(bfile_version_t)) {
        perr("Can't write Bfile: line %d, tried writing %zu bytes, wrote %zu", 
             __LINE__, sizeof(bfile_version_t), result);
        prospect_exit(1);
    }

} /* bfile_init_out() */

/*
 * int bfile_write(tBBUFTYPE, char*, ssize_t)
 *
 * Write out specifed bfile buffer type to file.  If writes go bad,
 * we exit here.
 */
void 
bfile_write(tBBUFTYPE type, void *buf, ssize_t size)
{
    char str[128];
    char mini[64];
    ssize_t result;
    bfile_bufhead_t bhead;

    if (!mTRACEOUT) return;

    str[0]='\0';
    strcat(str, "In bfile_write(");
    if (type>=BBUF_MAX_NAME || type<=BBUF_INVALID) 
        strcat(str, "Error_In_Buffer_Name");
    else 
        strcat(str, bbuf_name[type-BBUF_INVALID]);

    sprintf(mini, ", buf=%p, size=%zu)", buf, size);
    strcat(str, mini);
    mINFORM(str);

    /* create header for this buffer */
    bhead.size = sizeof(bfile_bufhead_t) + size;
    bhead.type = type;
    bhead.dcount = size;

    /* write out header... */
    result = write(gConf.fh_trace_out, &bhead, sizeof(bfile_bufhead_t));
    mINFORM(" wrote %zu bytes (bbuffer header)", result);
    if (result != sizeof(bfile_bufhead_t)) {
        perr("Can't write Bfile: line %d, tried writing %zu bytes, wrote %zu", 
             __LINE__, size, result);
        prospect_exit(1);
    }

    /* ... and write out rest of buffer */ 
    result = write(gConf.fh_trace_out, buf, size);
    mINFORM(" wrote %zu bytes (data buffer)", result);
    if (result != size) {
        perr("Can't write Bfile: line%d, tried writing %zu bytes, wrote %zu", 
             __LINE__, size, result);
        prospect_exit(1);
    }

} /* bfile_write() */

/*
 * void bfile_write_configs(void)
 *
 * Write both the gOp and gConf configuration structs to 
 * output trace file.
 */
void 
bfile_write_configs(void)
{
    mINFORM("In bfile_write_configs()");
    
    if (!mTRACEOUT) return;

    /* write out the gOp struct */
    bfile_write(BBUF_GOP_STRUCT, &gOp, sizeof(op_ctl_t));

    /* write out the gConf struct */
    bfile_write(BBUF_GCONF_STRUCT, &gConf, sizeof(prospect_config_t));

} /* bfile_write_configs() */

/*
 * void bfile_write_child(void)
 *
 * Write the gOp.my_child structs to the output trace file.
 * This happens as the last write, child has exited.
 */
void 
bfile_write_child(void)
{
    mINFORM("In bfile_write_child()");
    
    if (!mTRACEOUT) return;

    /* write out the gOp.my_child struct */
    bfile_write(BBUF_CHILD_STRUCT, &gConf.my_child, sizeof(mychild_t));

} /* bfile_write_child() */

/*
 * void bfile_write_rpath(region_t *r)
 *
 * Saves path of object defined by region r.
 */
void
bfile_write_rpath(region_t *r)
{
    bfile_filehead_t *fhead;

    mINFORM("In bfile_write_rpath(%p)", r);
    if (!r->rd_path) {
        mINFORM(" r->rd_path is NULL, returning");
        return;
    }
    mINFORM(" rd_pid=%u rd_start=%p strlen_path=%u rd_path='%s'", 
            r->rd_pid, r->rd_start, strlen(r->rd_path), r->rd_path);

    /* create buffer ... */
    fhead = (bfile_filehead_t *) 
        MALLOC(sizeof(bfile_filehead_t) + strlen(r->rd_path) +1);
    if (fhead == NULL) {
        ferr("malloc fails, line %d, file %s, bufsize %d", __LINE__, __FILE__,
             sizeof(bfile_filehead_t) + strlen(r->rd_path) +1);
        prospect_exit(1);
    }

    fhead->dcount = strlen(r->rd_path) + 1;
    fhead->size = sizeof(bfile_filehead_t) + fhead->dcount;
    fhead->type = BBUF_FILE_PATH;
    fhead->pid = r->rd_pid;
    fhead->start = r->rd_start;
    strcpy(fhead->data, r->rd_path);
    
    /* ... and save it to the bfile */
    bfile_write(BBUF_FILE_PATH, (char*) fhead, fhead->size);

    FREE(fhead);
} /* bfile_write_rpath() */

/*
 * void bfile_init_in(void)
 *
 * Initialize from input trace file.  The initial part of the
 * trace file contains data in the following sequence:
 *
 *      version number, 32 bit first in file
 *      kernel symbols structure, variable, if System.map read successfully
 *      the gOp op_ctl_t struct from the run
 *      the gConf prospect_config_t struct from the run
 *      the harvest of data from /proc (from glean_proc(), from the run)
 */
void 
bfile_init_in(void)
{
    bfile_version_t version;
    prospect_config_t *bConf;
    tBBUFTYPE type;
    ssize_t result, size;
    char *buf=NULL;

    mINFORM("In bfile_init_in()");
    if (!mTRACEIN) return;

    /* read in version string - must be first */
    result = read(gConf.fh_trace_in, &version, sizeof(bfile_version_t));
    if (result != sizeof(bfile_version_t))
    {
        perr("Can't read bfile: line %d, tried reading %zu bytes, wrote %zu", 
             __LINE__, sizeof(bfile_version_t), result);
        prospect_exit(1);
    }
    /* ... and check it */
    mINFORM(" read version=%d, my version=%d", version, bfile_version);
    if (version != bfile_version) {
        ferr(
"\n"
"Error: Version of bfile (set at %d) is incompatible with the currently\n"
"       running prospect (bfile level %d).  Please reconstruct the bfile\n"
"       by re-running the experiment with this version of prospect.\n",
                version, bfile_version);
        prospect_exit(1);
    }

    /* read in gOp struct */
    buf = bfile_read(&type, &size);
    if (type == BBUF_INVALID || buf == NULL) {
        ferr("could not read config1 buffer from bfile, read '%s'\n",
                bbuf_name[type - BBUF_INVALID]);
        prospect_exit(1);
    }
    if (type != BBUF_GOP_STRUCT) {
        ferr("error reading bfile, expecting buffer type '%s' but read '%s'\n",
             bbuf_name[BBUF_GOP_STRUCT-BBUF_INVALID],
             bbuf_name[type-BBUF_INVALID]);
        prospect_exit(1);
    }
    if ((unsigned int) size != sizeof(op_ctl_t)) {
        ferr("incomplete configuration structure 1 read from bfile\n");
        prospect_exit(1);
    }
    /* Ok, it's an gOp struct as expected, fill in data from struct but
     * be careful: some of those pointers were allocated memory...
     * First, just byte copy the buffer, akin to sticking an icepick in
     * one's head...
     */
    memcpy(&gOp, buf, sizeof(op_ctl_t));
    FREE(buf);  /* don't leak */
    /* now, allocate room for oprofile buffers and repoint */
    gOp.curbufhead = (struct op_buffer_head *) MALLOC(gOp.bufsize);
    if (gOp.curbufhead == NULL) {
        ferr("fatal malloc failure line %d, file %d\n", __LINE__, __FILE__);
        prospect_exit(1);
    }
    gOp.curnbuf = (struct op_note *) MALLOC(gOp.notesize);
    if (gOp.curnbuf == NULL) {
        ferr("fatal malloc failure line %d, file %d\n", __LINE__, __FILE__);
        prospect_exit(1);
    }
    mINFORM(" done reading gOp struct from bfile");

    /* read in gConf struct */
    buf = bfile_read(&type, &size);
    if (type == BBUF_INVALID || buf == NULL) {
        ferr("could not read config2 buffer from bfile, read '%s'\n",
                bbuf_name[type - BBUF_INVALID]);
        prospect_exit(1);
    }
    if (type != BBUF_GCONF_STRUCT) {
        ferr("error reading bfile, expecting buffer type '%s' but read '%s'\n",
             bbuf_name[BBUF_GCONF_STRUCT-BBUF_INVALID],
             bbuf_name[type-BBUF_INVALID]);
        prospect_exit(1);
    }
    if ((unsigned int) size != sizeof(prospect_config_t)) {
        ferr("incomplete configuration structure 2 read from bfile\n");
        prospect_exit(1);
    }
    /* Right, gConf struct as expected.  Only a few members are needed from
     * this.  Remember, the current gConf struct is what we want to do now,
     * the trace gConf struct is what was done then.
     */
    bConf = (prospect_config_t *) buf;
    strncpy(gConf.command_line, bConf->command_line, MAXCOMMANDLEN);
    gConf.my_pid = bConf->my_pid;
    gConf.my_sid = bConf->my_sid;
    gConf.my_pgrp = bConf->my_pgrp;
    memcpy(&gConf.my_utsname, &bConf->my_utsname, sizeof(struct utsname));
    strncpy(gConf.run_time, bConf->run_time, 32);
    gConf.cpumhz = bConf->cpumhz;
    gConf.cputype = bConf->cputype;
    gConf.numcpus = bConf->numcpus;
    /* allocate space for kernel profiles now we know numcpus */
    alloc_k_prof(); 
    /* that's it, don't leak */
    FREE(buf);
    mINFORM(" done reading gConf struct, run was at: %s", gConf.run_time);
} /* bfile_init_in() */

/*
 * static int bfile_read_fromfile(void *buf, size_t count)
 *
 * Reads the oprofile buffers from bfile.  A -1 is returned on eof and that will
 * break us out of the read/waitpid loop in the sequencer in rec_proc.c.  Main
 * sample and note buffers are recorded as /samples/notes/file/file/.../ groups.
 * If we read a file buffer, we read again until a real buffer.  All proc
 * buffers are recorded at the beginning of the file, so the first read for a
 * sample buffer will pick all those up.
 */
ssize_t 
bfile_read_fromfile(void *buf, size_t size) 
{ 
    char *dbuf=NULL;
    tBBUFTYPE type; ssize_t dsize;

    mINFORM("In op_read_from_file");
    do {
        dbuf = bfile_read(&type, &dsize);
        switch (type) {
            case BBUF_EOF:
                mINFORM(" read eof marker");
                if (dbuf) FREE(dbuf);
                return -1;

            case BBUF_PROC_STRUCT:
                /* process struct, process and read again */
                mINFORM(" read %zu bytes for a %s buffer of size=%zu",
                        dsize, bbuf_name[type-BBUF_INVALID], size);
                bfile_process_proc((bfile_prochead_t *)dbuf, dsize);
                /* if we're also tracing out, save this buffer */
                if (mTRACEOUT) bfile_write(BBUF_PROC_STRUCT, dbuf, dsize);
                if (dbuf) FREE(dbuf);
                break;

            case BBUF_FILE_PATH:
                /* process file object, then read again */
                mINFORM(" read file path buffer");
                bfile_process_rpath((bfile_filehead_t *)dbuf, dsize);
                /* if we're also tracing out, save this buffer */
                if (mTRACEOUT) bfile_write(BBUF_FILE_PATH, dbuf, dsize);
                if (dbuf) FREE(dbuf);
                break;

            case BBUF_CHILD_STRUCT:
                /* child buffer struct, process and read again */
                mINFORM(" read my_child struct buffer");
                memcpy(&gConf.my_child, dbuf, sizeof(mychild_t));
                /* the old process ptr is bogus in this context */
                gConf.my_child.p = getproc_by_pid(gConf.my_child.pid);
                /* if we're also tracing out, save this buffer */
                if (mTRACEOUT) bfile_write(BBUF_CHILD_STRUCT, dbuf, dsize);
                if (dbuf) FREE(dbuf);
                break;

            case BBUF_SAMPLES:
            case BBUF_NOTES:
                mINFORM(" read %zu bytes for a %s buffer of size=%zu",
                        dsize, bbuf_name[type-BBUF_INVALID], size);
                /* copy the buffer into pointer provided and return */
                if (dbuf)
                    memcpy(buf, dbuf, dsize);
                if (dbuf) FREE(dbuf);
                return dsize;

            case BBUF_INVALID:
                mINFORM(" invalid buffer read/ zero length buffer read");
                if (dbuf) FREE(dbuf);
                return 0;

            default:
                mINFORM(" read unexpected buffer of '%s'", 
                        bbuf_name[type-BBUF_INVALID]);
                ferr("read of unexpected '%s' from bfile spells error\n",
                     bbuf_name[type - BBUF_INVALID]);
                prospect_exit(1);
        }
    } while(1);
} /* bfile_read_from_file() */

/*
 * void bfile_process_rpath(dbuf, dsize);
 *
 * Process a file path record that's been read from a bfile and
 * put file path into process struct found by pid index, region
 * found by start index.
 */
void
bfile_process_rpath(bfile_filehead_t *buf, ssize_t size)
{
    process_t *p;
    region_t *r;

    mINFORM("In bfile_process_rpath(*buf=%p, size=%zu)", buf, size);
    if (buf==NULL) return;

    /* find process */
    p = getproc_by_pid(buf->pid);
    if (p==NULL) {
        mINFORM(" failed to find process struct for pid %d", buf->pid);
        if (gConf.bug_level)
            ferr("failed to find process %d to attach path '%s'\n",
                 buf->pid, buf->data);
            return;
    }

    /* find region in process */
    r = p->pr_vasHead->vh_fwd;
    while (r && r != (void*)p->pr_vasHead) {
        if (r->rd_start == buf->start) break;
        r = r->rd_fwd;
    }
    if (r == (void*)p->pr_vasHead) {
        mINFORM(" failed to find region struct %p in pid %d", 
                buf->start, buf->pid);
        if (gConf.bug_level)
            ferr("failed to find region %p in process %d for path '%s'\n",
                 buf->start, buf->pid, buf->data);
            return;
    }

    /* ok, dup the path */
    if (buf->data[0]) {
        mINFORM(" duplicated path '%s' to region %p in pid %u",
                buf->data, r->rd_start, buf->pid);
        r->rd_path = strdup(buf->data);
        mPTREC("file_path: pid=%u region=%p path=%s\n",
                buf->pid, r->rd_start, buf->data);
        gPstats[PRO_MAPS]++;
        /* and set the process name too */
#ifdef __ia64__
        if ((unsigned long)r->rd_start == 0x4000000000000000L) {
#else
        if (r == p->pr_vasHead->vh_fwd) {
#endif
            char s[32];

            p->pr_path = strdup(r->rd_path);
            mINFORM(" duped path to process path: %s", p->pr_path);
            extract_basename(s, p->pr_path);
            p->pr_myKname = strdup(s);
            mINFORM(" duped base name for kname: %s", p->pr_myKname);
        }
    }
} /* bfile_process_rpath() */

/*
 * void * bflie_read(tBBUFTYPE *bt, ssize_t *size)
 *
 * Basic read from bfile routine.  Returns an allocated buffer full of
 * data from file.  The passed parameters of buffer type *bt and *size
 * are set to the corresponding values.  The underlying assumption is 
 * that all data in a bfile has been wrapped in a bfile_bufhead_t struct.
 */
void *
bfile_read(tBBUFTYPE *bt, ssize_t *size)
{
    ssize_t count;
    bfile_bufhead_t bhead;
    void *buf=NULL;

    mINFORM("In bfile_read()");
    if (!mTRACEIN) {
        if (bt) *bt = BBUF_INVALID;
        return NULL;
    }

    /* read in buffer header */
    count = read(gConf.fh_trace_in, &bhead, sizeof(bfile_bufhead_t));
    if (count == -1) {
        perr("Read from bfile fails, file %s, line %s", __FILE__, __LINE__);
        prospect_exit(1);
    }
    if (count == 0) {
        if (bt) *bt = BBUF_EOF;
        return NULL;
    }
    if (count != sizeof(bfile_bufhead_t) || bhead.dcount == 0) {
        if (bt) *bt = BBUF_INVALID;
        return NULL;
    }
    mINFORM(" read buffer header of %d bytes, buftype is %s",
            count, bbuf_name[bhead.type - BBUF_INVALID]);

    /* read in the buffer data */
    buf = (char *) MALLOC(bhead.dcount);
    if (buf == NULL) {
        ferr("malloc fails, line %d, file %s, bufsize %d", __LINE__, __FILE__,
             bhead.dcount);
        prospect_exit(1);
    }
    mINFORM(" allocated %d bytes for buffer payload", bhead.dcount);

    count = read(gConf.fh_trace_in, buf, bhead.dcount);
    if (count == -1) {
        perr("Read from bfile fails, file %s, line %s", __FILE__, __LINE__);
        prospect_exit(1);
    }
    mINFORM(" read %d bytes of payload", count);
    if (count != bhead.dcount) {
        if (bt) *bt =BBUF_INVALID;
        return NULL;
    }

    if(size) *size = bhead.dcount;
    if(bt) *bt = bhead.type; 
    return buf;
} /* bfile_read() */

/*
 * void bfile_write_proc(pid_t pid)
 *
 * Write out process struct identified by pid arg to bfile.  Used
 * after reading in process info from /proc in glean_proc().  The
 * process struct contains pointers and dynamically allocated strings,
 * so we need to build a string table to save it to disk.
 */
void 
bfile_write_proc(pid_t pid)
{
    process_t *p;
    region_t *r;
    bfile_prochead_t *pbuf;
    unsigned int ii, numstrings, strtblsize, rsize, bufsz, numnulls;
    char *nullstr="null", *proc_p, *reg_p, *str_p;

    mINFORM("In bfile_write_proc(%ld)", pid);
    if ((p=getproc_by_pid(pid)) == NULL) {
        mINFORM(" pid %ld does not seem to exits in store");
        return;
    }

    /* calculate the size required for the string table */
    numstrings = strtblsize = numnulls = 0;
    numstrings++;
    if (p->pr_path) strtblsize += strlen(p->pr_path)+1;
    else { strtblsize += strlen(nullstr)+1; numnulls++; }
    numstrings++;
    if (p->pr_myKname) strtblsize += strlen(p->pr_myKname)+1;
    else { strtblsize += strlen(nullstr)+1; numnulls++; }
    r = p->pr_vasHead->vh_fwd;
    for (ii=0; ii<p->pr_vasHead->vh_entries; ii++) {
        numstrings++;
        if (r->rd_name) strtblsize += strlen(r->rd_name)+1;
        else { strtblsize += strlen(nullstr)+1; numnulls++; }
        numstrings++;
        if (r->rd_path) strtblsize += strlen(r->rd_path)+1;
        else { strtblsize += strlen(nullstr)+1; numnulls++; }
        r=r->rd_fwd;
    }
    mINFORM(" calculated string table size: %u for %u strings, %u nulls",
            strtblsize, numstrings, numnulls);
    mINFORM(" number of regions: %u", p->pr_vasHead->vh_entries);

    /* allocate space for buffer */
    rsize = p->pr_vasHead->vh_entries * sizeof(region_t);
    bufsz = sizeof(bfile_prochead_t) + sizeof(process_t) + rsize + strtblsize;
    pbuf=(bfile_prochead_t *) MALLOC(bufsz);
    if (pbuf==NULL) {
        ferr("malloc fails, line %d, file %s, bufsize %d", 
             __LINE__, __FILE__, bufsz);
        prospect_exit(1);
    }
    mINFORM(" allocated buffer of %u bytes", bufsz);

    /* assemble bfile_prochead_t buffer and save to disk */
    pbuf->size = bufsz;
    pbuf->type = BBUF_PROC_STRUCT;
    pbuf->pid = pid;
    pbuf->numstrings = numstrings;
    pbuf->numregions = p->pr_vasHead->vh_entries;
    pbuf->dcount = bufsz - sizeof(bfile_prochead_t); 
    /* set helper pointers */
    proc_p = (char*)pbuf   + sizeof(bfile_prochead_t);
    reg_p  = proc_p + sizeof(process_t);
    str_p  = reg_p  + rsize;

    /* copy the process_t struct to buffer */
    memcpy(proc_p, p, sizeof(process_t));

    /* copy all the region_t structs to buffer */
    r = p->pr_vasHead->vh_fwd;
    for (ii=0; ii<p->pr_vasHead->vh_entries; ii++) {
        memcpy(reg_p, r, sizeof(region_t));
        r=r->rd_fwd;
        reg_p += sizeof(region_t);
    }

    /* copy all strings to string table in buffer */
    if (p->pr_path) strcpy(str_p, p->pr_path);
    else strcpy(str_p, nullstr);
    str_p += strlen(str_p)+1;
    if (p->pr_myKname) strcpy(str_p, p->pr_myKname);
    else strcpy(str_p, nullstr);
    str_p += strlen(str_p)+1;
    r = p->pr_vasHead->vh_fwd;
    for (ii=0; ii<p->pr_vasHead->vh_entries; ii++) {
        if (r->rd_name) strcpy(str_p, r->rd_name);
        else strcpy(str_p, nullstr);
        str_p += strlen(str_p)+1;
        if (r->rd_path) strcpy(str_p, r->rd_path);
        else strcpy(str_p, nullstr);
        str_p += strlen(str_p)+1;
        r=r->rd_fwd;
    }

    /* wrap the buffer in the bfile wrapper and write to disk */
    bfile_write(BBUF_PROC_STRUCT, (char*) pbuf, pbuf->size);

} /* bfile_write_proc() */

/*
 * void bfile_process_proc(bfile_prochead_t *buf, ssize_t size)
 *
 * If while reading in from bfile a buffer is identified as a process
 * struct buffer, this function decodes it into our space and saves the
 * new process in the global process list.  Note that procs can show up
 * multiple times, in that case the last one read wins.
 */
void 
bfile_process_proc(bfile_prochead_t *buf, ssize_t size)
{
    process_t *newp, *oldp;
    region_t *oldr;
    char *strptr, *regptr;
    unsigned int ii;
    char regline[MAXPATHLEN];

    mINFORM("In bfile_process_proc(%p, %zu)", buf, size);
    mINFORM(" pid=%u numregions=%u numstrings=%u dcount=%u", 
            buf->pid, buf->numregions, buf->numstrings, buf->dcount);

    /* allocate space for new process and set utility ptrs */
    newp = getproc_by_pid(buf->pid);
    if (!newp) newp = alloc_proc(buf->pid);
    oldp = (process_t *)buf->data;
    regptr = (char *)oldp + sizeof(process_t);
    strptr = regptr + (buf->numregions*sizeof(region_t));
    
    /* assign the carry over items from saved proc */
    newp->pr_myPid = oldp->pr_myPid;
    newp->pr_myParentPid = oldp->pr_myParentPid;
    newp->pr_myGroupPid = oldp->pr_myGroupPid;
    newp->pr_mySid = oldp->pr_mySid;
    newp->pr_flags = oldp->pr_flags;
    newp->pr_isKthread = oldp->pr_isKthread;
    if (newp->pr_isKthread) gProclist.pl_numKthreads++;
    newp->pr_birthBy = oldp->pr_birthBy;
    newp->pr_endBy = oldp->pr_endBy;
    newp->pr_exec_times = oldp->pr_exec_times;
    newp->pr_is32bit = oldp->pr_is32bit;
    newp->pr_vasHead->vh_proc = newp;
    newp->pr_vasHead->vh_pid = newp->pr_myPid;

    /* Assign top level process strings, if the new string is null leave
     * the old one  alone, otherwise replace it with the new one.
     */ 
    mINFORM(" assigning saved process path: %s", strptr);
    if (strcmp(strptr, "null")) {
        if (newp->pr_path) FREE(newp->pr_path);
        newp->pr_path = strdup(strptr);
    }
    strptr += strlen(strptr)+1;
    mINFORM(" assigning saved kname: %s", strptr);
    if (strcmp(strptr, "null")) {
        if (newp->pr_myKname) FREE(newp->pr_myKname);
        newp->pr_myKname = strdup(strptr);
    }
    strptr += strlen(strptr)+1;

    /* If there are regions in the new proc struct, free any old regions
     * and allocate/link in the new ones.
     */
    if (buf->numregions) {
        if (newp->pr_vasHead->vh_entries) {
            region_t *r, *rtofree;
            r=newp->pr_vasHead->vh_fwd;
            ii=0;
            do {
                rtofree = r;
                r = r->rd_fwd;
                if (rtofree->rd_name) FREE(rtofree->rd_name);
                if (rtofree->rd_path) FREE(rtofree->rd_path);
                FREE(rtofree);
                ii++;
            } while (r != (void*) newp->pr_vasHead);
            mINFORM(" cleared %d maps from original proc struct", ii);
            newp->pr_vasHead->vh_fwd = NULL;
            newp->pr_vasHead->vh_bck = NULL;
            newp->pr_vasHead->vh_entries = 0;
        }

        for (ii=0; ii<buf->numregions; ii++) {
            oldr = (region_t *)regptr;
            /* skip over rd_name to rd_path in string table */
            strptr += strlen(strptr)+1;
            /* To link in the new region we use the code that reads /proc 
             * by first dummying up a /proc/pid/maps line and then calling
             * that function.
             */
            sprintf(regline,"%lx-%lx r-xp %lx  %s",
                    (unsigned long) oldr->rd_start, 
                    (unsigned long) oldr->rd_end, 
                    oldr->rd_offset, strptr);
            add_map_to_proc(newp, regline);

            /* advance to next region in file buffer */
            regptr += sizeof(region_t);
            /* and advance to next string set */
            strptr += strlen(strptr)+1;
        }
        mINFORM(" added %d new maps", ii);
    }

    /* stick new proc into the process list */
    putproc_by_pid(buf->pid, newp);

} /* bfile_process_proc() */

